"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_httpserver_1 = require("../../../../app.httpserver");
const app_logs_1 = require("../../../../app.logs");
const business_common_1 = require("../../../../business/business.common");
const business_hooks_1 = require("../../../../business/business.hooks");
const business_report_export_1 = require("../../../../business/report/business.report.export");
const report_util_1 = require("../../../../business/report/report.util");
const business_visitor_1 = require("../../../../business/visitor/business.visitor");
const dal_access_models_1 = require("../../../../dal/access/dal.access.models");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const restapi_1 = require("../../../../lib/es/models/restapi");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_visitor_v1_model_1 = require("../api.visitor.v1.model");
const dal_memcache_1 = require("../../../../dal/access/dal.memcache");
const messageBroker_notification_pub_1 = require("../../../../messageBroker/messageBroker.notification.pub");
module.exports.listUnterminatedVisits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.ListUnterminatedVisitorsRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listUnterminatedVisits(organizationId, {
        take: request.take,
        skip: request.skip,
        sortProfileFormField: request.sortProfileFormField,
        sortVisitFormField: request.sortVisitFormField,
        sortAscending: request.sortAscending,
        filter: !request.filter
            ? null
            : {
                profileFields: request.filter.profileFields,
                visitFields: request.filter.visitFields,
                state: request.filter.state,
            },
        visitorRegistrationPointId: request.visitorRegistrationPointId,
    }, organizationVisitorModuleSettings);
    dbResult = (0, business_visitor_1.hideSensitiveFieldsOfVisits)(organizationId, authResult, dbResult, organizationVisitorModuleSettings);
    res.json({
        total: dbResult.total,
        take: dbResult.options.take,
        skip: dbResult.options.skip,
        items: dbResult.items.map((t) => {
            return {
                visitorProfileId: t.visitorProfileId,
                visitId: t.visitId,
                visitStartUtc: t.visitStartUtc,
                visitorProfileFields: t.visitorProfileFields,
                visitFields: t.visitFields,
                state: t.state,
                thumbnail: t.thumbnail,
            };
        }),
    });
});
module.exports.listSelfUnterminatedVisits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.SelfListUnterminatedVisitorsRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listUnterminatedVisits(organizationId, {
        take: request.take,
        skip: request.skip,
        sortProfileFormField: request.sortProfileFormField,
        sortVisitFormField: request.sortVisitFormField,
        sortAscending: request.sortAscending,
        filter: !request.filter
            ? null
            : {
                profileFields: request.filter.profileFields,
                visitFields: request.filter.visitFields,
                state: request.filter.state,
            },
        visitorRegistrationPointId: null,
    }, organizationVisitorModuleSettings, authResult.jwt.userId);
    dbResult = (0, business_visitor_1.hideSensitiveFieldsOfVisits)(organizationId, authResult.jwt, dbResult, organizationVisitorModuleSettings);
    res.json({
        total: dbResult.total,
        take: dbResult.options.take,
        skip: dbResult.options.skip,
        items: dbResult.items.map((t) => {
            return {
                visitorProfileId: t.visitorProfileId,
                visitId: t.visitId,
                visitStartUtc: t.visitStartUtc,
                visitorProfileFields: t.visitorProfileFields,
                visitFields: t.visitFields,
                state: t.state,
                thumbnail: t.thumbnail,
            };
        }),
    });
});
module.exports.genericUnterminatedVisitSearch = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.GenericSearchRequest(req.body);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.genericUnterminatedVisitSearch(organizationId, organizationVisitorModuleSettings, request, hasOrganizationWide, authResult.userId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.searchVisitorProfileFormField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.FieldSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings.visitorProfileFormFields);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.searchVisitorProfileFormField(organizationId, organizationVisitorModuleSettings, {
        take: request.take,
        skip: request.skip,
        formField: request.formField,
        filter: request.filter,
    }, hasOrganizationWide, authResult.userId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        fieldName: request.fieldName,
        filter: request.filter,
        items: dbResult.items.map((t) => {
            return {
                captionLines: t.captionLines,
                value: t.value,
            };
        }),
    });
});
module.exports.searchVisitorProfileForm = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.FormSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), true, organizationVisitorModuleSettings);
    const dbResult = await dal_manager_1.dbManager.transaction(async (trx) => {
        let dbResult = await dal_manager_1.dbManager.accessVisitor.searchVisitorProfileForm(organizationId, organizationVisitorModuleSettings, {
            take: request.take,
            skip: request.skip,
            filterFields: request.fields.map((t) => {
                return {
                    name: t.fieldName,
                    filter: t.filter,
                    formField: t.formField,
                };
            }),
        }, hasOrganizationWide, authResult.userId, trx);
        return dbResult;
    });
    let result = dbResult.items.map((t) => {
        return {
            id: t.id,
            captionLines: t.captionLines,
            isUniqueMatch: t.isUniqueMatch,
            suggestionType: restapi_1.SuggestionType.Internal,
            states: t.states,
            searchFields: t.searchFields.map((ts) => {
                return {
                    name: ts.name,
                    value: ts.value,
                };
            }),
        };
    });
    (0, business_visitor_1.hideSensitiveFieldsOfVisitorFormSearch)(organizationId, req.auth_token, result, organizationVisitorModuleSettings);
    const afterHook = business_hooks_1.armonHookManager.getAfterSearchVisitorProfileForm(organizationId);
    if (afterHook) {
        const hookResult = await afterHook({
            organizationId: organizationId,
            request,
            dbResult,
        });
        if (hookResult) {
            dbResult.total = 1;
            result = hookResult;
        }
    }
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: result,
    });
});
module.exports.searchVisitHistoryFormField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.FieldSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings.visitFormFields);
    let dbResult = await dal_manager_1.dbManager.accessLog.searchVisitHistoryFormField(organizationId, organizationVisitorModuleSettings, {
        take: request.take,
        skip: request.skip,
        formField: request.formField,
        filter: request.filter,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        fieldName: request.fieldName,
        filter: request.filter,
        items: dbResult.items.map((t) => {
            return {
                captionLines: t.captionLines,
                value: t.value,
            };
        }),
    });
});
module.exports.searchVisitFormField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.FieldSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings.visitFormFields);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.searchVisitFormField(organizationId, organizationVisitorModuleSettings, {
        take: request.take,
        skip: request.skip,
        formField: request.formField,
        filter: request.filter,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        fieldName: request.fieldName,
        filter: request.filter,
        items: dbResult.items.map((t) => {
            return {
                captionLines: t.captionLines,
                value: t.value,
            };
        }),
    });
});
module.exports.customSearchVisitorField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.GenericSearchRequest(req.body);
    let result = {
        total: 0,
        items: [],
    };
    let hook = business_hooks_1.armonHookManager.getCustomVisitorGenericSearchHook(organizationId);
    if (hook) {
        result = await hook({
            organizationId: organizationId,
            organizationSettings: organizationVisitorModuleSettings,
            filter: request.filter,
            skip: request.skip,
            take: request.take,
        });
    }
    res.json({
        total: result.total,
        take: request.take,
        skip: request.skip,
        items: result.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.customSearchVisitField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitField = req.swagger.params.field.value;
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    if (!organizationVisitorModuleSettings.visitFormFields.some((f) => f.name != visitField)) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.INVALIDVISITFIELD");
    }
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.GenericSearchRequest(req.body);
    let result = {
        total: 0,
        items: [],
    };
    let hook = business_hooks_1.armonHookManager.getCustomVisitGenericSearchHook(organizationId);
    if (hook) {
        result = await hook({
            organizationId: organizationId,
            organizationSettings: organizationVisitorModuleSettings,
            filter: request.filter,
            skip: request.skip,
            take: request.take,
            field: visitField,
        });
    }
    res.json({
        total: result.total,
        take: request.take,
        skip: request.skip,
        items: result.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.getVisitorDetailed = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "visitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterOrVisitorReadRequest)(req, organizationId);
    let hasVisitorWritePermission = (0, api_securityhelper_1.hasVisitorWrite)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitorDetailedWithExpectedVisits(organizationId, visitorProfileId, hasVisitorWritePermission, authResult.jwt.userId, organizationVisitorModuleSettings);
    dbResult.profile = (0, business_visitor_1.hideSensitiveFieldsOfVisitors)(organizationId, authResult.jwt, [dbResult.profile], organizationVisitorModuleSettings)[0];
    res.json({
        profile: {
            id: dbResult.profile.id,
            fields: dbResult.profile.fields,
            states: dbResult.profile.states,
            thumbnail: dbResult.profile.thumbnail,
        },
        lastVisit: !dbResult.lastVisit
            ? null
            : {
                id: dbResult.lastVisit.id,
                startUtc: dbResult.lastVisit.startUtc,
                endUtc: dbResult.lastVisit.endUtc,
                fields: dbResult.lastVisit.fields,
            },
        expectedVisits: dbResult.expectedVisits,
    });
});
module.exports.getVisitorDetailedByVisitId = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "visitId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterOrVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitorDetailedByVisitId(organizationId, visitId, organizationVisitorModuleSettings);
    dbResult.profile = (0, business_visitor_1.hideSensitiveFieldsOfVisitors)(organizationId, authResult.jwt, [dbResult.profile], organizationVisitorModuleSettings)[0];
    res.json({
        profile: {
            id: dbResult.profile.id,
            fields: dbResult.profile.fields,
            states: dbResult.profile.states,
        },
        lastVisit: !dbResult.lastVisit
            ? null
            : {
                id: dbResult.lastVisit.id,
                startUtc: dbResult.lastVisit.startUtc,
                endUtc: dbResult.lastVisit.endUtc,
                fields: dbResult.lastVisit.fields,
            },
    });
});
module.exports.terminateVisitByQr = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let visitorRegistrationPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "visitorRegistrationPointId");
    let preregisteredVisitId = await business_visitor_1.BusinessVisitor.endVisitByQrCode(organizationId, visitorRegistrationPointId, organizationVisitorModuleSettings);
    let visitDetailed = await dal_manager_1.dbManager.accessVisitor.getActiveVisitDetailed(organizationId, preregisteredVisitId, organizationVisitorModuleSettings);
    if (visitDetailed.startUtc) {
        if (preregisteredVisitId) {
            await dal_manager_1.dbManager.accessVisitor.terminateVisit(organizationId, preregisteredVisitId, organizationVisitorModuleSettings, authResult.userId);
            let hook = business_hooks_1.armonHookManager.getAfterTerminateVisitHook(organizationId);
            if (hook) {
                await hook({
                    organizationId: organizationId,
                    visitId: preregisteredVisitId,
                    userId: authResult.userId,
                    visitorRegistrationPointId: visitDetailed.visitorRegistrationPointId,
                });
            }
            res.json({});
            app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
        }
    }
    else {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.NOT_FOUND, "ERRORS.VISITOR.VISITNOTFOUND");
    }
});
module.exports.terminateVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "visitId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let visitDetailed = await dal_manager_1.dbManager.accessVisitor.getActiveVisitDetailed(organizationId, visitId, organizationVisitorModuleSettings);
    await dal_manager_1.dbManager.accessVisitor.terminateVisit(organizationId, visitId, organizationVisitorModuleSettings, authResult.userId);
    let hook = business_hooks_1.armonHookManager.getAfterTerminateVisitHook(organizationId);
    if (hook) {
        await hook({
            organizationId: organizationId,
            visitId: visitId,
            userId: authResult.userId,
            visitorRegistrationPointId: visitDetailed.visitorRegistrationPointId,
        });
    }
    res.json({});
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
});
module.exports.genericSearchVisitorProfile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.GenericSearchRequest(req.body);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.genericSearchVisitorProfile(organizationId, organizationVisitorModuleSettings, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
    }, hasOrganizationWide, authResult.userId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.listVisitorProfileWithState = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitorProfileListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listVisitorProfilesWithState(organizationId, organizationVisitorModuleSettings, request.take, request.skip, request.visitorProfileIds, request.state);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
            };
        }),
    });
});
module.exports.genericSearchVisitorProfileWithState = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let hasOrganizationWide = authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.visitor.getSearchAll());
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.GenericSearchRequestWithVisitState(req.body);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.genericSearchVisitorProfile(organizationId, organizationVisitorModuleSettings, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
        state: request.state,
    }, hasOrganizationWide, authResult.userId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.upsertVisitVisitorProfile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterOrVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.UpsertVisitorProfileRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    const beforeHook = business_hooks_1.armonHookManager.getBeforeVisitorUpsert(organizationId);
    if (beforeHook) {
        request = beforeHook(request, organizationVisitorModuleSettings);
    }
    let dbResult = await dal_manager_1.dbManager.accessVisitor.upsertVisitorProfile(organizationId, request.fields, organizationVisitorModuleSettings, authResult.userId, request.id);
    res.json({
        visitorProfileId: dbResult.visitorProfileId,
    });
    if (dbResult) {
        let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, dbResult.visitorProfileId);
        await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
    }
});
module.exports.obtainVisitIdViaQrCode = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitorRegistrationPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "visitorRegistrationPointId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let preregisteredVisitId = await business_visitor_1.BusinessVisitor.obtainVisitIdViaQrCode(organizationId, visitorRegistrationPointId, organizationVisitorModuleSettings);
    res.json(preregisteredVisitId);
});
module.exports.newVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.NewVisitRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    const dbResult = await dal_manager_1.dbManager.transaction(async (trx) => {
        const dbResult = await business_visitor_1.BusinessVisitor.newVisit(organizationId, authResult.userId, request, organizationVisitorModuleSettings, trx, locale);
        res.json({
            visitorProfileId: dbResult.visitorProfileId,
            activeVisitId: dbResult.visitId,
        });
        return dbResult;
    });
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
    if (dbResult.visitedPersonUserId) {
        const { eventId, instanceData } = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(organizationId, app_enums_1.enums.NotificationType.VisitorCame, trx);
            const instanceData = {
                a: dbResult.visitId,
                receiverUserIds: [dbResult.visitedPersonUserId],
            };
            const eventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                createdT: new Date(),
                notificationId,
                organizationId,
                instanceData,
                trx,
            });
            return { eventId, instanceData };
        });
        (0, messageBroker_notification_pub_1.publishToNotificationService)({
            a: 0,
            i: eventId,
            n: app_enums_1.enums.AmqpMessageCode.Empty,
            o: organizationId,
            v: "1",
            d: instanceData,
        }, 10);
    }
});
module.exports.deletePreregisteredVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.pid.value, "visitId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterDelete)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.deleteExpectedVisit(organizationId, authResult.userId, visitId);
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
    res.json();
});
module.exports.upsertSelfPreregisteredVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.UpsertPreregisteredVisitRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    const dbResult = await dal_manager_1.dbManager.transaction(async (trx) => {
        if (request.visitId) {
            const preregisteredVisitState = await dal_manager_1.dbManager.accessVisitor.getActiveVisitState(organizationId, request.visitId, trx);
            if (preregisteredVisitState !== restapi_1.OrganizationVisitorStates.Expected) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.VISITOR.EXPECTEDVISITNOTFOUND", null, true, false);
            }
        }
        const beforePreregisterhook = business_hooks_1.armonHookManager.getBeforePreregisterVisitHook(organizationId);
        if (beforePreregisterhook) {
            request.visitorProfileFields = await beforePreregisterhook({
                organizationId,
                requesterUserId: authResult.jwt.userId,
                visitFields: request.visitFields,
                trx: trx,
                visitId: request.visitId,
                force: request.force,
                visitorProfileId: request.visitorProfileId,
                visitorProfileFields: request.visitorProfileFields,
            });
        }
        const dbResult = await dal_manager_1.dbManager.accessVisitor.upsertPreregisteredVisit(organizationId, authResult.jwt.userId, authResult.jwt.userId, organizationVisitorModuleSettings, request.visitFields, request.visitorProfileFields, request.visitorProfileId, false, trx, request.visitId, locale);
        const afterPreregisterHook = business_hooks_1.armonHookManager.getAfterPreregisterVisitHook(organizationId);
        if (afterPreregisterHook) {
            await afterPreregisterHook({
                organizationId,
                visitFields: request.visitFields,
                visitorProfileFields: request.visitorProfileFields,
                trx,
                visitorProfileId: dbResult.visitorProfileId,
                expectedVisitId: dbResult.visitId,
                requestUserId: authResult.jwt.userId,
            });
        }
        return dbResult;
    });
    if (dbResult?.invitationInfo?.qrCodeData) {
        dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            (0, business_visitor_1.sendQrCodeToVisitor)(organizationId, dbResult.invitationInfo, dbResult.visitorUserId, organizationVisitorModuleSettings, trx);
        }, authResult.jwt.userId, organizationId);
    }
    res.json({
        visitorProfileId: dbResult.visitorProfileId,
        expectedVisitId: dbResult.visitId,
    });
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
});
module.exports.upsertPreregisteredVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterForOthersRequest)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.UpsertPreregisteredVisitRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    const visitedPersonUser = request.visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson && t.value);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    const dbResult = await dal_manager_1.dbManager.transaction(async (trx) => {
        if (request.visitId) {
            const preregisteredVisitState = await dal_manager_1.dbManager.accessVisitor.getActiveVisitState(organizationId, request.visitId, trx);
            if (preregisteredVisitState !== restapi_1.OrganizationVisitorStates.Expected) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.VISITOR.EXPECTEDVISITNOTFOUND", null, true, false);
            }
        }
        const beforePreregisterHook = business_hooks_1.armonHookManager.getBeforePreregisterVisitHook(organizationId);
        if (beforePreregisterHook) {
            request.visitorProfileFields = await beforePreregisterHook({
                organizationId,
                requesterUserId: authResult.userId,
                visitFields: request.visitFields,
                trx: trx,
                visitId: request.visitId,
                force: request.force,
                visitorProfileId: request.visitorProfileId,
                visitorProfileFields: request.visitorProfileFields,
                visitedPersonUser: visitedPersonUser,
            });
        }
        const dbResult = await dal_manager_1.dbManager.accessVisitor.upsertPreregisteredVisit(organizationId, authResult.userId, visitedPersonUser ? visitedPersonUser.value : null, organizationVisitorModuleSettings, request.visitFields, request.visitorProfileFields, request.visitorProfileId, true, trx, request.visitId, locale);
        const afterPreregisterHook = business_hooks_1.armonHookManager.getAfterPreregisterVisitHook(organizationId);
        if (afterPreregisterHook) {
            await afterPreregisterHook({
                organizationId,
                visitFields: request.visitFields,
                visitorProfileFields: request.visitorProfileFields,
                trx,
                visitorProfileId: dbResult.visitorProfileId,
                expectedVisitId: dbResult.visitId,
                requestUserId: authResult.userId,
            });
        }
        return dbResult;
    });
    if (dbResult?.invitationInfo?.qrCodeData) {
        dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            (0, business_visitor_1.sendQrCodeToVisitor)(organizationId, dbResult.invitationInfo, dbResult.visitorUserId, organizationVisitorModuleSettings, trx);
        }, authResult.userId, organizationId);
    }
    res.json({
        visitorProfileId: dbResult.visitorProfileId,
        expectedVisitId: dbResult.visitId,
    });
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
});
module.exports.listTerminatedVisitsAsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.ListTerminatedVisitsAsExcelRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings, {
        visitFields: organizationVisitorModuleSettings.visitFormFields
            .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
            .map((v) => v.name),
        visitorFields: organizationVisitorModuleSettings.visitorProfileFormFields
            .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
            .map((v) => v.name),
        hideUniqueIdForVisitor: organizationVisitorModuleSettings.uniqueIdSettings?.hide ? true : false,
        showNumberOfInitial: organizationVisitorModuleSettings.uniqueIdSettings?.showNumberOfInitial ?? null,
        showNumberOfLast: organizationVisitorModuleSettings.uniqueIdSettings?.showNumberOfLast ?? null,
    });
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            options.organizationVisitorModuleSettings = organizationVisitorModuleSettings;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.TerminatedVisits,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Terminated Visits is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listTerminatedVisits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.ListTerminatedVisitsRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listAllVisitsForVisitReport(organizationId, {
        take: request.take,
        skip: request.skip,
        sortStartUtcAsc: request.sortStartUtcAsc,
        startUtc: request.startUtc,
        endUtc: request.endUtc,
        visitorProfileIds: request.visitorProfileIds,
        visitorProfileFilterFields: request.visitorProfileFilterFields,
        visitFilterFields: request.visitFilterFields,
        type: request.type,
    }, request.visitorRegistrationPointId, organizationVisitorModuleSettings, authResult.userId);
    dbResult = (0, business_visitor_1.hideSensitiveFieldsOfVisits)(organizationId, authResult, dbResult, organizationVisitorModuleSettings);
    res.json({
        total: dbResult.total,
        take: dbResult.options.take,
        skip: dbResult.options.skip,
        items: dbResult.items.map((t) => {
            return {
                visitorProfileId: t.visitorProfileId,
                visitorUserId: t.visitorUserId,
                visitId: t.visitId,
                visitStartUtc: t.visitStartUtc,
                visitEndUtc: t.visitEndUtc,
                visitorProfileFields: t.visitorProfileFields,
                visitFields: t.visitFields,
            };
        }),
    });
});
module.exports.listSelfTerminatedVisitsAsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.ListTerminatedVisitsAsExcelRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    if (!request.visitFilterFields)
        request.visitFilterFields = [
            {
                name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson,
                value: authResult.jwt.userId,
            },
        ];
    else {
        request.visitFilterFields.push({
            name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson,
            value: authResult.jwt.userId,
        });
    }
    let result = await (0, business_report_export_1.generateUnterminatedVisitReport)(organizationId, authResult.jwt.userId, request, organizationVisitorModuleSettings);
    return;
});
module.exports.listSelfTerminatedVisits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.ListTerminatedVisitsRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    if (!request.visitFilterFields)
        request.visitFilterFields = [
            {
                name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson,
                value: authResult.jwt.userId,
            },
        ];
    else {
        request.visitFilterFields.push({
            name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson,
            value: authResult.jwt.userId,
        });
    }
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listAllVisitsForVisitReport(organizationId, {
        take: request.take,
        skip: request.skip,
        sortStartUtcAsc: request.sortStartUtcAsc,
        startUtc: request.startUtc,
        endUtc: request.endUtc,
        visitorProfileIds: request.visitorProfileIds,
        visitorProfileFilterFields: request.visitorProfileFilterFields,
        visitFilterFields: request.visitFilterFields,
        type: request.type,
    }, request.visitorRegistrationPointId, organizationVisitorModuleSettings, authResult.jwt.userId);
    dbResult = (0, business_visitor_1.hideSensitiveFieldsOfVisits)(organizationId, authResult.jwt, dbResult, organizationVisitorModuleSettings);
    res.json({
        total: dbResult.total,
        take: dbResult.options.take,
        skip: dbResult.options.skip,
        items: dbResult.items.map((t) => {
            return {
                visitorProfileId: t.visitorProfileId,
                visitorUserId: t.visitorUserId,
                visitId: t.visitId,
                visitStartUtc: t.visitStartUtc,
                visitEndUtc: t.visitEndUtc,
                visitorProfileFields: t.visitorProfileFields,
                visitFields: t.visitFields,
            };
        }),
    });
});
module.exports.deleteSelfExpectedVisit = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "visitId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.deleteExpectedVisit(organizationId, authResult.jwt.userId, visitId);
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
    res.json();
});
module.exports.findVisitorByCredentialData = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let hasIdentityRead = (0, api_securityhelper_1.hasIdentityBasicRead)(req, organizationId);
    let request = {
        credentialData: req.body.credentialData,
        credentialType: req.body.credentialType,
    };
    let dbResult = await dal_manager_1.dbManager.accessVisitor.findVisitorByCredentialData(organizationId, hasIdentityRead, request.credentialData, request.credentialType);
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "There is no visitor with the specified credential" }));
        return;
    }
    res.json({
        credentialType: dbResult.credentialType,
        existance: dbResult.existance,
        memberInfo: dbResult.member,
        visitor: dbResult.visitor
            ? {
                profile: {
                    id: dbResult.visitor.profile.id,
                    fields: dbResult.visitor.profile.fields,
                    states: dbResult.visitor.profile.states,
                },
                lastVisit: !dbResult.visitor.lastVisit
                    ? null
                    : {
                        id: dbResult.visitor.lastVisit.id,
                        startUtc: dbResult.visitor.lastVisit.startUtc,
                        endUtc: dbResult.visitor.lastVisit.endUtc,
                        fields: dbResult.visitor.lastVisit.fields,
                    },
            }
            : {},
    });
});
module.exports.findActiveVisitorByUniqueField = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.FindActiveVisitByUniqueFieldRequest(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.findActiveVisitorByUniqueField(organizationId, organizationVisitorModuleSettings, request.name, request.value, request.visitorRegistrationPointId);
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "There is no visitor with the specified unique field value" }));
        return;
    }
    res.json({
        profile: dbResult.profile,
        lastVisit: dbResult.lastVisit,
    });
});
module.exports.listVisitors = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorViewAllVisitors)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.ListVisitorsRequestApi(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listVisitors(organizationId, {
        take: request.take,
        skip: request.skip,
        visitorProfileFilterFields: request.visitorProfileFilterFields,
        showOnlyBlacklist: request.showOnlyBlacklist,
        showOnlyTemporaryPermitted: request.showOnlyTemporaryPermitted,
        visitorProfileId: request.visitorProfileId,
    }, organizationVisitorModuleSettings);
    dbResult.items = (0, business_visitor_1.hideSensitiveFieldsOfVisitors)(organizationId, authResult, dbResult.items, organizationVisitorModuleSettings);
    res.json({
        take: request.take,
        skip: request.skip,
        total: dbResult.pagination.total,
        items: dbResult.items,
    });
});
module.exports.listVisitorsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForVisitorViewAllVisitors)(req, organizationId);
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const request = new api_visitor_v1_model_1.RestVisitorApiV1.ListVisitorsRequestBase(req.body, new api_validatorhelper_1.ValidatorHelper(), organizationVisitorModuleSettings, {
        visitFields: organizationVisitorModuleSettings.visitFormFields
            .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
            .map((v) => v.name),
        visitorFields: organizationVisitorModuleSettings.visitorProfileFormFields
            .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
            .map((v) => v.name),
        hideUniqueIdForVisitor: organizationVisitorModuleSettings.uniqueIdSettings?.hide ? true : false,
        showNumberOfInitial: organizationVisitorModuleSettings.uniqueIdSettings?.showNumberOfInitial ?? null,
        showNumberOfLast: organizationVisitorModuleSettings.uniqueIdSettings?.showNumberOfLast ?? null,
    });
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.VisitorReport,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Visitor is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getVisitorProfile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorViewAllVisitors)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitor(organizationId, organizationVisitorProfileId, organizationVisitorModuleSettings);
    res.json(dbResult);
});
module.exports.upsertBlacklistInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorBlackList)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.UpsertBlackListInfo(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.upsertVisitorState(organizationId, authResult.userId, organizationVisitorProfileId, {
        state: app_enums_1.enums.OrganizationVisitorStates.Forbidden,
        startUtc: request.startUtc,
        endUtc: request.endUtc,
        note: request.note,
    });
    if (dbResult) {
        let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, organizationVisitorProfileId);
        await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
    }
    res.json();
});
module.exports.removeBlacklistInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorBlackList)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.removeVisitorState(organizationId, authResult.userId, organizationVisitorProfileId, app_enums_1.enums.OrganizationVisitorStates.Forbidden);
    let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, organizationVisitorProfileId);
    await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
    res.json();
});
module.exports.upsertTemporalPermissionInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorTemporaryPermit)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.UpsertTemporalPermissionInfo(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.upsertVisitorState(organizationId, authResult.userId, organizationVisitorProfileId, {
        state: app_enums_1.enums.OrganizationVisitorStates.TemporaryPermit,
        startUtc: request.startUtc,
        endUtc: request.endUtc,
        note: request.note,
    });
    if (dbResult) {
        let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, organizationVisitorProfileId);
        await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
    }
    res.json();
});
module.exports.removeTemporalPermissionInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForVisitorTemporaryPermit)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.removeVisitorState(organizationId, authResult.userId, organizationVisitorProfileId, app_enums_1.enums.OrganizationVisitorStates.TemporaryPermit);
    let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, organizationVisitorProfileId);
    await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
    res.json();
});
module.exports.getVisitorDashboardInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitorDashboardInfoRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitorDashboardInfo(organizationId, authResult.userId, request.date, request.visitorRegistrationPointId);
    res.json(dbResult);
});
module.exports.getVisitorDetailedDashboardInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "visitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitorDetailedDashboardInfo(organizationId, authResult.userId, visitorProfileId);
    res.json(dbResult);
});
module.exports.getVisitorProfileUserActions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const organizationVisitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "organizationVisitorProfileId");
    const authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    const dbResult = await dal_manager_1.dbManager.accessLog.listUserActionHistory(organizationId, organizationVisitorProfileId, "visitorProfileId", app_enums_1.enums.UserActionCategory.Visitor, [
        app_enums_1.enums.UserVisitorActionType.InsertVisitorProfile,
        app_enums_1.enums.UserVisitorActionType.UpdateVisitorProfile,
        app_enums_1.enums.UserVisitorActionType.MergeVisitorProfile,
        app_enums_1.enums.UserVisitorActionType.SetVisitorAsBlacklisted,
        app_enums_1.enums.UserVisitorActionType.SetVisitorAsTemporaryPermitted,
        app_enums_1.enums.UserVisitorActionType.UnsetVisitorAsBlacklisted,
        app_enums_1.enums.UserVisitorActionType.UnsetVisitorAsTemporaryPermitted,
        app_enums_1.enums.UserVisitorActionType.NewExpectedVisit,
    ]);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    const result = (0, business_visitor_1.transformVisitorActionItemsLocalizedMessages)(dbResult, locale);
    res.json({
        items: result,
    });
});
module.exports.getVisitUserActions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "visitId");
    const authResult = (0, api_securityhelper_1.authorizeForListVisitorReadRequest)(req, organizationId);
    const dbResult = await dal_manager_1.dbManager.accessLog.listUserActionHistory(organizationId, visitId, "visitId", app_enums_1.enums.UserActionCategory.Visitor, [
        app_enums_1.enums.UserVisitorActionType.NewVisit,
        app_enums_1.enums.UserVisitorActionType.TerminateVisit,
        app_enums_1.enums.UserVisitorActionType.NewExpectedVisit,
        app_enums_1.enums.UserVisitorActionType.DeleteExpectedVisit,
    ]);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    const result = (0, business_visitor_1.transformVisitorActionItemsLocalizedMessages)(dbResult, locale);
    res.json({
        items: result,
    });
});
module.exports.getVisitCountByVisitorRegistrationPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitCountsByFilter(organizationId, request, true);
    res.json({ items: dbResult });
});
module.exports.getVisitCountByReason = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitCountsByFilter(organizationId, request, false);
    res.json({ items: dbResult });
});
module.exports.getVisitProcessTimeByVisitorRegistrationPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitProcessTimeByFilter(organizationId, request, true);
    res.json({ items: dbResult });
});
module.exports.getVisitProcessTimeByReason = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getVisitProcessTimeByFilter(organizationId, request, false);
    res.json({ items: dbResult });
});
module.exports.getPreregistrationAnalysis = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessVisitor.getPreregistrationAnalysis(organizationId, request);
    res.json({ items: dbResult });
});
module.exports.getVisitorModuleDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let visitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let registrationPointsDetailed = await dal_manager_1.dbManager.accessVisitor.listRegistrationPointsDetailed(organizationId, authResult.userId);
    res.json({
        visitFormFields: visitorModuleSettings.visitFormFields,
        visitorProfileFormFields: visitorModuleSettings.visitorProfileFormFields,
        gdprSettings: visitorModuleSettings.gdprSettings,
        registrationPoints: registrationPointsDetailed,
    });
});
module.exports.deleteVisitorProfileAndvisits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "visitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.deleteVisitorData(organizationId, visitorProfileId);
    res.json({});
});
module.exports.makeVisitorProfileAnonym = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let visitorProfileId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "visitorProfileId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    await dal_manager_1.dbManager.accessVisitor.makeVisitorProfileAnonym(organizationId, visitorProfileId);
    res.json({});
});
module.exports.getVisitCountByGenericFilter = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisGenericFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (request.filterType === app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByLicensePlate) {
        let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
        if (!organizationVisitorModuleSettings ||
            !organizationVisitorModuleSettings.availableCredentials ||
            !organizationVisitorModuleSettings.availableCredentials.availableCredentialTypes.includes(app_enums_1.enums.CredentialType.VehiclePlate)) {
            res.json({
                pagination: {
                    take: request.pagination.take,
                    skip: request.pagination.skip,
                    total: 0,
                },
                filterType: request.filterType,
                items: [],
            });
            return;
        }
    }
    let dbResult = await dal_manager_1.dbManager.accessLog.getVisitCountsByGenericFilter(organizationId, request);
    let captionLines = [];
    switch (request.filterType) {
        case app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
            let visitorUserIdProfileIdMapping = await dal_manager_1.dbManager.accessVisitor.getUserIdsByVisitorProfileIds(organizationId, dbResult.items.map((d) => d.id));
            let visitorCaptionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, visitorUserIdProfileIdMapping.map((v) => v.visitorUserId));
            for (let vcl of visitorCaptionLines) {
                captionLines.push({
                    id: visitorUserIdProfileIdMapping.find((vm) => vm.visitorUserId === vcl.id).visitorProfileId,
                    captionLines: vcl.captionLines,
                });
            }
            break;
        case app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
            captionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, dbResult.items.map((d) => d.id));
            break;
        default:
            break;
    }
    let result = {
        pagination: dbResult.pagination,
        filterType: request.filterType,
        items: [],
    };
    for (let dbr of dbResult.items) {
        let captions = [];
        if (captionLines && captionLines.length > 0) {
            let captionLinesOfSingleUser = captionLines.find((cl) => cl.id === dbr.id);
            if (captionLinesOfSingleUser && captionLinesOfSingleUser.captionLines) {
                captions = captionLinesOfSingleUser.captionLines;
            }
        }
        result.items.push({
            captionLines: captions,
            name: dbr.name,
            count: parseInt(dbr.count),
        });
    }
    res.json(result);
});
module.exports.visitCountByGenericFilterExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisGenericFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.VisitorCountReport,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.analysisVisitorsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    let request = new api_visitor_v1_model_1.RestVisitorApiV1.VisitAnalysisFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let jwt = (0, api_securityhelper_1.authorizeForSearchVisitorRequest)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AnalysisVisitorsReport,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.sendVisitorInvitationNotification = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const visitId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "visitId");
    const authResult = (0, api_securityhelper_1.authorizeForVisitorPreregisterSelfRequest)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await (0, business_visitor_1.resendVisitorInvitationNotificaton)(organizationId, visitId, trx);
    }, authResult.jwt.userId, organizationId);
    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
    res.json();
});
module.exports.generateExcelForVisitorRegistrations = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let visitorExcelResponse = await business_visitor_1.BusinessVisitor.downloadVisitorExcel(organizationId, authResult.userId, locale, organizationVisitorModuleSettings, req.swagger.params.autoRegistration.value ? false : req.swagger.params.preRegistration.value);
    if (!visitorExcelResponse.errorMessage) {
        if (fs_1.default.existsSync(visitorExcelResponse.filePath)) {
            res.download(visitorExcelResponse.filePath, visitorExcelResponse.filename, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(visitorExcelResponse.filePath);
            });
        }
        else {
            res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json();
        }
    }
    else {
        res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
            message: visitorExcelResponse.errorMessage,
            noReport: true,
            showAsModal: false,
        });
    }
});
module.exports.uploadExcelForVisitorRegistrations = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForListVisitorWriteRequest)(req, organizationId);
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    const validation = await business_visitor_1.BusinessVisitor.uploadVisitorExcel(res, organizationId, authResult.userId, locale, organizationVisitorModuleSettings, req.swagger.params.file.originalValue.buffer, req.swagger.params.preRegistration.value, req.swagger.params.autoRegistration.value, req.swagger.params.visitFields.value);
    if (validation.validationSuccess) {
        res.json({});
    }
    else if (validation.errorMessage) {
        res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).json({
            message: validation.errorMessage,
            showAsModal: true,
            noReport: true,
        });
    }
    else {
        if (fs_1.default.existsSync(validation.filePath)) {
            res.download(validation.filePath, validation.filename, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(validation.filePath);
            });
        }
        else {
            res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json();
        }
    }
});
