"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiVisitorTabletConrollerV1 = void 0;
const express_1 = __importDefault(require("express"));
const ws_1 = __importDefault(require("ws"));
const business_device_1 = require("../../../business/business.device");
const dal_manager_1 = require("../../../dal/dal.manager");
const messageBroker_server_to_tablet_sub_1 = require("../../../messageBroker/messageBroker.server-to-tablet.sub");
const api_util_1 = require("../../api.util");
const api_validatorhelper_1 = require("../../api.validatorhelper");
const app_logs_1 = require("../../../app.logs");
const messageBroker_server_to_app_pub_1 = require("../../../messageBroker/messageBroker.server-to-app.pub");
const ws_usernsp_1 = require("../../../ws/ws.usernsp");
const messageBroker_server_to_app_sub_1 = require("../../../messageBroker/messageBroker.server-to-app.sub");
class ApiVisitorTabletConrollerV1 {
    constructor(app, server) {
        this.generateChallengeToken = async (req, res, next) => {
            let deviceId = req.params.did;
            if (!(0, api_validatorhelper_1.validateUUID)(deviceId, "deviceId")) {
                return res.status(401).json({ message: "Invalid request", errors: "Invalid Device UUID" });
            }
            try {
                let result = await dal_manager_1.dbManager.accessDevice.generateVisitorTabletChallenge(deviceId);
                res.json({
                    organizationPublicKey: result.orgPublicKey,
                    challenge: result.challenge,
                    iv: result.iv,
                });
            }
            catch (error) {
                return res.status(404).json({ message: "Device not found!" });
            }
        };
        this.auth = async (req, res, next) => {
            let authHeader = req.headers.authorization;
            if (!authHeader) {
                return res.status(401).json({ message: "Invalid authorization" });
            }
            let token = authHeader.replace("token,", "");
            let device = await dal_manager_1.dbManager.accessDevice.getTerminalByToken(token);
            if (!device) {
                return res.status(401).json({ message: "Unable to find device" });
            }
            let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(device.organizationId);
            if (!organizationVisitorModuleSettings) {
                return res.status(404).json({ message: "Organization Visitor Module Setting Not Found" });
            }
            if (!organizationVisitorModuleSettings.gdprSettings) {
                return res.status(404).json({ message: "Gdpr settings not found" });
            }
            let organizationLogo = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(device.organizationId);
            let organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(device.organizationId);
            let result = {
                anonymizationDuration: organizationVisitorModuleSettings.gdprSettings.anonymizationDuration,
                approvementTimeout: organizationVisitorModuleSettings.gdprSettings.approvementTimeout,
                canBeDeleted: organizationVisitorModuleSettings.gdprSettings.canBeDeleted,
                legalText: organizationVisitorModuleSettings.gdprSettings.legalText,
                logo: organizationLogo ? Buffer.from(organizationLogo).toString() : null,
                organizationName: organization.name,
                mainColor: organizationVisitorModuleSettings.gdprSettings.mainColor,
                showRejectButton: organizationVisitorModuleSettings.gdprSettings.showRejectButton,
                visitorPhotoState: organizationVisitorModuleSettings.gdprSettings.visitorPhotoState,
                visitorProfilesFormFields: organizationVisitorModuleSettings.visitorProfileFormFields,
                policies: organizationVisitorModuleSettings.policies ?? [],
            };
            res.json(result);
        };
        this._router = express_1.default.Router();
        this._clients = {};
        this._router.get("/challenge/:did", (0, api_util_1.globalRouteHandler)(this.generateChallengeToken));
        this._router.get("/auth", (0, api_util_1.globalRouteHandler)(this.auth));
        app.use("/visitor/tablet", express_1.default.json({ limit: "100mb" }), this._router);
        this._wss = new ws_1.default.Server({ clientTracking: true, noServer: true });
        server.on("upgrade", async (request, socket, head) => {
            if (request.url == "/visitor/tablet/connect") {
                let authHeader = request.headers["sec-websocket-protocol"];
                if (!authHeader || authHeader.split(",").length < 2) {
                    return;
                }
                let token = authHeader.replace("token,", "");
                let device = await dal_manager_1.dbManager.accessDevice.getTerminalByToken(token);
                if (!device) {
                    return;
                }
                request.deviceId = device.id;
                request.organizationId = device.organizationId;
                this._wss.handleUpgrade(request, socket, head, (ws) => {
                    this._wss.emit("connection", ws, request);
                });
            }
        });
        this._wss.on("connection", async (ws, request) => {
            ws.isAlive = true;
            ws.on("pong", () => {
                ws.isAlive = true;
            });
            const deviceId = request.deviceId;
            const organizationId = request.organizationId;
            app_logs_1.logger.info("New tablet device connected : " + deviceId);
            await this.openDeviceConnection(organizationId, deviceId, ws);
            ws.on("message", async (message) => {
                const clientAmqpSubscriber = this._clients[deviceId];
                if (clientAmqpSubscriber) {
                    await clientAmqpSubscriber.onSocketMessage(message);
                }
                else {
                    app_logs_1.logger.error(`Incoming web socket message from tablet ${deviceId} but no amqp subscribers found!!!!`);
                }
            });
            ws.on("open", async () => {
                app_logs_1.logger.info("New tablet device open : " + deviceId);
                await this.openDeviceConnection(organizationId, deviceId, ws);
            });
            ws.on("error", async (err) => {
                app_logs_1.logger.info(" tablet device : " + deviceId + " error : " + err);
                await this.closeDeviceConnection(organizationId, deviceId, ws);
            });
            ws.on("close", async (code, reason) => {
                app_logs_1.logger.info("tablet device closed : " + deviceId + " reason: " + reason + " code: " + code);
                await this.closeDeviceConnection(organizationId, deviceId, ws);
            });
        });
        this._pingTimer = setInterval(() => {
            this._wss.clients.forEach((ws) => {
                if (ws.isAlive === false) {
                    return ws.terminate();
                }
                ws.isAlive = false;
                ws.ping();
            });
        }, 5000);
    }
    async openDeviceConnection(organizationId, deviceId, ws) {
        let clientAmqpSubscriber = this._clients[deviceId];
        if (clientAmqpSubscriber) {
            app_logs_1.logger.error(`openDeviceConnection for tablet with id ${deviceId}, but there is already an amqp subscriber. Something is wrong! Closing deprecated amqp connection`);
            await clientAmqpSubscriber.closeChannel();
            delete this._clients[deviceId];
        }
        await (0, business_device_1.updateDeviceConnectionStatus)({
            organizationId,
            deviceId,
            isConnected: true,
            isRestart: false,
        });
        const amqpSub = new messageBroker_server_to_tablet_sub_1.AmqpServerToTabletSub(ws, deviceId);
        await amqpSub.init();
        this._clients[deviceId] = amqpSub;
        app_logs_1.logger.warn(`openDeviceConnection new amqp subscriber created successfully for tablet with id ${deviceId}!`);
        let registrationPointInfoByDeviceId = await dal_manager_1.dbManager.accessVisitor.getRegistrationPointByDeviceId(organizationId, deviceId);
        if (registrationPointInfoByDeviceId) {
            await messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.visitor + ".*", {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.visitorRegistrationDeviceStatusChanged,
                p: {
                    deviceId: deviceId,
                    deviceName: registrationPointInfoByDeviceId.deviceName,
                    registrationPointId: registrationPointInfoByDeviceId.id,
                    isConnected: true,
                },
            });
        }
    }
    async closeDeviceConnection(organizationId, deviceId, ws) {
        let clientAmqpSubscriber = this._clients[deviceId];
        if (clientAmqpSubscriber) {
            await clientAmqpSubscriber.closeChannel();
            delete this._clients[deviceId];
            app_logs_1.logger.warn(`closeDeviceConnection amqp subscriber connection closed successfully for tablet with id ${deviceId}!`);
        }
        else {
            app_logs_1.logger.error(`closeDeviceConnection web socket closing for tablet with id ${deviceId} but no amqp subscribers! Something is wrong!`);
        }
        await (0, business_device_1.updateDeviceConnectionStatus)({ organizationId, deviceId, isConnected: false, isRestart: false });
        let registrationPointInfoByDeviceId = await dal_manager_1.dbManager.accessVisitor.getRegistrationPointByDeviceId(organizationId, deviceId);
        if (registrationPointInfoByDeviceId) {
            await messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.visitor + ".*", {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.visitorRegistrationDeviceStatusChanged,
                p: {
                    deviceId: deviceId,
                    deviceName: registrationPointInfoByDeviceId.deviceName,
                    registrationPointId: registrationPointInfoByDeviceId.id,
                    isConnected: false,
                },
            });
        }
    }
}
exports.ApiVisitorTabletConrollerV1 = ApiVisitorTabletConrollerV1;
