"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const app_enums_1 = require("../../../../app.enums");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const business_summary_1 = require("../../../../business/business.summary");
const dal_access_error_1 = require("../../../../dal/access/dal.access.error");
module.exports.genericSearchUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dalResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityBasic(organizationId, authResult.hasOrganizationWide, {
        returnTotalCount: true,
        take: request.take,
        skip: request.skip,
        genericFilter: request.filter,
        identityStatus: app_enums_1.enums.IdentityStatusType.Enabled,
    }, authResult.jwt);
    res.json({
        total: dalResult.total,
        take: request.take,
        skip: request.skip,
        items: dalResult.items.map((t) => {
            let captionLines = [t.fullName];
            if (t.uniqueId) {
                captionLines.push(t.uniqueId);
            }
            if (t.email) {
                captionLines.push(t.email);
            }
            let matchItem = "";
            let filterItems = request.filter ? request.filter.split(" ") : [];
            for (let filter of filterItems) {
                filter = filter.trim();
                if (filter.length > 0) {
                    let regex = new RegExp(filter, "ig");
                    if (t.fullName.search(regex) >= 0) {
                        matchItem = t.fullName;
                    }
                    else if (t.uniqueId && t.uniqueId.search(regex) >= 0) {
                        matchItem = t.uniqueId;
                    }
                    else if (t.email && t.email.search(regex) >= 0) {
                        matchItem = t.email;
                    }
                    else {
                        if (t.organizationUnits && t.organizationUnits.length > 0) {
                            for (const unit of t.organizationUnits) {
                                if (unit.name.search(regex) >= 0) {
                                    matchItem = unit.name;
                                    break;
                                }
                            }
                        }
                        if (!matchItem) {
                            if (t.userGroups && t.userGroups.length > 0) {
                                for (const userGroup of t.userGroups) {
                                    if (userGroup.name.search(regex) >= 0) {
                                        matchItem = userGroup.name;
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    if (matchItem) {
                        break;
                    }
                }
            }
            return {
                id: t.id,
                captionLines: captionLines,
                matchItem: matchItem,
            };
        }),
    });
});
module.exports.genericSearchUserFromFullName = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (!authResult.hasOrganizationWide) {
        res.json({
            total: 0,
            take: request.take,
            skip: request.skip,
            items: [],
        });
    }
    let result = await dal_manager_1.dbManager.accessUser.genericSearchUserFromFullName(organizationId, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
    });
    res.json({
        total: result.total,
        take: request.take,
        skip: request.skip,
        items: result.items.map((t) => {
            return {
                id: t.id,
                userCaptions: t.userCaptions,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.genericSearchUserFromCaptionLines = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (!authResult.hasOrganizationWide) {
        res.json({
            total: 0,
            take: request.take,
            skip: request.skip,
            items: [],
        });
    }
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.genericSearchUserFromCaptionLines(organizationId, {
            take: request.take,
            skip: request.skip,
            filter: request.filter,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    res.json({
        total: result.total,
        take: request.take,
        skip: request.skip,
        items: result.items.map((t) => {
            return {
                id: t.id,
                userCaptions: t.userCaptions,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.listUsersOfSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    const request = req.body;
    const jwt = req.auth_token;
    const dbResult = await dal_manager_1.dbManager.accessUser.listUsersOfUserSelectionSession(organizationId, jwt.userId, sessionId, request.take, request.skip, request.filter);
    res.json(dbResult);
});
module.exports.askUsersForSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let jwt = req.auth_token;
    let request = req.body;
    let dbResult = await dal_manager_1.dbManager.accessUser.askUsersForSelectionSession(organizationId, jwt.userId, sessionId, request.userIds);
    res.json({
        expirationUtc: dbResult.expirationUtc,
        items: dbResult.items.map((u) => {
            return {
                userId: u.userId,
                status: u.status,
            };
        }),
        added: dbResult.added,
        removed: dbResult.removed,
    });
});
module.exports.cancelUserSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let jwt = req.auth_token;
    await dal_manager_1.dbManager.accessUser.cancelUserSelectionSession(organizationId, jwt.userId, sessionId);
    res.json({});
});
module.exports.applyUserSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let jwt = req.auth_token;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessUser.applyUserSelectionSession({ organizationId, requestUserId: jwt.userId, sessionId, trx });
    }, jwt.userId, organizationId);
    res.json({});
});
module.exports.applyUserSelectionSessionWithPrivileges = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let request = new api_user_v1_models_1.RestUserApiV1.ApplyUserSelectionSessionWithPrivilegesRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let jwt = req.auth_token;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let session = await dal_manager_1.dbManager.accessUser.validateUserSelectionSession(organizationId, jwt.userId, sessionId, trx);
        await dal_manager_1.dbManager.accessAccessControlPoint.upsertUserAccessControlPointGrantsBySelectionSession(organizationId, {
            requestUserId: jwt.userId,
            userSelectionSessionId: session.id,
            accessControlPointId: session.relatedItemId,
            privileges: request,
        }, trx);
    }, jwt.userId, organizationId);
    res.json({});
});
module.exports.addUserToSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessUser.addUserToUserSelectionSession(organizationId, jwt.userId, sessionId, userId);
    res.json({
        total: dbResult.total,
        added: dbResult.added,
        removed: dbResult.removed,
        expirationUtc: dbResult.expirationUtc,
    });
});
module.exports.removeUserFromSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessUser.removeUserFromUserSelectionSession(organizationId, jwt.userId, sessionId, userId);
    res.json({
        total: dbResult.total,
        added: dbResult.added,
        removed: dbResult.removed,
        expirationUtc: dbResult.expirationUtc,
    });
});
module.exports.removeAllUsersFromSelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessUser.removeAllUsersFromUserSelectionSession(organizationId, jwt.userId, sessionId);
    res.json({
        total: dbResult.total,
        added: dbResult.added,
        removed: dbResult.removed,
        expirationUtc: dbResult.expirationUtc,
    });
});
module.exports.addRemoveUsersToSelectionSessionAccordingToSearchResult = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    let searchRequest = {
        organizationId: organizationId,
        status: request.status,
        profileFilter: request.profileFilter,
        organizationUnits: request.organizationUnits || [],
        userGroupIds: request.userGroupIds,
        credentialExtensionFilters: [],
        workplans: request.workplans,
    };
    if (!authResult.hasOrganizationWide) {
        searchRequest.organizationUnits.push(...authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic()).map((unitId) => {
            return {
                id: unitId,
                includeDescendants: true,
            };
        }));
    }
    let dbResult = await dal_manager_1.dbManager.accessUser.addRemoveUsersToSelectionSessionBySearchResult(authResult.jwt.userId, organizationId, sessionId, request.action, searchRequest);
    res.json({
        total: dbResult.total,
        added: dbResult.added,
        removed: dbResult.removed,
        expirationUtc: dbResult.expirationUtc,
    });
});
module.exports.listUserUnitsHierarchically = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "id");
    let dbResult = await dal_manager_1.dbManager.accessUser.listUserUnitsHierarchically(jwt.getOrganizationId(), userId);
    res.json({ items: dbResult });
});
module.exports.listSelfUserUnitsHierarchically = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessUser.listUserUnitsHierarchically(jwt.getOrganizationId(), jwt.userId);
    res.json({ items: dbResult });
});
module.exports.upsertUserApplicationProfile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let profile = req.body.profile;
    await dal_manager_1.dbManager.accessUser.upsertUserApplicationProfile(organizationId, jwt.userId, profile);
    res.json();
});
module.exports.getUserApplicationProfile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessUser.getUserApplicationProfile(organizationId, jwt.userId);
    res.json({ profile: dbResult });
});
module.exports.getSelfAccessAndLoginSummary = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let userSummary = new business_summary_1.UserInfoSummary(organizationId, jwt.userId);
    let accessInfo = await userSummary.getAccessAndLoginSummary(organizationId);
    res.json(accessInfo);
});
module.exports.getUserSettings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const userSetting = await dal_manager_1.dbManager.accessUser.getUserSettings({ organizationId, userId: jwt.userId, trx });
        const customNotificationSettings = [];
        if (userSetting.notification.mediumSettings.custom) {
            for (const key of Object.keys(userSetting.notification.mediumSettings.custom)) {
                customNotificationSettings.push({
                    type: parseInt(key),
                    mediums: userSetting.notification.mediumSettings.custom[key],
                });
            }
        }
        res.json({
            locale: userSetting.locale,
            notification: {
                mediumSettings: {
                    general: userSetting.notification.mediumSettings.general,
                    custom: customNotificationSettings,
                },
            },
        });
    }, jwt.userId, organizationId);
});
module.exports.setUserSettings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const restApiSettings = new api_user_v1_models_1.RestUserApiV1.UpdateUserSettingsRequestVal(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let customNotificationSettings = {};
        if (restApiSettings.notification) {
            for (const customNotificationSetting of restApiSettings.notification.mediumSettings.custom) {
                customNotificationSettings[customNotificationSetting.type] = customNotificationSetting.mediums;
            }
        }
        const updatedUserNotificationSettings = {
            mediumSettings: {
                general: restApiSettings?.notification?.mediumSettings?.general,
                custom: customNotificationSettings,
            },
        };
        await dal_manager_1.dbManager.accessUser.setUserSettings({
            organizationId,
            userId: jwt.userId,
            trx,
            settings: {
                locale: restApiSettings.locale,
                notification: updatedUserNotificationSettings,
            },
        });
    }, jwt.userId, organizationId);
    res.json({});
});
module.exports.addAccessRightsForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.AddAccessRightsToUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const requesterUserAccessRights = await dal_manager_1.dbManager.accessIdentity.getUserAccessRights({
        organizationId,
        userId: auth.jwt.userId,
    });
    const apsThatRequesterUserDoesNotHaveGrantAuth = request.accessPointIds.filter((accessPointId) => !requesterUserAccessRights
        .filter((ap) => ap.grant)
        .map((ap) => ap.accessControlPointId)
        .includes(accessPointId));
    if (apsThatRequesterUserDoesNotHaveGrantAuth.length > 0) {
        (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.USER_HAS_NO_GRANT_FOR_AP", { ap: JSON.stringify(apsThatRequesterUserDoesNotHaveGrantAuth) });
    }
    await dal_manager_1.dbManager.accessIdentity.addAccessRightsToUser({
        organizationId,
        userId,
        accessPointIds: request.accessPointIds,
        accessRight: request.accessRight,
        requesterUserId: auth.jwt.userId,
    });
    res.json({});
});
module.exports.updateAccessRightsForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.UpdateAccessRightsToUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const requesterUserAccessRights = await dal_manager_1.dbManager.accessIdentity.getUserAccessRights({
        organizationId,
        userId: auth.jwt.userId,
    });
    const apsThatRequesterUserDoesNotHaveGrantAuth = request.accessPointIds.filter((accessPointId) => !requesterUserAccessRights
        .filter((ap) => ap.grant)
        .map((ap) => ap.accessControlPointId)
        .includes(accessPointId));
    if (apsThatRequesterUserDoesNotHaveGrantAuth.length > 0) {
        (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.USER_HAS_NO_GRANT_FOR_AP", { ap: JSON.stringify(apsThatRequesterUserDoesNotHaveGrantAuth) });
    }
    await dal_manager_1.dbManager.accessIdentity.updateAcessRightsToUser({
        organizationId,
        userId,
        accessPointIds: request.accessPointIds,
        accessRight: request.accessRight,
        requesterUserId: auth.jwt.userId,
    });
    res.json({});
});
module.exports.deleteAccessRightsOfUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.DeleteAccessRightsToUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const requesterUserAccessRights = await dal_manager_1.dbManager.accessIdentity.getUserAccessRights({
        organizationId,
        userId: auth.jwt.userId,
    });
    const apsThatRequesterUserDoesNotHaveGrantAuth = request.accessPointIds.filter((accessPointId) => !requesterUserAccessRights
        .filter((ap) => ap.grant)
        .map((ap) => ap.accessControlPointId)
        .includes(accessPointId));
    if (apsThatRequesterUserDoesNotHaveGrantAuth.length > 0) {
        (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.USER_HAS_NO_GRANT_FOR_AP", { ap: JSON.stringify(apsThatRequesterUserDoesNotHaveGrantAuth) });
    }
    await dal_manager_1.dbManager.accessIdentity.deleteAccessRightsFromUser({
        organizationId,
        userId,
        accessPointIds: request.accessPointIds,
        requesterUserId: auth.jwt.userId,
    });
    res.json();
});
module.exports.checkUserRights = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForGetIdentityDetailed)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.CheckUserRightsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const userRights = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.getCheckUserRights({
            organizationId,
            userId,
            filter: request,
            trx,
        });
    }, auth.jwt.userId, organizationId);
    res.json(userRights);
});
module.exports.activateUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessUser.activateUser({ organizationId, userId, trx });
    }, auth.jwt.userId, organizationId);
    res.json({});
});
