"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const business_main_1 = require("../../../../business/business.main");
const report_util_1 = require("../../../../business/report/report.util");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const controller_accesscontrolpoint_1 = require("./controller.accesscontrolpoint");
const path_1 = __importDefault(require("path"));
module.exports.listReportTemplates = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let request = req.body;
    if (!request?.sorting?.field) {
        const sorting = {
            sorting: {
                field: "name",
            },
        };
        request = { ...request, ...sorting };
    }
    const reportTempaltes = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        request.filter.createdByUserId = auth.jwt.userId;
        return dal_manager_1.dbManager.accessReport.listReportTemplates({
            ...request,
            organizationId,
            trx,
        });
    }, auth.jwt.userId, organizationId);
    res.json(reportTempaltes);
});
module.exports.updateReportTemplate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const reportTemplateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "reportTemplateId");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    delete req.body.date;
    delete req.body.dateRange;
    delete req.body.pagination;
    delete req.body.paginationRequest;
    const request = req.body;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessReport.updateReportTemplate({
            organizationId,
            reportTemplateId,
            ...request,
            trx,
        });
    }, auth.jwt.userId, organizationId);
    res.json();
});
module.exports.getReportTemplate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const templateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "templateId");
    (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const reportTemplate = await dal_manager_1.dbManager.accessReport.getReportTemplate(organizationId, templateId);
    reportTemplate ? res.json(reportTemplate) : res.status(404).json();
});
module.exports.addReportTemplate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    delete req.body.date;
    delete req.body.dateRange;
    delete req.body.pagination;
    delete req.body.paginationRequest;
    delete req.body.returnTotalCount;
    delete req.body.take;
    delete req.body.skip;
    const request = req.body;
    const reportTemplates = await dal_manager_1.dbManager.accessReport.addReportTemplate({
        ...request,
        organizationId,
        userId: authResult.jwt.userId,
        customReportId: request.customReportId,
    });
    res.json(reportTemplates);
});
module.exports.deleteReportTemplate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const reportTemplateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "templateId");
    const force = (0, api_validatorhelper_1.validateIsBoolean)(req.swagger.params.force.value, "force");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const reportTemplateCount = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessReport.deleteReportTemplate({ organizationId, userId: auth.jwt.userId, reportTemplateId, force, trx });
    }, auth.jwt.userId, organizationId);
    if (force) {
        res.json({});
    }
    else {
        res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json(reportTemplateCount);
    }
});
module.exports.listScheduledJobs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    const userRightsForScheduledJobs = {
        read: {
            hasOrganizationWide: false,
            permittedUnitIds: [],
        },
        write: {
            hasOrganizationWide: false,
            permittedUnitIds: [],
        },
    };
    try {
        const authResultForRead = (0, api_securityhelper_1.authorizeForScheduledJobsRead)(req, organizationId);
        userRightsForScheduledJobs.read.hasOrganizationWide = authResultForRead.hasOrganizationWide;
        userRightsForScheduledJobs.read.permittedUnitIds = authResultForRead.permittedUnitIds;
    }
    catch (error) {
        userRightsForScheduledJobs.read.hasOrganizationWide = false;
        userRightsForScheduledJobs.read.permittedUnitIds = [];
    }
    try {
        const authResultForWrite = (0, api_securityhelper_1.authorizeForScheduledJobsWrite)(req, organizationId);
        userRightsForScheduledJobs.write.hasOrganizationWide = authResultForWrite.hasOrganizationWide;
        userRightsForScheduledJobs.write.permittedUnitIds = authResultForWrite.permittedUnitIds;
    }
    catch (error) {
        userRightsForScheduledJobs.write.hasOrganizationWide = false;
        userRightsForScheduledJobs.write.permittedUnitIds = [];
    }
    let request = req.body;
    if (!request.pagination) {
        request.pagination = { take: 10 };
    }
    else if (!request.pagination.take) {
        request.pagination.take = 10;
    }
    if (!request?.sorting?.field) {
        const sorting = {
            sorting: {
                field: "nextExecutionDate",
            },
        };
        request = { ...request, ...sorting };
    }
    let userIdsUnderHierarchy = [];
    let unitsAndUsersUnderHirerarchy = [];
    if (!userRightsForScheduledJobs.read.hasOrganizationWide && !request.filter.createdByUserId) {
        unitsAndUsersUnderHirerarchy = await dal_manager_1.dbManager.accessOrganizationUnit.listUsersWithOrganizationUnits(organizationId, userRightsForScheduledJobs.read.permittedUnitIds);
        for (const unitAndUsers of unitsAndUsersUnderHirerarchy) {
            userIdsUnderHierarchy = userIdsUnderHierarchy.concat(unitAndUsers.userIds);
        }
        if (!userIdsUnderHierarchy?.length) {
            userIdsUnderHierarchy = [jwt.userId];
        }
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessSystem.listScheduledJobs({
            ...request,
            organizationId,
            trx,
            locale: (await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId })).l,
            userIdsUnderHierarchy: userIdsUnderHierarchy,
        });
    }, jwt.userId, organizationId);
    const unitsAndUsersUnderHierarchyForEdit = unitsAndUsersUnderHirerarchy.filter((u) => userRightsForScheduledJobs.write.permittedUnitIds.includes(u.organizationUnitId));
    for (const item of result.items) {
        item.isEditable = false;
        if (item.createdBy.id === jwt.userId || userRightsForScheduledJobs.write.hasOrganizationWide) {
            item.isEditable = true;
        }
        else {
            for (const unitAndUsers of unitsAndUsersUnderHierarchyForEdit) {
                if (unitAndUsers.userIds.includes(item.createdBy.id)) {
                    item.isEditable = true;
                    break;
                }
            }
        }
    }
    res.json(result);
});
module.exports.listNotifications = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const request = req.body;
    const notifications = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return request.filter.archived
            ? dal_manager_1.dbManager.accessNotifications.listDeletedNotifications(organizationId, request, trx)
            : dal_manager_1.dbManager.accessNotifications.listNotifications(organizationId, authResult.jwt.userId, request, trx);
    }, authResult.jwt.userId, organizationId);
    res.json(notifications);
});
module.exports.getAccessSnapshotWithLogId = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "snapshotId");
    let logId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "logId");
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessLog.getAccessSnapshotWithLogId(organizationId, authResult.jwt.userId, logId, snapshotId, app_config_1.appConfig.db.log.backup);
    if (dbResult.isArchived)
        throw (0, api_error_1.generateResourceRemovedError)({ message: "snapshot data archived" });
    if (dbResult.isUnauthorized)
        throw (0, api_error_1.generateUnauthroizedSnapshotAccessError)({ message: "You are not permitted to see snapshots of this access control point" });
    if (!dbResult.isSuccess)
        throw (0, api_error_1.generateFailedDependencyError)({ message: "Camera was unable to take snapshot" });
    res.json({
        snapshot: dbResult.snapshot,
    });
});
module.exports.getAccessSnapshot = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "snapshotId");
    let dbResult = await dal_manager_1.dbManager.accessLog.getAccessSnapshot(organizationId, snapshotId);
    if (!dbResult.success)
        throw (0, api_error_1.generateFailedDependencyError)({ message: "Camera was unable to take snapshot" });
    res.json({
        snapshot: dbResult.snapshot,
    });
});
module.exports.getCardActivityDetailByLogId = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let logId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.lid.value, "logId");
    let dbResult = await dal_manager_1.dbManager.accessReport.getCardActivityDetailByLogId(organizationId, logId);
    res.json(dbResult);
});
module.exports.getFirstAndLastAccessLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.FirstAndLastAccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessReport.filterFirstAndtLastAccessLogs(organizationId, filter, authResult.jwt.userId, !authResult.hasOrganizationWide);
    res.json(dbResult);
});
module.exports.getFirstAndLastAccessLogsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.FirstAndLastAccessLogsFilterExcel(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    if ((!filter.organizationUnitIds || filter.organizationUnitIds.length < 1) && !authResult.hasOrganizationWide) {
        filter.organizationUnitIds = authResult.permittedUnitIds;
        filter.filterOrganizationUnitMembersHierarchically = true;
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = filter;
            options.hasOrganizationWide = authResult.hasOrganizationWide;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.FirstAndLastAccessLogs,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.entryByUnitReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let filter = new api_user_v1_models_1.RestUserApiV1.EntryByUnitFilter({ ...req.body, organizationId, requesterUserId: authResult.jwt.userId }, new api_validatorhelper_1.ValidatorHelper());
    if ((!filter.organizationUnitIds || filter.organizationUnitIds.length < 1) && !authResult.hasOrganizationWide) {
        filter.organizationUnitIds = authResult.permittedUnitIds;
    }
    let dbResult = await dal_manager_1.dbManager.accessReport.getEntryByUnitData(filter);
    res.json(dbResult);
});
module.exports.entryByUnitReportExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let filter = new api_user_v1_models_1.RestUserApiV1.EntryByUnitFilter({ ...req.body, organizationId, requesterUserId: authResult.jwt.userId }, new api_validatorhelper_1.ValidatorHelper());
    if ((!filter.organizationUnitIds || filter.organizationUnitIds.length < 1) && !authResult.hasOrganizationWide) {
        filter.organizationUnitIds = authResult.permittedUnitIds;
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.EntryByUnit,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: filter,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getFirstAndLastAccessLogsForSingleUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.FirstAndLastAccessLogsForSingleUserFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessReport.filterFirstAndtLastAccessLogsForSingleUser(organizationId, filter, filter.pagination);
    res.json(dbResult);
});
module.exports.getFirstAndLastAccessLogsExcelForSingleUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.FirstAndLastAccessLogsForSingleUserFilterForExcel(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = filter;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.FirstAndLastAccessLogsForSingleUser,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAccessLogsWithUserCaptions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.AccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
    if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
        filter.accessControlPointIds = filter.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
    }
    else {
        filter.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
    }
    if (filter.accessControlPointIds.length < 1) {
        return res.json({
            total: 0,
            skip: filter.skip,
            take: filter.take,
            items: [],
        });
    }
    else {
        res.json(await dal_manager_1.dbManager.accessAccessLog.filterAccessLogsWithUserCaptions(organizationId, filter, authResult.jwt.userId, !authResult.hasOrganizationWide, authResult.permittedUnitIds));
    }
});
module.exports.getArventoAccessLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.ArventoAccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            filter.accessControlPointIds = filter.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            filter.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    res.json(await dal_manager_1.dbManager.accessAccessLog.getArventoAccessLogs(organizationId, filter, authResult.jwt.userId, !authResult.hasOrganizationWide, authResult.permittedUnitIds));
});
module.exports.getArventoAccessLogsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = new api_user_v1_models_1.RestUserApiV1.ArventoAccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
        if (request.accessControlPointIds && request.accessControlPointIds.length > 0) {
            request.accessControlPointIds = request.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            request.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    try {
        const reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            const filter = request;
            filter.permittedUnitIds = authResult.permittedUnitIds;
            filter.authorizeForFilter = !authResult.hasOrganizationWide;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.ArventoAccessLogsReport,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: filter,
                organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAccessLogsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = new api_user_v1_models_1.RestUserApiV1.AccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
        if (request.accessControlPointIds && request.accessControlPointIds.length > 0) {
            request.accessControlPointIds = request.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            request.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    try {
        const reportResult = await new Promise((resolve, reject) => {
            delete request.skip;
            delete request.take;
            delete request.returnTotalCount;
            const filter = request;
            filter.permittedUnitIds = authResult.permittedUnitIds;
            filter.authorizeForFilter = !authResult.hasOrganizationWide;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AccessLogs,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: filter,
                organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAccessLogsPDF = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.AccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            filter.accessControlPointIds = filter.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            filter.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete filter.skip;
            delete filter.take;
            delete filter.returnTotalCount;
            const options = filter;
            options.permittedUnitIds = authResult.permittedUnitIds;
            options.authorizeForFilter = !authResult.hasOrganizationWide;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AccessLogs,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getSensorEventsPDF = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.SensorEventsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSensorEvents)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete filter.pagination;
            let options = filter;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.SensorEvents,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getSensorEventsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.SensorEventsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSensorEvents)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete filter.pagination;
            let options = filter;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.SensorEvents,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAccessLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.AccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            filter.accessControlPointIds = filter.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            filter.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    res.json(await dal_manager_1.dbManager.accessAccessLog.filterAccessLogs(organizationId, filter, authResult.jwt.userId, !authResult.hasOrganizationWide, true, userAccessRights));
});
module.exports.getMyAccessLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.MyAccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessSelfReports)(req, organizationId);
    res.json(await dal_manager_1.dbManager.accessAccessLog.filterAccessLogsWithUserCaptions(organizationId, {
        startUtc: filter.startUtc,
        endUtc: filter.endUtc,
        accessControlPointIds: filter.accessControlPointIds,
        sortDateDesc: filter.sortDateDesc,
        accessResult: filter.accessResult,
        take: filter.take,
        skip: filter.skip,
        returnTotalCount: filter.returnTotalCount,
        direction: filter.direction,
        showOnlyRemoteAccess: filter.showOnlyRemoteAccess,
        showOnlyManuallyInserted: filter.showOnlyManuallyInserted,
        userIds: [authResult.jwt.userId],
        authorizeForFilter: false,
        reasons: filter.reasons,
        credentialTypes: filter.credentialTypes,
    }, authResult.jwt.userId, false));
});
module.exports.getVisitorAccessLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.VisitorAccessLogsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForVisitorAccessReport)(req, organizationId);
    let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
    if (!authResult.jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            filter.accessControlPointIds = filter.accessControlPointIds.filter((u) => userAccessRights.some((a) => u == a.accessControlPointId && a.read));
        }
        else {
            filter.accessControlPointIds = userAccessRights.filter((a) => a.read).map((a) => a.accessControlPointId);
        }
    }
    res.json(await dal_manager_1.dbManager.accessAccessLog.filterVisitorAccessLogs(organizationId, filter, authResult.jwt.userId, !authResult.hasOrganizationWide, true, userAccessRights));
});
module.exports.getAccessControlPointSummaryReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let filter = new api_user_v1_models_1.RestUserApiV1.AccessControlPointSummaryReportFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForAccessControlPointSummaryReport)(req, organizationId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointSummaryReport(organizationId, accessControlPointId, authResult.jwt.userId, filter, authResult.hasOrganizationWide, authResult.jwt));
});
module.exports.getAccessControlPointDetailedReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    throw new Error("Not implemented!");
});
module.exports.getTerminalEventsReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.TerminalEventsReportRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessLog.listTerminalEventsReport({ organizationId, filter, requesterUserId: authResult.userId });
    res.json(dbResult);
});
module.exports.getSensorEventsReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.SensorEventsReportRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessDevice.reportSensorsPg({ organizationId, filter, requesterUserId: authResult.userId });
    res.json(dbResult);
});
module.exports.getSensorInformationReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.SensorInformationReportRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessDevice.reportSensorInformation({ organizationId, filter, requesterUserId: authResult.userId });
    res.json(dbResult);
});
module.exports.getCustomReportMetadata = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let result = [];
    let customReportsOfOrganizations = await dal_manager_1.dbManager.accessReport.getCustomReportList(organizationId);
    for (const customReport of customReportsOfOrganizations) {
        try {
            if (customReport.minPermissions && customReport.minPermissions.length > 0) {
                (0, api_securityhelper_1.authorizeForCustomReport)(req, organizationId, customReport.minPermissions);
            }
            customReport.filters = customReport.filters.sort((a, b) => b.type - a.type);
            result.push(customReport);
        }
        catch (err) {
            app_logs_1.logger.info("User has no permission to see report " + customReport.name);
        }
    }
    res.json(result);
});
module.exports.getCustomReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const customReportId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "customReportId");
    const customReport = await dal_manager_1.dbManager.accessReport.getCustomReport(customReportId, organizationId);
    try {
        if (customReport.minPermissions && customReport.minPermissions.length > 0) {
            (0, api_securityhelper_1.authorizeForCustomReport)(req, organizationId, customReport.minPermissions);
        }
        customReport.filters = customReport.filters.sort((a, b) => b.type - a.type);
        res.json(customReport);
    }
    catch (err) {
        app_logs_1.logger.info("User has no permission to see report " + customReport.name);
    }
});
module.exports.generateCustomReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let reportId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "reportId");
    let customReportDef = await dal_manager_1.dbManager.accessReport.getCustomReport(reportId, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.CustomReportFilterRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let userId = null;
    if (customReportDef.minPermissions && customReportDef.minPermissions.length > 0) {
        let authResult = (0, api_securityhelper_1.authorizeForCustomReport)(req, organizationId, customReportDef.minPermissions);
        userId = authResult.jwt.userId;
    }
    else {
        userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.CustomReport,
                format: request.format,
                filter: {
                    filters: request.filters,
                    reportId: reportId,
                },
                organizationId: organizationId,
                requesterUserId: userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 10000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getPreparedReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const reportId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "reportId");
    const userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    const jwt = req.auth_token;
    const getPreparedReport = await dal_manager_1.dbManager.accessReport.getPreparedReport(reportId, jwt.getOrganizationId());
    if (!getPreparedReport) {
        res.status(404).json({
            message: "Report not found!",
        });
    }
    if (getPreparedReport.expirationDt < new Date()) {
        res.status(410).json({
            message: "Report had expired!",
        });
    }
    try {
        const contentType = getPreparedReport.format === app_enums_1.enums.ReportFormat.Excel
            ? dal_constants_1.DalConstants.excelContentType
            : getPreparedReport.format === app_enums_1.enums.ReportFormat.Pdf
                ? dal_constants_1.DalConstants.pdfContentType
                : dal_constants_1.DalConstants.csvContentType;
        res.sendFile(reportId, {
            root: app_config_1.appConfig.preparedReportDirectory,
            headers: {
                "content-type": contentType,
            },
        });
    }
    catch (error) {
        res.status(410).json({
            message: "Report had expired!",
        });
    }
});
module.exports.getHelpDocument = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    if (app_config_1.appConfig.organizationalFileStorageDirectory &&
        fs_1.default.existsSync(app_config_1.appConfig.organizationalFileStorageDirectory) &&
        fs_1.default.existsSync(path_1.default.join(app_config_1.appConfig.organizationalFileStorageDirectory, dal_constants_1.DalConstants.helpDocumentId))) {
        res.sendFile(dal_constants_1.DalConstants.helpDocumentId, {
            root: app_config_1.appConfig.organizationalFileStorageDirectory,
            headers: {
                "content-type": dal_constants_1.DalConstants.pdfContentType,
            },
        });
    }
    else {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Help Document Not Found!" });
    }
});
module.exports.getAuditIdentityActionHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    if (!authResult.hasOrganizationWide && !request.organizationUnitIds?.length) {
        request.organizationUnitIds = authResult.permittedUnitIds;
        request.filterOrganizationUnitMembersHierarchically = true;
    }
    const dbResult = await dal_manager_1.dbManager.accessLog.listUsersIdentityActionHistory(organizationId, app_enums_1.enums.UserActionCategory.Main, request.actionType
        ? [request.actionType]
        : [
            app_enums_1.enums.UserMainActionType.AddAccessRight,
            app_enums_1.enums.UserMainActionType.AddCredential,
            app_enums_1.enums.UserMainActionType.InsertIdentity,
            app_enums_1.enums.UserMainActionType.RemoveAccessRight,
            app_enums_1.enums.UserMainActionType.RemoveCredential,
            app_enums_1.enums.UserMainActionType.UpdateIdentity,
            app_enums_1.enums.UserMainActionType.UpdateProfile,
            app_enums_1.enums.UserMainActionType.UpdateCredential,
            app_enums_1.enums.UserMainActionType.AssignUserToUserGroup,
            app_enums_1.enums.UserMainActionType.RemoveUserFromUserGroup,
            app_enums_1.enums.UserMainActionType.AssignUserToOrganizationUnit,
            app_enums_1.enums.UserMainActionType.RemoveUserFromOrganizationUnit,
            app_enums_1.enums.UserMainActionType.UpdateUserOrganizationUnitRole,
            app_enums_1.enums.UserMainActionType.AccountDeleted,
            app_enums_1.enums.UserMainActionType.UsernameUpdated,
            app_enums_1.enums.UserMainActionType.AccountCreated,
        ], request);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    const roles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(organizationId);
    res.json({
        items: (0, business_main_1.transformAuditMainActionItemsLocalizedMessages)(dbResult.items, locale, roles),
        pagination: dbResult.pagination,
    });
});
module.exports.getAuditIdentityActionHistoryExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            options.permittedUnitIds = authResult.hasOrganizationWide ? undefined : authResult.permittedUnitIds;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AuditLogsIdentity,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAuditAccessControlPointActionHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const userAccessRightsAcpList = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
    const dbResult = await dal_manager_1.dbManager.accessLog.listUsersAccessControlPointActionHistory(organizationId, app_enums_1.enums.UserActionCategory.Main, request.actionType
        ? [request.actionType]
        : [app_enums_1.enums.UserMainActionType.UpdateAccessControlPoint, app_enums_1.enums.UserMainActionType.UpdateAccessControlPointMapping, app_enums_1.enums.UserMainActionType.InsertAccessControlPoint], request, userAccessRightsAcpList.filter((acp) => acp.read).map((acp) => acp.accessControlPointId));
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    res.json({
        items: (0, business_main_1.transformAuditMainActionItemsLocalizedMessages)(dbResult.items, locale),
        pagination: dbResult.pagination,
    });
});
module.exports.getAuditAccessControlPointActionHistoryExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AuditLogsAccessControlPoint,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.accessControlPointsDetailsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendanceControlReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.AccessControlPointFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AccessControlPoints,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, "Erişim_Kontrol_Raporu" + ".xlsx", (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.accessControlPointsDetailsPdf = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendanceControlReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.AccessControlPointFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AccessControlPoints,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, "Erişim_Kontrol_Raporu" + ".xlsx", (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAuditTerminalActionHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const terminals = await dal_manager_1.dbManager.accessDevice.listTerminals(organizationId, {
        pagination: {
            skip: 0,
            take: 10000,
        },
    }, authResult.jwt.userId);
    const dbResult = await dal_manager_1.dbManager.accessLog.listUsersTerminalActionHistory(organizationId, app_enums_1.enums.UserActionCategory.Main, request.actionType
        ? [request.actionType]
        : [app_enums_1.enums.UserMainActionType.UpdateStatusSensor, app_enums_1.enums.UserMainActionType.UpdateRelay, app_enums_1.enums.UserMainActionType.UpdateReader, app_enums_1.enums.UserMainActionType.UpdateCounterSensor], request, terminals.items.map((i) => i.id));
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    res.json({
        items: (0, business_main_1.transformAuditMainActionItemsLocalizedMessages)(dbResult.items, locale),
        pagination: dbResult.pagination,
    });
});
module.exports.getAuditTerminalActionHistoryExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AuditLogsTerminal,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getAccessGrantedIdentitiesForAccessControlPointAsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    new api_validatorhelper_1.ValidatorHelper().validateUUID("organizationId", false, req.swagger.params.oid.value).validateUUID("accessControlPointId", false, req.swagger.params.aid.value).finalize();
    let organizationId = req.swagger.params.oid.value;
    let accessControlPointId = req.swagger.params.aid.value;
    let jwt = req.auth_token;
    await (0, controller_accesscontrolpoint_1.canViewAccessControlPoint)(organizationId, jwt.userId, accessControlPointId);
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = {
                accessControlPointId: accessControlPointId,
            };
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.AccessControlPointUsers,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listSelfOrganizationFeedbackExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_user_v1_models_1.RestUserApiV1.ListFeedbackRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.requestUserId = jwt.userId;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.SystemFeedback,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listOrganizationFeedbackExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let jwt = req.auth_token;
    let request = new api_user_v1_models_1.RestUserApiV1.ListFeedbackRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.requestUserId = jwt.userId;
    const authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.SystemFeedback,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
