"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const business_system_1 = require("../../../../business/business.system");
const report_util_1 = require("../../../../business/report/report.util");
const dal_access_error_1 = require("../../../../dal/access/dal.access.error");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const i18n_1 = __importDefault(require("i18n"));
const api_user_v1_models_1 = require("../api.user.v1.models");
module.exports.upsertRegion = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    const region = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        if (request.id) {
            let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.userId, regionId: request.id, trx });
            const missingPermissions = [];
            if (!userRights || (!userRights.read && !userRights.write)) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
            }
            else {
                if (!userRights.read) {
                    missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
                }
                else if (!userRights.write) {
                    missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
                }
            }
            if (missingPermissions.length) {
                (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
            }
        }
        return dal_manager_1.dbManager.accessRegion.upsertRegion(organizationId, request, trx);
    }, authResult.userId, organizationId);
    res.json(region);
});
module.exports.removeRegion = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    const force = req.swagger.params.force.value;
    const authResult = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId, trx });
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.userId, regionId, trx });
        const missingPermissions = [];
        if (!userRights || (!userRights.read && !userRights.write)) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
        }
        else {
            if (!userRights.read) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
            }
            else if (!userRights.write) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
            }
        }
        if (missingPermissions.length) {
            (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        if (!force) {
            const warningsToReport = [];
            const accessRuleSetsOfRegion = await dal_manager_1.dbManager.accessRegion.getAccessRuleSetIdNamePairsOfRegion(organizationId, regionId, trx);
            const regionTicketsOfRegion = await dal_manager_1.dbManager.accessRegion.getRegionTicketIdNamePairsOfRegion(organizationId, regionId, trx);
            const usedForAccessNotifications = await dal_manager_1.dbManager.accessRegion.isRegionUsedForAccessNotifications(organizationId, regionId, trx);
            const usedForForbiddances = await dal_manager_1.dbManager.accessRegion.isRegionUsedForForbiddances(organizationId, regionId, trx);
            if (usedForForbiddances) {
                warningsToReport.push(i18n_1.default.__({ phrase: "ERRORS.REGION.IN_USE_FORBIDDANCES", locale }));
            }
            if (usedForAccessNotifications) {
                warningsToReport.push(i18n_1.default.__({ phrase: "ERRORS.REGION.IN_USE_ACCESS_NOTIFICATION", locale }));
            }
            if (accessRuleSetsOfRegion.length) {
                const arn = `"` + accessRuleSetsOfRegion.map((m) => m.name).join(`", "`) + `"`;
                warningsToReport.push(i18n_1.default.__({ phrase: "ERRORS.REGION.IN_USE_ACCESS_RULE_SETS", locale }, { arn }));
            }
            if (regionTicketsOfRegion.length) {
                const rtn = `"` + regionTicketsOfRegion.map((m) => m.name).join(`", "`) + `"`;
                warningsToReport.push(i18n_1.default.__({ phrase: "ERRORS.REGION.IN_USE_REGION_TICKET", locale }, { rtn }));
            }
            if (warningsToReport.length) {
                res.status(app_enums_1.enums.HttpStatusCode.REGION_DELETE_DEPENDENCY).json({
                    warnings: warningsToReport,
                });
                return;
            }
        }
        await dal_manager_1.dbManager.accessRegion.removeRegion({ organizationId, regionId, trx });
        res.json({});
    }, authResult.userId, organizationId);
});
module.exports.getRegionDetailed = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    const authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        return await dal_manager_1.dbManager.accessRegion.getRegionDetailed({ organizationId, regionId, trx });
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.getAccessControlPointsOfRegion = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    const authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        return await dal_manager_1.dbManager.accessRegion.getAccessControlPointsOfRegion(organizationId, {
            regionId,
            options: {
                take: request.take,
                skip: request.skip,
                filter: request.filter,
            },
        }, trx);
    }, authResult.jwt.userId, organizationId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.upsertRegionAccessControlPoints = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    let request = req.body;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.userId, regionId, trx });
        const missingPermissions = [];
        if (!userRights || (!userRights.read && !userRights.write)) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
        }
        else {
            if (!userRights.read) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
            }
            else if (!userRights.write) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
            }
        }
        if (missingPermissions.length) {
            (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        await dal_manager_1.dbManager.accessRegion.upsertRegionAccessControlPoints(organizationId, regionId, request);
    }, authResult.userId, organizationId);
    res.json({});
});
module.exports.genericSearchRegion = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return await dal_manager_1.dbManager.accessRegion.genericSearchRegion({
            organizationId,
            options: {
                take: request.take,
                skip: request.skip,
                filter: request.filter,
            },
            requesterUserId: authResult.jwt.userId,
            trx,
        });
    }, authResult.jwt.userId, organizationId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.listRegions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    const regionIds = [];
    let request = new api_user_v1_models_1.RestUserApiV1.ListRegionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const regionsWithAdministrators = await dal_manager_1.dbManager.accessRegion.getAdministratorsOfRegions({
            organizationId,
            regionIds: request.regionIds,
            trx,
        });
        for (const regionId of request.regionIds) {
            const regionWithAdmins = regionsWithAdministrators.find((r) => r.id === regionId);
            if (regionWithAdmins) {
                const userRightsForRegion = regionWithAdmins.userRegionRights.find((urr) => urr.userId === authResult.jwt.userId);
                if (!userRightsForRegion || !userRightsForRegion.userRights.read) {
                    continue;
                }
            }
            regionIds.push(regionId);
        }
        return await dal_manager_1.dbManager.accessRegion.listRegions({
            organizationId,
            options: {
                regionIds: regionIds,
                skip: request.skip,
                take: request.take,
            },
            requesterUserId: authResult.jwt.userId,
            trx,
        });
    }, authResult.jwt.userId, organizationId);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
            };
        }),
    });
});
module.exports.regionStateReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.RegionStateReportRequestBody(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        return await dal_manager_1.dbManager.accessRegion.getRegionStateReportNew(organizationId, {
            pagination: request.pagination,
            regionId: regionId,
            userIds: request.userIds,
            userGroupIds: request.userGroupIds,
            organizationUnitIds: request.organizationUnitIds,
            applyOrganizationUnitFilterHierarchically: request.applyOrganizationUnitFilterHierarchically,
            state: request.state,
            status: request.status,
            includeUserCaptions: request.includeUserCaptions,
            sortOrder: request.sortOrder,
            sortType: request.sortType,
        }, authResult.jwt.userId);
    }, authResult.jwt.userId, organizationId);
    if (request.nestedPagination) {
        res.json(dbResult);
    }
    else {
        res.json({
            total: dbResult.pagination.total,
            take: dbResult.pagination.take,
            skip: dbResult.pagination.skip,
            items: dbResult.items,
        });
    }
});
module.exports.regionStateSummaryForDashboard = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.RegionStateReportRequestBody(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessRegion.getRegionStateSummaryForDashboard(organizationId, {
            pagination: request.pagination,
            regionId: regionId,
            userIds: request.userIds,
            userGroupIds: request.userGroupIds,
            organizationUnitIds: request.organizationUnitIds,
            applyOrganizationUnitFilterHierarchically: request.applyOrganizationUnitFilterHierarchically,
            state: request.state,
            status: request.status,
        }, authResult.jwt.userId, trx);
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.regionStateReportExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.RegionStateReportRequestBody(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
    }, authResult.jwt.userId, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let filters = request;
            filters.regionId = regionId;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.RegionState,
                format: app_enums_1.enums.ReportFormat.Excel,
                organizationId,
                filter: filters,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Region State is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw new Error("Timeout");
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.unlockUserStateForRegion = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let jwt = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: jwt.userId, regionId, trx });
        const missingPermissions = [];
        if (!userRights || (!userRights.read && !userRights.write)) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
        }
        else {
            if (!userRights.read) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
            }
            else if (!userRights.write) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
            }
        }
        if (missingPermissions.length) {
            (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        await dal_manager_1.dbManager.accessRegion.unlockUserStateForRegion(organizationId, { regionId, userId }, trx);
    }, jwt.userId, organizationId);
    res.json({});
});
module.exports.regionEmergencyStart = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemEmergency)(req, organizationId);
    const terminalIds = await dal_manager_1.dbManager.accessRegion.getRegionTerminalIds(organizationId, regionId);
    if (terminalIds.length === 0) {
        (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.GENERAL.NO_TERMINAL_IN_REGION");
    }
    const region = (await dal_manager_1.dbManager.accessRegion.getRegionIdName(organizationId, [regionId]))[0];
    (0, business_system_1.handleRegionEmergency)({
        organizationId,
        regionId: region.id,
        regionName: region.name,
        relatedTerminalIds: terminalIds,
        state: app_enums_1.enums.EmergencyState.Emergency,
    });
    res.json({});
});
module.exports.regionEmergencyEnd = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemEmergency)(req, organizationId);
    const terminalIds = await dal_manager_1.dbManager.accessRegion.getRegionTerminalIds(organizationId, regionId);
    const region = (await dal_manager_1.dbManager.accessRegion.getRegionIdName(organizationId, [regionId]))[0];
    (0, business_system_1.handleRegionEmergency)({
        organizationId,
        regionId: region.id,
        regionName: region.name,
        relatedTerminalIds: terminalIds,
        state: app_enums_1.enums.EmergencyState.Normal,
    });
    res.json({});
});
module.exports.getRegionList = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = req.body;
    const authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    const regionDetails = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const regions = await dal_manager_1.dbManager.accessRegion.genericSearchRegion({
            organizationId,
            options: {
                take: request.pagination.take,
                skip: request.pagination.skip,
                filter: request.filter,
            },
            requesterUserId: authResult.jwt.userId,
            trx,
        });
        return dal_manager_1.dbManager.accessRegion.getRegionList({
            organizationId,
            regionIds: regions.items.map((r) => r.id),
            options: { ...request.pagination },
            trx,
            requesterUserId: authResult.jwt.userId,
        });
    }, authResult.jwt.userId, organizationId);
    res.json(regionDetails);
});
module.exports.upsertRegionAdministrators = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.userId, regionId });
    const missingPermissions = [];
    if (!userRights || (!userRights.read && !userRights.write)) {
        missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
    }
    else {
        if (!userRights.read) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
        }
        else if (!userRights.write) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
        }
    }
    if (missingPermissions.length) {
        (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
    }
    const accessRights = new api_user_v1_models_1.RestUserApiV1.UpsertAdministrator(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessRegion.upsertRegionAdministrator(organizationId, authResult.userId, regionId, accessRights);
    res.json(dbResult);
});
module.exports.removeRegionAdministrator = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userRegionRightId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.urrid.value, "userRegionRightId");
    let authResult = (0, api_securityhelper_1.authorizeForWriteRegion)(req, organizationId);
    let regionId = await dal_manager_1.dbManager.accessRegion.getRegionIdForUserRegionRightId(organizationId, userRegionRightId);
    if (regionId) {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.userId, regionId });
        const missingPermissions = [];
        if (!userRights || (!userRights.read && !userRights.write)) {
            missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific());
        }
        else {
            if (!userRights.read) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getReadRegionSpecific());
            }
            else if (!userRights.write) {
                missingPermissions.push(predefined_permissions_1.Permissions.region.getWriteRegionSpecific());
            }
        }
        if (missingPermissions.length) {
            (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.region.getWriteRegionSpecific(), predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        await dal_manager_1.dbManager.accessRegion.removeRegionAdministrator(organizationId, userRegionRightId);
        res.send(200);
    }
    else {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.NOT_FOUND, "ERRORS.GENERAL.REGIONRIGHTNOTFOUND");
    }
});
module.exports.listRegionAdministrators = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForGetRegions)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId });
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this region", [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
    }
    const accessRights = new api_user_v1_models_1.RestUserApiV1.ListRegionAdministratorsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessRegion.getRegionAdministrators(organizationId, regionId, accessRights.pagination);
    res.json(dbResult);
});
module.exports.durationInRegionReportOverview = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.DurationInRegionReportOverviewFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        return await dal_manager_1.dbManager.accessRegion.getDurationInRegionReportOverview(organizationId, authResult.jwt.userId, trx, Object.assign(request, { regionId: regionId }), undefined);
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.durationInRegionReportOverviewExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "regionId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.DurationInRegionReportOverviewFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
    }, authResult.jwt.userId, organizationId);
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let filters = request;
            filters.regionId = regionId;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.DurationInRegionReportOverview,
                format: app_enums_1.enums.ReportFormat.Excel,
                organizationId,
                filter: filters,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Duration In Region is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw new Error("Timeout");
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.durationInRegionReportDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.rid.value, "regionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let authResult = (0, api_securityhelper_1.authorizeForRegionStateReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.DurationInRegionReportDetailFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let userRights = await dal_manager_1.dbManager.accessRegion.getUserRegionRights({ organizationId, requestingUserId: authResult.jwt.userId, regionId, trx });
        if (!userRights || !userRights.read) {
            (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.region.getReadRegionSpecific()], [predefined_permissions_1.Permissions.region.getReadRegionSpecific()]);
        }
        return await dal_manager_1.dbManager.accessRegion.getDurationInRegionReportDetail(organizationId, Object.assign(request, { regionId: regionId, userId: userId }), authResult.jwt.userId, trx);
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
