"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const app_enums_1 = require("../../../../app.enums");
const dal_manager_1 = require("../../../../dal/dal.manager");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
module.exports.addNotification = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const request = req.body;
    const notificationId = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessNotifications.addNotification({
            createdByUserId: auth.jwt.userId,
            data: request,
            organizationId,
            trx,
        });
    }, auth.jwt.userId, organizationId);
    res.json(notificationId);
});
module.exports.updateNotification = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const notificationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "notificationId");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    const request = req.body;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessNotifications.updateNotification({
            id: notificationId,
            organizationId,
            data: request,
            trx,
        });
    }, auth.jwt.userId, organizationId);
    res.json({});
});
module.exports.deleteNotification = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const notificationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "notificationId");
    const auth = (0, api_securityhelper_1.authorizeForAccessReport)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessNotifications.deleteNotification({ id: notificationId, organizationId, trx });
    }, auth.jwt.userId, organizationId);
    res.json({});
});
const deprecatedListNotification = async (req, res, next, state) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const request = req.body;
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const list = await dal_manager_1.dbManager.accessNotifications.listNotificationInstance(organizationId, {
            pagination: {
                skip: request.skip,
                take: request.take,
            },
            filter: {
                archived: false,
                mediums: {
                    pushNotification: request.medium === app_enums_1.enums.NotificationMedium.PushNotification,
                    web: request.medium !== app_enums_1.enums.NotificationMedium.PushNotification,
                    email: false,
                    sms: false,
                },
                userIds: [jwt.userId],
                states: [state],
                types: request.type ? [request.type] : undefined,
                eventTime: request.dateRange
                    ? {
                        start: request.dateRange.startDateTime,
                        end: request.dateRange.endDateTime,
                    }
                    : undefined,
            },
            sorting: {
                field: "createdT",
                order: "DESC",
            },
            locale,
        }, trx);
        return {
            pagination: list.pagination,
            notifications: list.items?.length > 0
                ? list.items.map((m) => {
                    const instance = m.instances[0];
                    return {
                        message: {
                            body: instance.content?.body,
                            title: instance.content?.title,
                        },
                        notificationDate: m.createdT,
                        notificationType: m.type,
                        organizationId,
                        userId: m.userBadge.id,
                        id: instance.notificationInstanceId,
                        medium: m.instances[0].medium,
                        notificationEventId: m.eventId,
                        data: instance.content?.data,
                    };
                })
                : [],
        };
    }, jwt.userId, organizationId);
    res.json(result);
};
module.exports.listUnreadNotifications = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    await deprecatedListNotification(req, res, next, 4);
});
module.exports.listReadNotifications = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    await deprecatedListNotification(req, res, next, 5);
});
module.exports.markNotificationAsRead = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const instanceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.nid.value, "notificationId");
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const unreadCount = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId,
            organizationId,
            state: 5,
            trx,
        });
        return dal_manager_1.dbManager.accessNotifications.getUnreadNotificationsCount({ organizationId, userId: jwt.userId, trx });
    }, jwt.userId, organizationId);
    res.json(unreadCount);
});
module.exports.markAllNotificationsAsRead = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const unreadCount = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessNotifications.setAllNotificationInstancesOfUserAsRead({
            userId: jwt.userId,
            medium: req.body.medium,
            trx,
            organizationId,
        });
        return dal_manager_1.dbManager.accessNotifications.getUnreadNotificationsCount({ organizationId, userId: jwt.userId, trx });
    }, jwt.userId, organizationId);
    res.json(unreadCount);
});
module.exports.getUnreadNotificationCount = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const unreadCount = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessNotifications.getUnreadNotificationsCount({ organizationId, userId: jwt.userId, trx });
    }, jwt.userId, organizationId);
    res.json(unreadCount);
});
module.exports.listNotificationInstances = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const request = req.body;
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
    request.locale = locale;
    const result = await dal_manager_1.dbManager.organizationTransaction((trx) => {
        return dal_manager_1.dbManager.accessNotifications.listNotificationInstance(organizationId, request, trx);
    }, jwt.userId, organizationId);
    res.json(result);
});
module.exports.listNotificationEvents = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const request = req.body;
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return request.filter.archived
            ? dal_manager_1.dbManager.accessNotifications.listDeletedNotificationEvents({
                ...request,
                organizationId,
                trx,
                requesterUserId: jwt.userId,
            })
            : dal_manager_1.dbManager.accessNotifications.listNotificationEvents({
                ...request,
                organizationId,
                trx,
                requesterUserId: jwt.userId,
            });
    }, jwt.userId, organizationId);
    res.json(result);
});
