"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const lodash_1 = __importDefault(require("lodash"));
const path_1 = __importDefault(require("path"));
const i18n_1 = __importDefault(require("i18n"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const app_util_1 = require("../../../../app.util");
const business_system_1 = require("../../../../business/business.system");
const report_util_1 = require("../../../../business/report/report.util");
const dal_access_cache_1 = require("../../../../dal/access/access-interfaces/dal.access.cache");
const dal_access_error_1 = require("../../../../dal/access/dal.access.error");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const messageBroker_server_to_device_pub_1 = require("../../../../messageBroker/messageBroker.server-to-device.pub");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
module.exports.getArmonControlPanelDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "armonControlPanelDeviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let dbDetailed = await dal_manager_1.dbManager.accessDevice.getArmonControlPanelDetailes(organizationId, deviceId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let response = {
        accessControlPoints: dbDetailed.accessControlPoints,
        configuration: dbDetailed.configuration,
        deviceId: dbDetailed.deviceId,
        extensionControlPanels: dbDetailed.extensionControlPanels.map((ep) => {
            if (ep.io && ep.io.dryContactInputs) {
                ep.io.dryContactInputs = ep.io.dryContactInputs.map((dryContactInput) => {
                    switch (dryContactInput.type) {
                        case app_enums_1.enums.DryContactType.ExitButton:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.DryContactType.StatusSensor:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.DryContactType.Counter:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                            break;
                        default:
                            break;
                    }
                    delete dryContactInput["settings"];
                    return dryContactInput;
                });
            }
            return ep;
        }),
        integrations: dbDetailed.integrations.map((r) => {
            return {
                model: r.model,
                antennas: r.type != app_enums_1.enums.ArmonControlPanelIntegrationType.ImpinjSpeedWayRs485
                    ? []
                    : r.settings.antennas,
                lanes: r.type == app_enums_1.enums.ArmonControlPanelIntegrationType.ImpinjSpeedWayRs485
                    ? []
                    : r.settings.lanes,
                extensionFields: r.extensionFields,
                integratingDeviceId: r.integratingDeviceId,
                location: r.location,
                name: r.name,
                serialNumber: r.serialNumber,
                type: r.type,
            };
        }),
        io: {
            analogInputs: dbDetailed.io.analogInputs,
            batteryPlugged: dbDetailed.io.batteryPlugged,
            dryContactInputs: dbDetailed.io.dryContactInputs.map((dryContactInput) => {
                switch (dryContactInput.type) {
                    case app_enums_1.enums.DryContactType.ExitButton:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.DryContactType.StatusSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.DryContactType.Counter:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                        break;
                    default:
                        break;
                }
                delete dryContactInput["settings"];
                return dryContactInput;
            }),
            emergencyInputPlugged: dbDetailed.io.emergencyInputPlugged,
            relays: dbDetailed.io.relays,
            wiegandReaders: dbDetailed.io.wiegandReaders,
        },
        settings: dbDetailed.settings,
    };
    res.json(response);
});
module.exports.getArmonControlPanelDetailsV2 = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "armonControlPanelDeviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let dbDetailed = await dal_manager_1.dbManager.accessDevice.getArmonControlPanelDetailesV2(organizationId, deviceId);
    let response = {
        accessControlPoints: dbDetailed.accessControlPoints,
        configuration: dbDetailed.configuration,
        deviceId: dbDetailed.deviceId,
        extensionControlPanels: dbDetailed.extensionControlPanels.map((ep) => {
            if (ep.io && ep.io.dryContactInputs) {
                ep.io.dryContactInputs = ep.io.dryContactInputs.map((dryContactInput) => {
                    switch (dryContactInput.type) {
                        case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                            break;
                        default:
                            break;
                    }
                    delete dryContactInput["settings"];
                    return dryContactInput;
                });
            }
            return ep;
        }),
        integrations: dbDetailed.integrations.map((r) => {
            return {
                model: r.model,
                extensionFields: r.extensionFields,
                integratingDeviceId: r.integratingDeviceId,
                location: r.location,
                name: r.name,
                serialNumber: r.serialNumber,
                type: r.type,
            };
        }),
        io: {
            analogInputs: dbDetailed.io.analogInputs,
            batteryPlugged: dbDetailed.io.batteryPlugged,
            dryContactInputs: dbDetailed.io.dryContactInputs.map((dryContactInput) => {
                switch (dryContactInput.type) {
                    case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                        break;
                    default:
                        break;
                }
                delete dryContactInput["settings"];
                return dryContactInput;
            }),
            emergencyInputPlugged: dbDetailed.io.emergencyInputPlugged,
            relays: dbDetailed.io.relays,
            readers: dbDetailed.io.readers,
        },
        settings: dbDetailed.settings,
    };
    res.json(response);
});
module.exports.getAperioAH40Details = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "armonControlPanelDeviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let dbDetailed = await dal_manager_1.dbManager.accessDevice.getAperioAH40Details(organizationId, deviceId);
    let response = {
        accessControlPoints: dbDetailed.accessControlPoints,
        configuration: dbDetailed.configuration,
        deviceId: dbDetailed.deviceId,
        locks: dbDetailed.locks.map((ep) => {
            if (ep.io && ep.io.dryContactInputs) {
                ep.io.dryContactInputs = ep.io.dryContactInputs.map((dryContactInput) => {
                    switch (dryContactInput.type) {
                        case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                            break;
                        case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                            dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                            break;
                        default:
                            break;
                    }
                    delete dryContactInput["settings"];
                    return dryContactInput;
                });
            }
            return ep;
        }),
        integrations: dbDetailed.integrations,
        io: {
            dryContactInputs: dbDetailed.io.dryContactInputs.map((dryContactInput) => {
                switch (dryContactInput.type) {
                    case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                        break;
                    default:
                        break;
                }
                delete dryContactInput["settings"];
                return dryContactInput;
            }),
            relays: dbDetailed.io.relays,
            readers: dbDetailed.io.readers,
        },
        settings: dbDetailed.settings,
    };
    res.json(response);
});
module.exports.getHikVisionDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "armonControlPanelDeviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let dbDetailed = await dal_manager_1.dbManager.accessDevice.getHikVisionDetails(organizationId, deviceId);
    let response = {
        accessControlPoints: dbDetailed.accessControlPoints,
        configuration: dbDetailed.configuration,
        deviceId: dbDetailed.deviceId,
        integrations: dbDetailed.integrations,
        io: {
            dryContactInputs: dbDetailed.io.dryContactInputs.map((dryContactInput) => {
                switch (dryContactInput.type) {
                    case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                        break;
                    default:
                        break;
                }
                delete dryContactInput["settings"];
                return dryContactInput;
            }),
            relays: dbDetailed.io.relays,
            readers: dbDetailed.io.readers,
        },
        settings: dbDetailed.settings,
    };
    res.json(response);
});
module.exports.getArmonOneDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "armonControlPanelDeviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let dbDetailed = await dal_manager_1.dbManager.accessDevice.getArmonOneDetails(organizationId, deviceId);
    let response = {
        accessControlPoints: dbDetailed.accessControlPoints,
        configuration: dbDetailed.configuration,
        deviceId: dbDetailed.deviceId,
        integrations: dbDetailed.integrations,
        io: {
            dryContactInputs: dbDetailed.io.dryContactInputs.map((dryContactInput) => {
                switch (dryContactInput.type) {
                    case app_enums_1.enums.libEnumsV2.DryContactType.ExitButton:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { exitButtonSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.StatusSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { statusSensorSettings: dryContactInput.settings });
                        break;
                    case app_enums_1.enums.libEnumsV2.DryContactType.CounterSensor:
                        dryContactInput = lodash_1.default.merge(dryContactInput, { counterSettings: dryContactInput.settings });
                        break;
                    default:
                        break;
                }
                delete dryContactInput["settings"];
                return dryContactInput;
            }),
            relays: dbDetailed.io.relays,
            readers: dbDetailed.io.readers,
        },
        settings: dbDetailed.settings,
    };
    res.json(response);
});
module.exports.getGenericDeviceDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let authResult = (0, api_securityhelper_1.authorizeForGetDeviceDetails)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    let device = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!device) {
        (0, api_error_1.generateNotFoundApiError)({ message: "device not found" });
    }
    res.json({
        adapterId: device.adapterId,
        brand: device.brand,
        model: device.model,
        id: device.id,
        ip: device.ip,
        location: device.location,
        name: device.name,
        port: device.port,
        serialNumber: device.serialNumber,
    });
});
module.exports.updateDeviceInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    let deviceUpdate = req.body;
    await dal_manager_1.dbManager.accessDevice.updateDeviceInfo(organizationId, deviceId, deviceUpdate.name, deviceUpdate.location);
    res.json({});
});
module.exports.listTerminals = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.ListTerminalsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let terminals = await dal_manager_1.dbManager.accessDevice.listTerminals(organizationId, request, authResult.userId);
    res.json(terminals);
});
module.exports.listTerminalsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.ListTerminalsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let jwt = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    request["pagination"] = {
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
    };
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.TerminalList,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listAdapters = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.ListAdaptersRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let adapters = await dal_manager_1.dbManager.accessDevice.listAdapters(organizationId, request);
    res.json(adapters);
});
module.exports.getDeviceStatusInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let deviceStatus;
    let cacheValue = await dal_manager_1.dbManager.accessRedisCache.getValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId);
    if (!cacheValue) {
        deviceStatus = await dal_manager_1.dbManager.accessDevice.getDeviceStatus(organizationId);
        if (deviceStatus) {
            await dal_manager_1.dbManager.accessRedisCache.setValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId, JSON.stringify(deviceStatus));
            cacheValue = deviceStatus;
        }
    }
    else {
        deviceStatus = JSON.parse(cacheValue);
    }
    res.json(deviceStatus);
});
module.exports.genericSearchTerminals = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let terminals = await dal_manager_1.dbManager.accessDevice.listTerminals(organizationId, {
        pagination: {
            skip: 0,
            take: 1000,
        },
    }, authResult.userId);
    let dbResult = await dal_manager_1.dbManager.accessDevice.genericSearchTerminals(organizationId, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
        deviceIds: terminals.items.map((i) => i.id),
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.listTerminalDryContactInputs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "terminalId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = req.body;
    let dbResult = await dal_manager_1.dbManager.accessDevice.listTerminalDryContactInputs(organizationId, terminalId, {
        take: request.take,
        skip: request.skip,
    });
    res.json({
        pagination: dbResult.pagination,
        items: dbResult.items,
    });
});
module.exports.listDryContactInputs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.ListDryContactInputRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessDevice.listTerminalDryContactInputsById(organizationId, request.dryContactInputIds, {
        take: request.take,
        skip: request.skip,
    });
    res.json({
        total: dbResult.pagination.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items,
    });
});
module.exports.genericListSensors = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericListSensorsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessDevice.genericListSensors(organizationId, request.sensorIds, {
        take: request.take,
        skip: request.skip,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items,
    });
});
module.exports.genericSearchSensors = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessDevice.genericSearchSensors(organizationId, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.genericListTerminals = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericListTerminalsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessDevice.genericListTerminals(organizationId, request.terminalIds, {
        take: request.take,
        skip: request.skip,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items,
    });
});
module.exports.getDeviceCertificate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "device is not found" });
    }
    let certPath = path_1.default.resolve(app_config_1.appConfig.deviceCertFolderPath, deviceId + ".zip");
    if (!fs_1.default.existsSync(certPath)) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "device certificate is not found" });
    }
    res.download(certPath);
});
module.exports.emergencyEnd = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, terminalId);
    let authResult = (0, api_securityhelper_1.authorizeForSystemEmergency)(req, organizationId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    if (deviceBasic.brand != app_enums_1.enums.DeviceBrand.Armon && deviceBasic.brand !== app_enums_1.enums.DeviceBrand.AssaAbloy) {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    if (deviceBasic.brand === app_enums_1.enums.DeviceBrand.AssaAbloy && deviceBasic.model !== "AH40") {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    await (0, business_system_1.handleTerminalEmergency)({ organizationId, terminalId: deviceBasic.id, state: app_enums_1.enums.EmergencyState.Normal, terminalName: deviceBasic.name });
    res.json();
});
module.exports.emergencyStart = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    const deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, terminalId);
    const authResult = (0, api_securityhelper_1.authorizeForSystemEmergency)(req, organizationId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    if (deviceBasic.brand != app_enums_1.enums.DeviceBrand.Armon && deviceBasic.brand !== app_enums_1.enums.DeviceBrand.AssaAbloy) {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    if (deviceBasic.brand === app_enums_1.enums.DeviceBrand.AssaAbloy && deviceBasic.model !== "AH40") {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    await (0, business_system_1.handleTerminalEmergency)({ organizationId, terminalId, state: app_enums_1.enums.EmergencyState.Emergency, terminalName: deviceBasic.name });
    res.json();
});
module.exports.restartTerminal = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    if (deviceBasic.brand != app_enums_1.enums.DeviceBrand.Armon && deviceBasic.brand !== app_enums_1.enums.DeviceBrand.AssaAbloy) {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    if (deviceBasic.brand === app_enums_1.enums.DeviceBrand.AssaAbloy && deviceBasic.model !== "AH40") {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    let rebootPromise = new Promise((resolve, reject) => {
        messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
            e: app_enums_1.enums.ServerToDeviceRPCMessageType.Reboot,
            p: null,
        }, 60 * 1000, (err, success) => {
            if (err) {
                return reject(err);
            }
            resolve(null);
        });
    });
    await rebootPromise.catch((err) => {
        throw (0, api_error_1.generateNotFoundApiError)({ message: err ? err.message : "reboot error" });
    });
    await dal_manager_1.dbManager.accessLog.insertSystemStatusLogForTerminalAction(organizationId, deviceId, { e: app_enums_1.enums.TerminalAction.Restart });
    res.json();
});
module.exports.softwareUpdateInterface = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let helper = new api_validatorhelper_1.ValidatorHelper();
    helper
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateEnum("state", false, app_enums_1.enums.SoftwareUpdateInterfaceState, req.swagger.params.state.value)
        .finalize();
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let state = req.swagger.params.state.value;
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    if (deviceBasic.brand != app_enums_1.enums.DeviceBrand.Armon && deviceBasic.model !== "VPass") {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    let updateInterfacePromise = new Promise((resolve, reject) => {
        messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
            e: app_enums_1.enums.ServerToDeviceRPCMessageType.SoftwareUpdateInterfaceCommand,
            p: {
                c: state === app_enums_1.enums.SoftwareUpdateInterfaceState.Enable ? app_enums_1.enums.libEnumsV2.SoftwareUpdateInterfaceCommand.Enable : app_enums_1.enums.libEnumsV2.SoftwareUpdateInterfaceCommand.Disable,
            },
        }, 60 * 1000, (err, result) => {
            if (err) {
                return reject(err);
            }
            resolve(result);
        });
    });
    const result = await updateInterfacePromise.catch((err) => {
        throw (0, api_error_1.generateNotFoundApiError)({ message: err ? err.message : "software update interface state error" });
    });
    if (result.s) {
        res.status(200).send();
        await dal_manager_1.dbManager.accessLog.insertSystemStatusLogForTerminalAction(organizationId, deviceId, {
            e: app_enums_1.enums.TerminalAction.SoftwareUpdateInterface,
            s: state,
        });
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return dal_manager_1.dbManager.accessDevice.updateDeviceSoftwareUpdateInterfaceState(organizationId, { deviceId, state: state }, trx);
        }, authResult.userId, organizationId);
    }
    else {
        app_logs_1.logger.info(`Error while ${state === app_enums_1.enums.SoftwareUpdateInterfaceState.Enable ? "Enabling" : "Disabling"} Software Update Interface`);
        app_logs_1.logger.error(result.er);
        res.status(500).send(`Error while ${state === app_enums_1.enums.SoftwareUpdateInterfaceState.Enable ? "Enabling" : "Disabling"} Software Update Interface`);
    }
});
module.exports.serverConnection = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let helper = new api_validatorhelper_1.ValidatorHelper();
    helper
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateEnum("command", false, app_enums_1.enums.ServerConnectionCommand, req.swagger.params.command.value)
        .finalize();
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let command = req.swagger.params.command.value;
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    if (deviceBasic.brand != app_enums_1.enums.DeviceBrand.Armon && deviceBasic.model !== "VPass") {
        throw (0, api_error_1.generateBadRequestApiError)({ message: "Incompatible device type" });
    }
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    let updateInterfacePromise = new Promise((resolve, reject) => {
        messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
            e: app_enums_1.enums.ServerToDeviceRPCMessageType.OpenVPNConnectCommand,
            p: {
                c: command === app_enums_1.enums.ServerConnectionCommand.Connect ? app_enums_1.enums.libEnumsV2.OpenVPNCommand.Connect : app_enums_1.enums.libEnumsV2.OpenVPNCommand.Disconnect,
            },
        }, 60 * 1000, (err, result) => {
            if (err) {
                return reject(err);
            }
            resolve(result);
        });
    });
    const result = await updateInterfacePromise.catch((err) => {
        throw (0, api_error_1.generateNotFoundApiError)({ message: err ? err.message : " server connect / disconnect error" });
    });
    if (result.s) {
        res.status(200).send();
        await dal_manager_1.dbManager.accessLog.insertSystemStatusLogForTerminalAction(organizationId, deviceId, {
            e: app_enums_1.enums.TerminalAction.ServerConnection,
            s: command,
        });
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return dal_manager_1.dbManager.accessDevice.updateServerConnectionState(organizationId, { deviceId, command }, trx);
        }, authResult.userId, organizationId);
    }
    else {
        app_logs_1.logger.info(`Error while ${command === app_enums_1.enums.ServerConnectionCommand.Connect ? "Connecting" : "Disconnecting"} Device to Server`);
        app_logs_1.logger.error(result.er);
        res.status(500).send(`Error while ${command === app_enums_1.enums.ServerConnectionCommand.Connect ? "Connecting" : "Disconnecting"} Device to Server`);
    }
});
module.exports.getTerminalActionSnapshot = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    let sid = uuid_1.default.v4();
    let snapshotPromise = new Promise((resolve, reject) => {
        messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
            e: app_enums_1.enums.ServerToDeviceRPCMessageType.TakeSnapshot,
            p: {
                sid: sid,
            },
        }, 60 * 1000, (err, success) => {
            if (err) {
                return reject(err);
            }
            resolve(null);
        });
    });
    await snapshotPromise.catch((err) => {
        throw (0, api_error_1.generateNotFoundApiError)({ message: err.message });
    });
    let snapshot = await dal_manager_1.dbManager.accessRedisCache.getValue(sid);
    res.json(snapshot);
});
module.exports.startTerminalFingerprintCapture = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.body.userId, "userId");
    let credentialIndex = (0, app_util_1.validateEnum)(req.body.finger, app_enums_1.enums.CredentialIndex);
    let credentialType = (0, app_util_1.validateEnum)(req.body.credentialType, app_enums_1.enums.CredentialType);
    let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, deviceId);
    if (!deviceBasic) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "Device not found on system" });
    }
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    let response = null;
    let fpCapturePromise = new Promise((resolve, reject) => {
        messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
            e: app_enums_1.enums.ServerToDeviceRPCMessageType.CredentialRegisterMode,
            p: {
                uid: userId,
                t: credentialType,
                i: credentialIndex,
            },
        }, 60 * 1000, (err, success) => {
            if (err) {
                return reject(err);
            }
            response = success;
            resolve(null);
        });
    });
    await fpCapturePromise.catch((err) => {
        throw (0, api_error_1.generateNotFoundApiError)({ message: err.message });
    });
    if (!response) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "unable to get response from device" });
    }
    if (response.cid) {
        await dal_manager_1.dbManager.accessLog.insertSystemStatusLogForTerminalAction(organizationId, deviceId, { e: app_enums_1.enums.TerminalAction.FingerPrintCapture, cid: response.cid });
        let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, [userId]);
        let userCaptionInfo = userCaptions.find((u) => u.id == userId);
        let credential = await dal_manager_1.dbManager.accessUser.getUserOrganizationCredential(organizationId, response.cid);
        if (!credential) {
            return res.json({
                failReason: app_enums_1.enums.CredentialCreateErrorReason.FingerPrintRegisterFailed,
                success: false,
            });
        }
        res.json({
            userCaptions: userCaptionInfo ? userCaptionInfo.captionLines : [],
            userId: userId,
            success: true,
        });
    }
    else {
        res.json({
            success: false,
            failReason: response.f,
        });
    }
});
module.exports.listTerminalAdministrators = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.tid.value, "terminalId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, terminalId);
    if (!userRights.read) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with read permission for this device", [predefined_permissions_1.Permissions.terminal.getRead()], [predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    const accessRights = new api_user_v1_models_1.RestUserApiV1.TerminalAdministratorsList(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessDevice.getTerminalAdministrators(organizationId, terminalId, accessRights.pagination);
    res.json(dbResult);
});
module.exports.upsertTerminalAdministrators = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.tid.value, "terminalId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, terminalId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
    }
    const accessRights = new api_user_v1_models_1.RestUserApiV1.UpsertAdministrator(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessDevice.upsertTerminalAdministrator(organizationId, authResult.userId, terminalId, accessRights);
    res.json(dbResult);
});
module.exports.removeTerminalAdministrator = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userTerminalRightId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.utrid.value, "userTerminalRightId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let terminalId = await dal_manager_1.dbManager.accessDevice.getTerminalIdFosUserTerminalRightId(organizationId, userTerminalRightId);
    if (terminalId) {
        let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, terminalId);
        if (!userRights.write) {
            (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this device", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getRead(), predefined_permissions_1.Permissions.terminal.getWrite()]);
        }
        await dal_manager_1.dbManager.accessDevice.removeTerminalAdministrator(organizationId, userTerminalRightId);
        res.sendStatus(200);
    }
    else {
        (0, dal_access_error_1.throwDbAccessNotFoundError)("User Terminal Right could not be found");
    }
});
module.exports.listTerminalChanges = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const { userId } = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    const { pagination } = new api_user_v1_models_1.RestUserApiV1.ListTerminalChangesRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const terminalId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.tid.value, "terminalId");
    const terminalChanges = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessDevice.listTerminalChanges({ organizationId, terminalId, requesterUserId: userId, pagination, trx });
    }, userId, organizationId);
    res.json(terminalChanges);
});
module.exports.syncTerminals = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const { userId } = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.SyncTerminalsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let terminalIds = [];
    if (request.applyToAll) {
        let terminals = await dal_manager_1.dbManager.accessDevice.listTerminals(organizationId, {
            pagination: {
                skip: 0,
                take: 1000,
            },
        }, userId);
        terminalIds = terminals.items.map((m) => m.id);
    }
    else {
        terminalIds = request.terminalIds;
    }
    for (const tid of terminalIds) {
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            await dal_manager_1.dbManager.accessDevice.syncTerminal(tid, organizationId, trx);
        }, userId, organizationId);
    }
    res.json();
});
module.exports.searchPendingTabletApprovements = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.ListCheckInTabletApprovements(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessDevice.listPendingTabletApprovements(organizationId, {
            pagination: request.pagination,
            status: request.status,
            filter: request.filter,
        }, trx);
    }, authResult.userId, organizationId);
    res.json({
        pagination: {
            total: dbResult.total,
            skip: request.pagination.skip,
            take: request.pagination.take,
        },
        items: dbResult.items.map((m) => {
            return {
                id: m.id,
                creationTime: m.creationT,
                referenceCode: m.referenceCode,
                isApproved: m.isApproved,
                name: m.name,
                publicKey: m.publicKey,
            };
        }),
    });
});
module.exports.approvePendingTablet = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.ApprovePendingTablet(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessDevice.approveTablet(organizationId, request, trx);
    }, authResult.userId, organizationId);
    if (dbResult.success) {
        res.json({});
    }
    else {
        let message = i18n_1.default.__({ phrase: dbResult.errorReason, locale: req.locale ?? "tr" });
        res.status(dbResult.errorCode).json({ message: message, noReport: true, showAsModal: true });
    }
});
