"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.canChangeStateOfAccessControlPoint = exports.canViewAccessControlPoint = void 0;
const bluebird_1 = require("bluebird");
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_httpserver_1 = require("../../../../app.httpserver");
const business_device_1 = require("../../../../business/business.device");
const business_log_1 = require("../../../../business/business.log");
const dal_memcache_1 = require("../../../../dal/access/dal.memcache");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_manager_1 = require("../../../../dal/dal.manager");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const messageBroker_notification_pub_1 = require("../../../../messageBroker/messageBroker.notification.pub");
const messageBroker_server_to_app_pub_1 = require("../../../../messageBroker/messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("../../../../messageBroker/messageBroker.server-to-app.sub");
const messageBroker_server_to_device_pub_1 = require("../../../../messageBroker/messageBroker.server-to-device.pub");
const messagebroker_models_1 = require("../../../../messageBroker/messagebroker.models");
const ws_usernsp_1 = require("../../../../ws/ws.usernsp");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const uuid = require("uuid");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const enums_1 = require("../../../../lib/access-model/v2/enums");
const dal_access_psql_log_1 = require("../../../../dal/access/psql/dal.access.psql.log");
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const business_accesslogs_excel_1 = require("../../../../business/business.accesslogs.excel");
async function canViewAccessControlPoint(organizationId, userId, acpId) {
    let userRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, userId);
    let acpRights = userRights.find((a) => a.accessControlPointId == acpId);
    if (!acpRights || !acpRights.read) {
        throw (0, api_error_1.generateConflictError)({ details: acpId, message: "Unable to see access control point" });
    }
}
exports.canViewAccessControlPoint = canViewAccessControlPoint;
async function canChangeStateOfAccessControlPoint(params) {
    let userRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(params.organizationId, params.userId);
    let acpRights = userRights.find((a) => a.accessControlPointId == params.accessControlPointId);
    if (!acpRights || !acpRights.config) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS, "ERRORS.GENERAL.ACPSTATECHANGEPERMISSIONERROR", null, true);
    }
}
exports.canChangeStateOfAccessControlPoint = canChangeStateOfAccessControlPoint;
module.exports.listSelfAccessControlPointPrivileges = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_user_v1_models_1.RestUserApiV1.ListAccessControlPointPrivilegesRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.listUserAccessRightsWithPagination(organizationId, jwt.userId, {
        take: request.take,
        skip: request.skip,
    }, request.accessControlPointIds);
    res.json({
        total: dbResult.paginationResponse.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items,
    });
});
module.exports.genericSearchAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequestWithPrivileges(req.body);
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.genericSearchAccessControlPoint(organizationId, jwt.userId, request);
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.listAccessControlPoints = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_user_v1_models_1.RestUserApiV1.ListAccessControlPointRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.listAccessControlPoints(organizationId, jwt.userId, {
        take: request.take,
        skip: request.skip,
        accessControlPointIds: request.accessControlPointIds,
        read: request.read,
        access: request.access,
        remoteAccess: request.remoteAccess,
        config: request.config,
        grant: request.grant,
        snapshot: request.snapshot,
    });
    res.json({
        total: dbResult.pagination.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.getAccessControlPointBasic = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let jwt = req.auth_token;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasicWithUserRights(organizationId, [accessControlPointId], jwt.userId, trx);
        dbResult[0].userRights?.read
            ? res.json({
                id: dbResult[0].id,
                name: dbResult[0].name,
                location: dbResult[0].location,
                remoteAvailable: dbResult[0].remoteAvailable,
                isRemoteDefault: dbResult[0].isRemoteDefault,
                type: dbResult[0].type,
                cameraInfo: dbResult[0].cameraInfo,
                userRights: dbResult[0].userRights,
                defaultPrivileges: dbResult[0].defaultPrivileges,
            })
            : res.json();
    }, jwt.userId, organizationId);
});
module.exports.getMultipleAccessControlPointsBasic = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const validator = new api_validatorhelper_1.ValidatorHelper();
    validator.validateUUID("organizationId", false, req.swagger.params.oid.value);
    validator.validateUUIDArray("accessControlPointIds", false, req.body.accessControlPointIds);
    validator.finalize();
    let organizationId = req.swagger.params.oid.value;
    let accessControlPointIds = req.body.accessControlPointIds;
    let jwt = req.auth_token;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasicWithUserRights(organizationId, accessControlPointIds, jwt.userId, trx);
        res.json({
            items: dbResult
                .filter((f) => f.userRights?.read)
                .map((r) => {
                return {
                    id: r.id,
                    name: r.name,
                    location: r.location,
                    remoteAvailable: r.remoteAvailable,
                    isRemoteDefault: r.isRemoteDefault,
                    type: r.type,
                    cameraInfo: r.cameraInfo,
                    userRights: r.userRights,
                    defaultPrivileges: r.defaultPrivileges,
                };
            }),
        });
    }, jwt.userId, organizationId);
});
module.exports.searchAccessControlPointBasic = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = new api_user_v1_models_1.RestUserApiV1.AccessControlPointFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let jwt = req.auth_token;
    let hasSystemRead = jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead());
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.searchAccessControlPoint(organizationId, filter, hasSystemRead, jwt.userId);
    res.json(dbResult);
});
module.exports.searchRemoteAccessGrantedAccessControlPoints = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = req.body;
    let jwt = (0, api_securityhelper_1.authorizeForSearchRemoteAccessGrantedAccessControlPoints)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.searchRemoteAccessGrantedAccessControlPoints(organizationId, jwt.userId, filter);
    res.json(dbResult);
});
module.exports.searchAvailableAccessControlPointsByAuth = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let filter = req.body;
    let jwt = (0, api_securityhelper_1.authorizeForSearchRemoteAccessGrantedAccessControlPoints)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessAccessControlPoint.searchAccessGrantedAccessControlPointsByAuth({
            organizationId: organizationId,
            userId: jwt.userId,
            filter: filter,
            trx: trx,
        });
    }, jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.obtainUserSelectionSessionForAccessControlPointGrant = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let force = req.swagger.params.force.value;
    let authResult = (0, api_securityhelper_1.authorizeForObtainUserSelectionSessionForAccessControlPointGrant)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.obtainUserSelectionSession(organizationId, {
            requestUserId: authResult.jwt.userId,
            type: app_enums_1.enums.UserSelectionSessionType.AccessControlPointGrant,
            relatedItemId: accessControlPointId,
            forceToCreate: force,
            isAuthorizedForAllUsers: authResult.hasOrganizationWideIdentityEdit,
            authorizedOrganizationUnitIds: authResult.readBasicOrganizationUnitIds ?? null,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    if (!dbResult.upserted) {
        throw (0, api_error_1.generateConflictError)({
            message: "There is already a session for that access control point. Please see detailes.",
            details: dbResult.overwritten,
        });
    }
    let result = {
        sessionId: dbResult.sessionId,
        expirationUtc: dbResult.expirationUtc,
        justRefreshed: dbResult.justRefreshed,
        overwritten: dbResult.overwritten
            ? {
                ownerUserId: dbResult.overwritten.ownerUserId,
                sessionId: dbResult.overwritten.sessionId,
                expirationUtc: dbResult.overwritten.expirationUtc,
                captionLines: dbResult.overwritten.captionLines,
            }
            : undefined,
    };
    res.json(result);
});
module.exports.getDetailedAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let jwt = req.auth_token;
    await canViewAccessControlPoint(organizationId, jwt.userId, accessControlPointId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailes(organizationId, accessControlPointId));
});
module.exports.getDetailedAccessControlPointV2 = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let jwt = req.auth_token;
    await canViewAccessControlPoint(organizationId, jwt.userId, accessControlPointId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV2(organizationId, accessControlPointId));
});
module.exports.getDetailedAccessControlPointV3 = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let accessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "accessControlPointId");
    let jwt = req.auth_token;
    await canViewAccessControlPoint(organizationId, jwt.userId, accessControlPointId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV3(organizationId, jwt.userId, accessControlPointId));
});
module.exports.getDetailedAccessControlPointByDeviceId = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let deviceId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.did.value, "deviceId");
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesByDeviceId(organizationId, deviceId));
});
module.exports.getCounterLogReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAccessControlPointSummaryReport)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.CounterSensorReportFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessLog.getCounterSensorReport(organizationId, request);
    res.json(dbResult);
});
module.exports.insertManualLog = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const request = new api_user_v1_models_1.RestUserApiV1.InsertManualLogRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    const organizationId = req.swagger.params.oid.value;
    const authResult = (0, api_securityhelper_1.authorizeForManualLogInsert)(req, organizationId);
    let dbResult;
    const accessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, request.accessControlPointId);
    if (!accessControlPoint) {
        return Promise.reject("access control point not found");
    }
    if (accessControlPoint.type === app_enums_1.enums.AccessControlPointType.VirtualMobileCheckin) {
        dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return (0, business_log_1.insertMobileManualLog)({ organizationId, requesterUserId: authResult.jwt.userId, accessControlPoint, logRequest: request, isQrTriggered: false, trx });
        }, authResult.jwt.userId, organizationId);
    }
    else {
        dbResult = await (0, business_log_1.insertManualLog)(organizationId, authResult.jwt.userId, accessControlPoint, request);
    }
    await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, {
        credentialOwnerUserId: request.userId,
        generationTime: request.actionUtc.toISOString(),
        redisCache: dal_manager_1.dbManager.accessRedisCache,
        logId: dbResult,
    });
    res.json(dbResult);
});
module.exports.generateExcelForAccessLogImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForManualLogInsert)(req, organizationId);
    let dateTimeCombined = req.swagger.params.combined.value;
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    let accessLogsExcel = new business_accesslogs_excel_1.AccessLogsImportUploadExcel(organizationId, authResult.jwt.userId, locale);
    let accessLogsExcelResponse = {};
    const withTimeout = (promise, timeoutMs, errorMessage) => {
        return Promise.race([promise, new Promise((_, reject) => setTimeout(() => reject(new Error(errorMessage)), timeoutMs))]);
    };
    try {
        accessLogsExcelResponse = await withTimeout(accessLogsExcel.generateExcel(locale, dateTimeCombined), 10 * 1000, i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL-EXCEL.TIMEOUT.WRITE", locale }));
    }
    catch (error) {
        accessLogsExcelResponse.errorMessage = error.message;
        return res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json(accessLogsExcelResponse);
    }
    if (!accessLogsExcelResponse.errorMessage) {
        if (fs_1.default.existsSync(accessLogsExcelResponse.filePath)) {
            res.download(accessLogsExcelResponse.filePath, accessLogsExcelResponse.filename, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(accessLogsExcelResponse.filePath);
            });
        }
        else {
            res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json();
        }
    }
});
module.exports.uploadExcelForAccessLogImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let defaultAccessControlPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.acpid.value, "accessControlPointId");
    let isPacsCalcEnabled = req.swagger.params.pacs.value;
    const authResult = (0, api_securityhelper_1.authorizeForManualLogInsert)(req, organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({
        organizationId,
        userId: authResult.jwt.userId,
    });
    let accessLogsExcel = new business_accesslogs_excel_1.AccessLogsImportUploadExcel(organizationId, authResult.jwt.userId, locale);
    let accessLogsExcelResponse = {};
    const withTimeout = (promise, timeoutMs, errorMessage) => {
        return Promise.race([promise, new Promise((_, reject) => setTimeout(() => reject(new Error(errorMessage)), timeoutMs))]);
    };
    try {
        await withTimeout(accessLogsExcel.loadExcel(req.swagger.params.file.originalValue.buffer), 10 * 1000, i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.LOAD", locale }));
    }
    catch (error) {
        accessLogsExcelResponse.errorMessage = error.message;
        return res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json(accessLogsExcelResponse);
    }
    let validation;
    try {
        validation = await withTimeout(accessLogsExcel.read(), 10 * 1000, i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.READ", locale }));
    }
    catch (error) {
        accessLogsExcelResponse.errorMessage = error.message;
        return res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json(accessLogsExcelResponse);
    }
    if (validation.validationSuccess) {
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            accessLogsExcel.uploadExcel(trx, defaultAccessControlPointId, authResult.jwt.userId, isPacsCalcEnabled);
        });
        return res.json({});
    }
    else {
        return res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
            message: validation.errorMessage,
            errorCells: validation.errorCells,
            errorUsers: validation.errorUsers,
            errorDates: validation.errorDates,
            showAsModal: true,
            noReport: true,
        });
    }
});
module.exports.insertMultipleManualLogs = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = new api_user_v1_models_1.RestUserApiV1.InsertMultipleManualLogRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    let organizationId = req.swagger.params.oid.value;
    let authResult = (0, api_securityhelper_1.authorizeForManualLogInsert)(req, organizationId);
    let dbResult = await (0, business_log_1.insertMultipleManualLogs)(organizationId, authResult.jwt.userId, request);
    res.json(dbResult);
});
module.exports.deleteManualLog = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let logId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "logId");
    let authResult = (0, api_securityhelper_1.authorizeForManualLogDelete)(req, organizationId);
    await (0, business_log_1.deleteManualLog)(organizationId, logId);
    res.json();
});
module.exports.postRemoteStateCommand = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const request = new api_user_v1_models_1.RestUserApiV1.AccessControlPointRemoteOrderState(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    const organizationId = req.swagger.params.oid.value;
    const userId = req.auth_token?.userId;
    await canChangeStateOfAccessControlPoint({ organizationId, userId, accessControlPointId: request.accessControlPointId });
    const accessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, request.accessControlPointId);
    if (accessControlPoint.deviceId) {
        const acpStateChangePromise = new Promise((resolve, reject) => {
            messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(accessControlPoint.deviceId, {
                e: messagebroker_models_1.ServerToDeviceRPCMessageType.ChangeAccessControlPointState,
                p: {
                    a: request.accessControlPointId,
                    o: userId,
                    sl: request.stateList.map((m) => {
                        return {
                            s: m.state,
                            d: m.direction,
                        };
                    }),
                },
            }, 10000, (err, success) => {
                if (err) {
                    return reject(err);
                }
                resolve(null);
            });
        });
        try {
            await acpStateChangePromise;
        }
        catch (error) {
            res.status(404);
            res.json((0, api_error_1.generateTimeoutApiError)(new bluebird_1.TimeoutError(error.message)));
        }
    }
    res.json();
});
module.exports.addAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const validator = new api_validatorhelper_1.ValidatorHelper();
    let request = new api_user_v1_models_1.RestUserApiV1.AddAccessPointRequest(req.body, validator.validateUUID("organizationId", false, req.swagger.params.oid.value));
    validator.finalize();
    const organizationId = req.swagger.params.oid.value;
    const jwt = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    const accessPointId = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessAccessControlPoint.addAccessControlPoint({ organizationId, requestUserId: jwt.userId, request, trx });
    }, jwt.userId, organizationId);
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV3(organizationId, jwt.userId, accessPointId));
});
module.exports.updateAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const validator = new api_validatorhelper_1.ValidatorHelper();
    let request = new api_user_v1_models_1.RestUserApiV1.UpdateAccessPointRequest(req.body, validator.validateUUID("organizationId", false, req.swagger.params.oid.value));
    if (!request.id) {
        throw new Error("Access Control Point ID is not provided");
    }
    let organizationId = req.swagger.params.oid.value;
    let jwt = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    let accessControlPointBasic = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, request.id);
    if (!accessControlPointBasic && !accessControlPointBasic.remoteAvailable && request.defaultPrivileges && request.defaultPrivileges.remoteAccess) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.GENERAL.REMOTEACCESSERROR");
    }
    if ((accessControlPointBasic.id === dal_constants_1.DalConstants.defaultMobileAcpId || accessControlPointBasic.id === dal_constants_1.DalConstants.defaultArventoDriverAcpId) &&
        (accessControlPointBasic.location != request.location ||
            accessControlPointBasic.type != request.accessControlPointType ||
            accessControlPointBasic.isRemoteDefault != request.isRemoteDefault ||
            accessControlPointBasic.remoteAvailable != request.remoteAvailable)) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.GENERAL.DEFAULTMOBILEACPEDITERROR");
    }
    const accessPointId = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessAccessControlPoint.updateAccessControlPoint({
            organizationId,
            requestUserId: jwt.userId,
            request,
            trx,
        });
    }, jwt.userId, organizationId);
    const result = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV3(organizationId, jwt.userId, accessPointId);
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId);
    res.json(result);
});
module.exports.deleteAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let ids = new api_validatorhelper_1.ValidatorHelper().validateUUIDsAndFinalize([
        { field: "organizationId", optional: false, input: req.swagger.params.oid.value },
        { field: "accessControlPointId", optional: false, input: req.swagger.params.aid.value },
    ]);
    let accessControlPointId = req.swagger.params.aid.value;
    let organizationId = req.swagger.params.oid.value;
    let jwt = (0, api_securityhelper_1.authorizeForWriteSystem)(req, ids.organizationId);
    let logResult = await dal_manager_1.dbManager.accessLog.getAccessLogsReport(organizationId, {
        pagination: { take: app_config_1.appConfig.reportExportRowLimit, skip: undefined },
        accessControlPointIds: [accessControlPointId],
        sortDateDesc: false,
        startUtc: (0, moment_1.default)(new Date()).add(-2, "year").toDate(),
        endUtc: new Date(),
    }, false);
    if (logResult.totalCount > 0) {
        throw (0, api_error_1.generateConflictError)({ message: "Log exists for access control point", details: "" });
    }
    await dal_manager_1.dbManager.accessAccessControlPoint.deleteAccessControlPoint(ids.organizationId, ids.accessControlPointId);
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(ids.organizationId);
    res.json({});
});
module.exports.updateAccessControlPointMapping = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.updateAccessControlPointMappingRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("organizationId", false, req.swagger.params.oid.value).validateUUID("accessControlPointId", false, req.swagger.params.aid.value));
    let organizationId = req.swagger.params.oid.value;
    let accessControlPointId = req.swagger.params.aid.value;
    const auth = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    let accessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.updateAccessControlPointMapping(organizationId, accessControlPointId, request, auth.userId);
    res.json(accessControlPoint);
});
module.exports.updateRelay = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.updateRelayRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("relayId", false, req.swagger.params.rid.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let relayId = req.swagger.params.rid.value;
    let authResult = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this relay", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getWrite(), predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    await dal_manager_1.dbManager.accessAccessControlPoint.updateRelay(organizationId, deviceId, relayId, request, authResult.userId);
    res.json({});
});
module.exports.updateStatusSensor = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.updateStatusSensorRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("statusSensorId", false, req.swagger.params.sid.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let statusSensorId = req.swagger.params.sid.value;
    let auth = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    await dal_manager_1.dbManager.accessAccessControlPoint.updateStatusSensor(organizationId, deviceId, statusSensorId, request, auth.userId);
    res.json({});
});
module.exports.updateCounterSensor = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.updateCounterSensorRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("counterSensorId", false, req.swagger.params.ccid.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let counterSensorId = req.swagger.params.ccid.value;
    const auth = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    await dal_manager_1.dbManager.accessAccessControlPoint.updateCounterSensor(organizationId, deviceId, counterSensorId, request, auth.userId);
    res.json({});
});
module.exports.updateWiegandReader = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.updateWiegandReaderRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("wiegandReaderId", false, req.swagger.params.wid.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let wiegandReaderId = req.swagger.params.wid.value;
    let authResult = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    let userRights = await dal_manager_1.dbManager.accessDevice.getUserTerminalRights(organizationId, authResult.userId, deviceId);
    if (!userRights.write) {
        (0, api_securityhelper_1.throwAuthError)("User not authorized with write permission for this reader", [predefined_permissions_1.Permissions.terminal.getWrite()], [predefined_permissions_1.Permissions.terminal.getWrite(), predefined_permissions_1.Permissions.terminal.getRead()]);
    }
    await dal_manager_1.dbManager.accessAccessControlPoint.updateWiegandReader(organizationId, deviceId, wiegandReaderId, request, authResult.userId);
    res.json({});
});
module.exports.updateHikVisionLicensePlateCameraLaneInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.UpdateHikVisionLicensePlateCameraLaneRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("hikVisionId", false, req.swagger.params.id.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let hikVisionId = req.swagger.params.id.value;
    (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    await dal_manager_1.dbManager.accessAccessControlPoint.updateHikVisionLicensePlateCameraLaneInfo(organizationId, deviceId, hikVisionId, request);
    res.json({});
});
module.exports.updateImpinjSpeedWayGateWayAntennaInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = api_user_v1_models_1.RestUserApiV1.UpdateImpinjSpeedWayGateWayAntennaRequest(req.body, new api_validatorhelper_1.ValidatorHelper()
        .validateUUID("organizationId", false, req.swagger.params.oid.value)
        .validateUUID("deviceId", false, req.swagger.params.did.value)
        .validateUUID("impinjSpeedWayId", false, req.swagger.params.id.value));
    let organizationId = req.swagger.params.oid.value;
    let deviceId = req.swagger.params.did.value;
    let impinjSpeedWayId = req.swagger.params.id.value;
    (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    await dal_manager_1.dbManager.accessAccessControlPoint.updateImpinjSpeedWayGateWayAntennaInfo(organizationId, deviceId, impinjSpeedWayId, request);
    res.json({});
});
module.exports.getAccessGrantedIdentitiesForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    new api_validatorhelper_1.ValidatorHelper().validateUUID("organizationId", false, req.swagger.params.oid.value).validateUUID("accessControlPointId", false, req.swagger.params.aid.value).finalize();
    let organizationId = req.swagger.params.oid.value;
    let accessControlPointId = req.swagger.params.aid.value;
    let jwt = req.auth_token;
    await canViewAccessControlPoint(organizationId, jwt.userId, accessControlPointId);
    res.json(await dal_manager_1.dbManager.accessAccessControlPoint.getAccessGrantedIdentitiesForAccessControlPoint(organizationId, accessControlPointId));
});
module.exports.listPrivilegedUsersForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    new api_validatorhelper_1.ValidatorHelper().validateUUID("organizationId", false, req.swagger.params.oid.value).validateUUID("accessControlPointId", false, req.swagger.params.aid.value).finalize();
    let organizationId = req.swagger.params.oid.value;
    let accessControlPointId = req.swagger.params.aid.value;
    let jwt = req.auth_token;
    await canViewAccessControlPoint(organizationId, jwt.userId, accessControlPointId);
    let request = new api_user_v1_models_1.RestUserApiV1.ListPrivilegedUsersRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessAccessControlPoint.listPrivilegedUsersForAccessControlPoint(organizationId, accessControlPointId, request);
    res.json(dbResult);
});
module.exports.mobileAccessReadCard = (0, api_util_1.globalRouteHandler)(async (req, res) => {
    let request = new api_user_v1_models_1.RestUserApiV1.MobileAccess(req.body, new api_validatorhelper_1.ValidatorHelper());
    let organizationId = req.swagger.params.oid.value;
    let authResult = (0, api_securityhelper_1.authorizeForManualLogInsert)(req, organizationId);
    let accessControlPointId = req.swagger.params.aid.value;
    if (!request.credentialData && !request.userId) {
    }
    let credentialId = null;
    let logItem = null;
    let insertedLog = null;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        if (!request.credentialData) {
            const credentials = await dal_manager_1.dbManager.accessIdentity.findTransformedMifareCredentialDatasOfUser(organizationId, {
                userId: request.userId,
                requesterUserId: authResult.jwt.userId,
            }, trx);
            const chosenCredential = credentials.find((c) => c.type === app_enums_1.enums.CredentialType.MiFare);
            if (chosenCredential) {
                request.credentialData = chosenCredential.data;
                credentialId = chosenCredential.id;
            }
        }
        logItem = await (0, business_device_1.updateLogFromMobileDevice)(organizationId, {
            id: uuid.v4(),
            v: request.isVisitor || false,
            o: dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId,
            a: accessControlPointId,
            u: new Date().toISOString(),
            c: request.credentialData,
            cx: [
                {
                    t: app_enums_1.enums.CredentialType.MiFare,
                    d: request.credentialData,
                    i: credentialId,
                },
            ],
            d: request.direction,
            oId: organizationId,
            r: null,
            s: null,
            ul: null,
            rm: app_enums_1.enums.LogReceiveMethod.Mobile,
        });
        insertedLog = await dal_manager_1.dbManager.accessLog.addAccessLogFromDevice(organizationId, logItem, trx);
        let now = new Date();
        if (insertedLog.rg && insertedLog.rg.length > 0) {
            for (const regionLog of insertedLog.rg) {
                let regionNotification = {
                    i: regionLog.i,
                    s: regionLog.s.valueOf(),
                    ee: regionLog.ee,
                    xe: regionLog.xe,
                    o: insertedLog.o,
                    t: insertedLog.u,
                };
                (0, cli_queries_1.upsertRegionStates)(organizationId, [{ accessControlPointId: insertedLog.a, log: regionNotification }], trx);
                if (regionLog.ru && regionLog.ru.c) {
                    (0, cli_queries_1.upsertAccessRuleHistory)(organizationId, {
                        date: now,
                        accessRuleId: regionLog.ru.i,
                        userId: insertedLog.o,
                        count: regionLog.ru.c,
                        accessControlPointId: insertedLog.a,
                        actionDateISO: insertedLog.u,
                    }, trx);
                }
                if (regionLog.rti) {
                    (0, cli_queries_1.updateUserRegionTicketUnits)(organizationId, { userId: insertedLog.o, accessControlPointId: insertedLog.a, decrementUnit: regionLog.rti.c, regionTicketId: regionLog.rti.i }, trx);
                }
            }
        }
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.activity + "." + (logItem.a ? logItem.a : "*") + "." + (logItem.d ? logItem.d : "*") + "." + (logItem.s ? logItem.s : "*"), {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.newlog,
            p: insertedLog,
            oId: organizationId,
        });
        if (logItem.v && logItem.rg && logItem.rg.length > 0) {
            setTimeout(() => {
            }, 100, this);
        }
        await dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.NewAccessLog,
            organizationId: organizationId,
            userId: logItem.o,
            acpId: logItem.a,
            timestamp: new Date(logItem.u),
            direction: logItem.d,
        });
    }, authResult.jwt.userId, organizationId);
    if (insertedLog.d && insertedLog.d === enums_1.EnumsV2.AccessDirection.Entrance && insertedLog.s === true) {
        await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, { credentialOwnerUserId: insertedLog.o, generationTime: insertedLog.u, redisCache: dal_manager_1.dbManager.accessRedisCache, logId: insertedLog.id });
    }
    res.json({
        userId: logItem?.o,
        fullname: logItem?.on,
        captionLines: logItem?.o ? await dal_manager_1.dbManager.accessUser.getSingleUserOrganizationCaptionLines(organizationId, logItem?.o) : undefined,
        reason: logItem?.r,
        success: logItem?.s,
    });
});
module.exports.addQRCodeForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = req.swagger.params.oid.value;
    const request = req.body;
    if (!request.accessPointId && !request.qrCodeData && !request.config?.direction && !request.config?.trigger?.type) {
        throw new Error("Bad Request");
    }
    const { userId } = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    const { qrCodeId, eventId, instanceData } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const qrCodeId = await dal_manager_1.dbManager.accessAccessControlPoint.addQRCodeForAccessPoint({ organizationId, userId, request, trx });
        const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(organizationId, app_enums_1.enums.NotificationType.AccessPointQrCodeUpdatedToUser, trx);
        const mobileUserIds = await dal_manager_1.dbManager.accessUser.getMobileUserIds({ organizationId, trx });
        const instanceData = {
            a: request.accessPointId,
            receiverUserIds: mobileUserIds,
        };
        const eventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
            createdT: new Date(),
            notificationId,
            organizationId,
            trx,
            instanceData,
        });
        return { qrCodeId, eventId, instanceData };
    }, userId, organizationId);
    (0, messageBroker_notification_pub_1.publishToNotificationService)({
        a: 0,
        i: eventId,
        n: app_enums_1.enums.AmqpMessageCode.Empty,
        o: organizationId,
        v: "1",
        d: instanceData,
    });
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId);
    res.json(qrCodeId);
});
module.exports.updateQRCodeForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const qrCodeId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.qid.value, "qrCodeId");
    const request = req.body;
    if (!request?.config && !request.config.direction && !request.config.trigger?.type) {
        throw new Error("Bad Request");
    }
    const jwt = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    const { eventId, instanceData } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const accessPointId = await dal_manager_1.dbManager.accessAccessControlPoint.updateQRCodeForAccessPoint({ id: qrCodeId, organizationId, request, trx });
        const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(organizationId, app_enums_1.enums.NotificationType.AccessPointQrCodeUpdatedToUser, trx);
        const mobileUserIds = await dal_manager_1.dbManager.accessUser.getMobileUserIds({ organizationId, trx });
        const instanceData = {
            a: accessPointId,
            receiverUserIds: mobileUserIds,
        };
        const eventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
            createdT: new Date(),
            notificationId,
            organizationId,
            trx,
            instanceData,
        });
        return { eventId, instanceData };
    }, jwt.userId, organizationId);
    (0, messageBroker_notification_pub_1.publishToNotificationService)({
        a: 0,
        i: eventId,
        n: app_enums_1.enums.AmqpMessageCode.Empty,
        o: organizationId,
        v: "1",
        d: instanceData,
    });
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId);
    res.json({
        id: qrCodeId,
    });
});
module.exports.listQRCodesForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const accessPointId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.qid.value, "accessControlPointId");
    const request = req.body;
    const jwt = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessAccessControlPoint.listQRCodesForAccessPoint({ id: accessPointId, organizationId, request, trx });
    }, jwt.userId, organizationId);
    res.json(result);
});
module.exports.getQRCodeForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const qrCodeId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.qid.value, "qrCodeId");
    const jwt = (0, api_securityhelper_1.authorizeForSystemRead)(req, organizationId);
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessAccessControlPoint.getQRCodeForAccessPoint({ id: qrCodeId, organizationId, trx });
    }, jwt.userId, organizationId);
    res.json(result);
});
module.exports.deleteQRCodeForAccessControlPoint = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const qrCodeId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.qid.value, "qrCodeId");
    const jwt = (0, api_securityhelper_1.authorizeForWriteSystem)(req, organizationId);
    const { eventId, instanceData } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const accessPointId = await dal_manager_1.dbManager.accessAccessControlPoint.deleteQRCodeForAccessPoint({ id: qrCodeId, organizationId, trx });
        const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(organizationId, app_enums_1.enums.NotificationType.AccessPointQrCodeUpdatedToUser, trx);
        const mobileUserIds = await dal_manager_1.dbManager.accessUser.getMobileUserIds({ organizationId, trx });
        const instanceData = {
            a: accessPointId,
            receiverUserIds: mobileUserIds,
        };
        const eventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
            createdT: new Date(),
            notificationId,
            organizationId,
            trx,
            instanceData,
        });
        return { eventId, instanceData };
    }, jwt.userId, organizationId);
    (0, messageBroker_notification_pub_1.publishToNotificationService)({
        a: 0,
        i: eventId,
        n: app_enums_1.enums.AmqpMessageCode.Empty,
        o: organizationId,
        v: "1",
        d: instanceData,
    });
    app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId);
    res.sendStatus(200);
});
