"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const business_hooks_1 = require("../../../../business/business.hooks");
const business_summary_1 = require("../../../../business/business.summary");
const report_util_1 = require("../../../../business/report/report.util");
const dal_manager_1 = require("../../../../dal/dal.manager");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../../../user/v1/api.user.v1.models");
const api_social_v1_model_1 = require("../api.social.v1.model");
const uuid = require("uuid");
module.exports.upsertRegionTicket = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.RegionTicketUpsertRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessSocial.upsertRegionTicket(organizationId, authResult.jwt.userId, {
        id: request.id,
        regionId: request.regionId,
        name: request.name,
        organizationId: organizationId,
        requiredUnits: request.requiredUnits,
    });
    res.json(dbResult);
});
module.exports.listRegionTickets = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    const request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const listRegionTickets = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessSocial.listRegionTickets({
            organizationId,
            pagination: {
                take: request.take,
                skip: request.skip,
            },
            requesterUserId: authResult.jwt.userId,
            regionIds: null,
            regionTicketIds: request.regionTicketIds,
            trx,
        });
    }, authResult.jwt.userId, organizationId);
    res.json({
        total: listRegionTickets.total,
        take: listRegionTickets.take,
        skip: listRegionTickets.skip,
        items: listRegionTickets.items.map((i) => {
            return {
                id: i.id,
                captionLines: [i.name, i.regionName],
            };
        }),
    });
});
module.exports.listRegionTicketDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    const request = new api_social_v1_model_1.RestSocialApiV1.RegionTicketListDetailRequestFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessSocial.listRegionTickets({
            organizationId,
            pagination: request.pagination,
            regionIds: request.regionIds,
            regionTicketIds: request.regionTicketIds,
            requesterUserId: authResult.jwt.userId,
            trx,
        });
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.getRegionTicket = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionTicketId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "id");
    let authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessSocial.getRegionTicket(organizationId, regionTicketId);
    res.json(dbResult);
});
module.exports.deleteRegionTicket = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let regionTicketId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "id");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    await dal_manager_1.dbManager.accessSocial.deleteRegionTicket(organizationId, regionTicketId);
    res.json();
});
module.exports.listRegionTicketUserUnits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketUserUnitsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessSocial.listUserRegionTicketUnits(organizationId, request.pagination, {
        userIds: request.userIds,
        userGroupIds: request.userGroupIds,
        organizationUnitIds: request.organizationUnitIds,
    });
    res.json(dbResult);
});
module.exports.listRegionTicketUserUnitsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketUserUnitsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.RegionTicketUserUnits,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report RegionTicketUserUnitsExcel is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.assignRegionTicketUserUnits = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.AssignUserRegionTicketUnitRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = [];
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        let beforeAssingRegionTicketUnitsHook = business_hooks_1.armonHookManager.getBeforeAssingUserRegionTicketUnitsHook(organizationId);
        const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
        if (beforeAssingRegionTicketUnitsHook) {
            let hookErrorResult = await beforeAssingRegionTicketUnitsHook(organizationId, authResult.jwt.userId, sessionId, request.amount);
            if (hookErrorResult.length > 0) {
                let userCaptionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, hookErrorResult.map((r) => r.userId));
                for (const hookError of hookErrorResult) {
                    result.push({
                        userId: hookError.userId,
                        captionLines: userCaptionLines.find((ucl) => ucl.id === hookError.userId).captionLines,
                        message: i18n_1.default.__({ phrase: "ERRORS.SOCIAL.MAX_COIN_BALANCE_LIMIT_EXCEEDED", locale }, { maxValue: hookError.maxValue.toString() }),
                    });
                }
            }
        }
        await dal_manager_1.dbManager.accessSocial.assignRegionTicketUserUnits(organizationId, {
            userSelectionSessionId: sessionId,
            amount: request.amount,
            note: request.note,
            requestUserId: authResult.jwt.userId,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    res.json({ failedUsers: result });
});
module.exports.listRegionTicketTransactions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketTransactionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.accessLog.listRegionTicketTransactions(organizationId, request.pagination, request.dateRange, request.actionUserId, request.targetUserId);
    res.json(result);
});
module.exports.listRegionTicketTransactionsExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketTransactionFilterExcel(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.RegionTicketTransactions,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report listRegionTicketTransactionsExcel is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listRegionTicketTransactionDetails = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let transactionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.tid.value, "tid");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketTransactionDetailFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.accessLog.listRegionTicketTransactionDetails(organizationId, transactionId, request.pagination);
    res.json(result);
});
module.exports.listUserTicketUsageReport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketUsageReportFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.accessSocial.listRegionTicketUsageReport(organizationId, request.pagination, {
        dateRange: request.dateRange,
        userIds: request.userIds,
        userGroupIds: request.userGroupIds,
        organizationUnitIds: request.organizationUnitIds,
        regionTicketIds: request.regionTicketIds,
    });
    res.json(result);
});
module.exports.listUserTicketUsageReportExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialWrite)(req, organizationId);
    let request = new api_social_v1_model_1.RestSocialApiV1.ListRegionTicketUsageReportFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.UserTicketUsage,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report UserTickerUsageExcel is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.obtainUserSelectionSessionForRegionTicket = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let force = req.swagger.params.force.value;
    let authResult = (0, api_securityhelper_1.authorizeForObtainUserSelectionSessionForGeneric)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.obtainUserSelectionSession(organizationId, {
            requestUserId: authResult.jwt.userId,
            type: app_enums_1.enums.UserSelectionSessionType.RegionTicketTransaction,
            relatedItemId: uuid.v4(),
            forceToCreate: force,
            isAuthorizedForAllUsers: authResult.hasOrganizationWideWritePPermission && authResult.hasOrganizationWideReadBasic,
            authorizedOrganizationUnitIds: authResult.readBasicOrganizationUnitIds ?? null,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    if (!dbResult.upserted) {
        throw (0, api_error_1.generateConflictError)({
            message: "There is already a session for that work plan. Please see detailes.",
            details: dbResult.overwritten,
        });
    }
    let result = {
        sessionId: dbResult.sessionId,
        expirationUtc: dbResult.expirationUtc,
        justRefreshed: dbResult.justRefreshed,
        overwritten: dbResult.overwritten
            ? {
                ownerUserId: dbResult.overwritten.ownerUserId,
                sessionId: dbResult.overwritten.sessionId,
                expirationUtc: dbResult.overwritten.expirationUtc,
                captionLines: dbResult.overwritten.captionLines,
            }
            : undefined,
    };
    res.json(result);
});
module.exports.genericSearchRegionTicket = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSocialRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessSocial.genericSearchRegionTicket(organizationId, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
    });
    res.json({
        total: dbResult.total,
        take: request.take,
        skip: request.skip,
        items: dbResult.items.map((t) => {
            return {
                id: t.id,
                captionLines: t.captionLines,
                matchItem: t.matchItem,
            };
        }),
    });
});
module.exports.getSelfSocialDashboardSummary = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let userSummary = new business_summary_1.UserInfoSummary(organizationId, jwt.userId);
    let socialInfo = await userSummary.getUserSocialDetails();
    res.json(socialInfo);
});
