"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const app_enums_1 = require("../../../../app.enums");
const dal_manager_1 = require("../../../../dal/dal.manager");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_pacs_v1_models_1 = require("../api.pacs.v1.models");
const api_models_common_1 = require("../../../api.models.common");
const luxon_1 = require("luxon");
const business_pacs_1 = require("../../../../business/pacs/business.pacs");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
module.exports.applyUserSelectionSessionForWorkPlan = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let sessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "sessionId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.ApplyUserSelectionWorkPlanRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.accessPacs2.applyUserSelectionSessionForWorkPlanWithMultipleRanges({
        organizationId: organizationId,
        sessionId: sessionId,
        userId: authResult.jwt.userId,
        enforce: request.enforce,
        ranges: request.ranges,
    });
    res.json({});
});
module.exports.getEmployeeCountAtDate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let workPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workPlanId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    let requestedDateTime = new api_pacs_v1_models_1.RestPacsApiV1.DateTime(req.body, new api_validatorhelper_1.ValidatorHelper()).dateTime;
    let count = await dal_manager_1.dbManager.accessPacs.getWorkPlanAssignmentCount(organizationId, workPlanId, requestedDateTime);
    res.json(count);
});
module.exports.getEmployeeListAtDate = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let workPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workPlanId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    let requestedDateTime = new api_pacs_v1_models_1.RestPacsApiV1.DateTime(req.body, new api_validatorhelper_1.ValidatorHelper()).dateTime;
    let pagination = new api_models_common_1.ApiModelsCommon.PaginationRequest(req.body.pagination, 1000, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.accessPacs2.workplanEmployeeListAtDate(organizationId, workPlanId, new Date(requestedDateTime), {
        take: pagination.take,
        skip: pagination.skip,
    }, req.body.filter);
    res.json(result);
});
module.exports.upsertWorkPlan = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertWorkPlanRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.permissionRequiredForExtraWorking =
        request.type === app_enums_1.enums.WorkPlanType.FULL_FLEXIBLE || request.type === app_enums_1.enums.WorkPlanType.HALF_FLEXIBLE ? false : request.permissionRequiredForExtraWorking;
    let dbResult = await dal_manager_1.dbManager.accessPacs2.upsertWorkPlan(organizationId, authResult.jwt.userId, authResult.hasOrganizationWide, request);
    if (!dbResult) {
        next((0, api_error_1.generateBadRequestApiError)({ message: "invalid workplan request" }));
        return;
    }
    res.json(dbResult);
});
module.exports.removeWorkPlan = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let workPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workPlanId");
    let enforce = req.swagger.params.enforce.value;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const userFilters = (await trx.query(`
						SELECT uf.*
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" uf
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}" as ufwp
							ON uf.id = ufwp."filterId"
						WHERE ufwp."workPlanId" = $1
						`, [workPlanId])).rows;
        if (userFilters.length > 0) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.USER_FILTER.USER_FILTER_USAGE_WORK_PLAN", { userFilterNames: userFilters.map((elem) => elem.name).join(", ") }, true, true);
        }
        await dal_manager_1.dbManager.accessPacs2.removeWorkPlan(organizationId, workPlanId, enforce, trx);
    }, authResult.jwt.userId, organizationId);
    res.json();
});
module.exports.getWorkPlan = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    let workPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workPlanId");
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessPacs.getWorkPlan(trx, organizationId, authResult.jwt.userId, workPlanId);
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.listWorkPlans = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    const workPlanList = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return await dal_manager_1.dbManager.accessPacs.listWorkPlans(trx, organizationId, authResult.jwt.userId, req.body.type, req.body.name);
    }, authResult.jwt.userId, organizationId);
    res.json(workPlanList);
});
module.exports.listWorkPlanByIds = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(organizationId, req.body.workPlanIds);
    res.json({
        items: dbResult.items.map((w) => {
            return {
                assignedUserCount: w.assignedUserCount,
                id: w.id,
                name: w.name,
                type: w.type,
                regionId: w.regionId,
                regionName: w.regionName,
                accessCheckType: w.accessCheckType,
                offset: w.offset,
                allowMobileCheckins: w.allowMobileCheckins,
                mobileCheckinRequiresLocation: w.mobileCheckinRequiresLocation,
                allowUnreliableCheckins: w.allowUnreliableCheckins,
                permissionRequiredForExtraWorking: w.permissionRequiredForExtraWorking,
                geoLocationCount: w.geoLocationCount,
                colorCode: w.colorCode,
                organizationUnitId: w.organizationUnitId,
            };
        }),
    });
});
module.exports.upsertWorkPlanRegularPeriod = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertRegularWorkPlanPeriodRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    res.json(await dal_manager_1.dbManager.accessPacs2.upsertRegularWorkPlanPeriod(organizationId, request));
});
module.exports.upsertWorkPlanHalfFlexiblePeriod = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertHalfFlexibleWorkPlanPeriodRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    res.json(await dal_manager_1.dbManager.accessPacs2.upsertHalfFlexibleWorkPlanPeriod(organizationId, request));
});
module.exports.upsertWorkPlanFullFlexiblePeriod = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertFullFlexibleWorkPlanPeriodRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    res.json(await dal_manager_1.dbManager.accessPacs2.upsertFullFlexibleWorkPlanPeriod(organizationId, request));
});
module.exports.upsertWorkPlanShiftPeriod = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertShiftWorkPlanPeriodRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    res.json(await dal_manager_1.dbManager.accessPacs2.upsertShiftWorkPlanPeriod(organizationId, request));
});
module.exports.removeWorkPlanPeriod = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let workPlanPeriodId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workplanPeriodId");
    await dal_manager_1.dbManager.accessPacs2.removeWorkPlanPeriod(organizationId, workPlanPeriodId);
    res.json();
});
module.exports.obtainUserSelectionSessionForUserWorkPlan = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let validator = new api_validatorhelper_1.ValidatorHelper();
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let workPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "workPlanId");
    validator.validateRegex("operation", req.swagger.params.operation.value, false, /1|2/);
    let force = (0, api_validatorhelper_1.validateIsBoolean)(req.swagger.params.force.value, "force");
    validator.finalize();
    let operation = req.swagger.params.operation.value === 1 ? app_enums_1.enums.UserSelectionSessionType.UserWorkPlanAdd : app_enums_1.enums.UserSelectionSessionType.UserWorkPlanRemove;
    let authResult = (0, api_securityhelper_1.authorizeForObtainUserSelectionSessionForGeneric)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.obtainUserSelectionSession(organizationId, {
            requestUserId: authResult.jwt.userId,
            type: operation,
            relatedItemId: workPlanId,
            forceToCreate: force,
            isAuthorizedForAllUsers: authResult.hasOrganizationWideWritePPermission && authResult.hasOrganizationWideReadBasic,
            authorizedOrganizationUnitIds: authResult.readBasicOrganizationUnitIds ?? null,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    if (!dbResult.upserted) {
        throw (0, api_error_1.generateConflictError)({
            message: "There is already a session for that work plan. Please see detailes.",
            details: dbResult.overwritten,
        });
    }
    let result = {
        sessionId: dbResult.sessionId,
        expirationUtc: dbResult.expirationUtc,
        justRefreshed: dbResult.justRefreshed,
        overwritten: dbResult.overwritten
            ? {
                ownerUserId: dbResult.overwritten.ownerUserId,
                sessionId: dbResult.overwritten.sessionId,
                expirationUtc: dbResult.overwritten.expirationUtc,
                captionLines: dbResult.overwritten.captionLines,
            }
            : undefined,
    };
    res.json(result);
});
module.exports.removeUserWorkPlanMembership = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    const userWorkPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userWorkPlanId");
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessPacs2.removeUserWorkplanMembership(trx, organizationId, userWorkPlanId);
    }, auth.jwt.userId, organizationId);
    res.json();
});
module.exports.updateUserWorkPlanEntry = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    const userWorkPlanId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userWorkPlanId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.FullOptionalDateRange(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessPacs2.updateUserWorkplanMembership(trx, organizationId, userWorkPlanId, {
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
        });
    }, auth.jwt.userId, organizationId);
    res.json();
});
module.exports.updateUserWorkPlanMembershipInRange = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.UserWorkplanRangeRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessPacs2.updateUserWorkplanMembershipInRange(trx, organizationId, {
            userId: request.userId,
            range: {
                startDateTime: request.range.startDateTime,
                endDateTime: request.range.endDateTime,
            },
            items: request.items.map((i) => {
                return {
                    workplanId: i.workplanId,
                    range: {
                        startDateTime: i.range.startDateTime,
                        endDateTime: i.range.endDateTime,
                    },
                };
            }),
        });
    }, auth.jwt.userId, organizationId);
    res.json();
});
module.exports.upsertAutoShift = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const shiftId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "shiftId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.AutoShiftUpsertRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs2.upsertAutoShift(organizationId, {
            id: shiftId,
            supportMobileCheckin: request.supportMobileCheckin,
            supportManualLog: request.supportManualLog,
            useOnlyFirstAccess: request.useOnlyFirstAccess,
            name: request.name,
            groupIds: request.groupIds,
            regionIds: request.regionIds,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
        }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.upsertAutoShiftRule = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const shiftId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "shiftId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.AuthoShiftUpsertRuleRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs2.upsertAutoShiftRule(organizationId, {
            id: shiftId,
            rules: request.rules,
        }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.deleteAutoShift = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const shiftId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "shiftId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs2.deleteAutoShift(organizationId, { id: shiftId }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.listAutoShift = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.AutoShiftListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs2.listAutoShift(organizationId, {
            pagination: { skip: request.pagination?.skip, take: request.pagination?.take },
            filter: {
                name: request.filter?.name,
                startDateTime: request.filter?.startDateTime ? luxon_1.DateTime.fromJSDate(request.filter?.startDateTime) : undefined,
                endDateTime: request.filter?.endDateTime ? luxon_1.DateTime.fromJSDate(request.filter?.endDateTime) : undefined,
                groupIds: request.filter?.groupIds,
                regionIds: request.filter?.regionIds,
            },
        }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.reCalculateShift = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const shiftId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "shiftId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.ReCalculateRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs2.reCalculateShift(organizationId, { id: shiftId, start: request.start, end: request.end }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.upsertWorkPlanShiftImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.UpsertWorkPlanShiftImportRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs.upsertWorkPlanShiftImport(organizationId, { id: request.id, name: request.name, defaultWorkPlanId: request.defaultWorkPlanId, workPlanImportRules: request.workPlanImportRules }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.deleteWorkPlanShiftImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const shiftTemplateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "shiftTemplateId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanWrite)(req, organizationId);
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const result = await dal_manager_1.dbManager.accessPacs.deleteWorkPlanShiftImport(organizationId, { id: shiftTemplateId }, trx);
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(result);
});
module.exports.listShiftImports = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanRead)(req, organizationId);
    let jwt = req.auth_token;
    const request = new api_pacs_v1_models_1.RestPacsApiV1.WorkPlanShiftImportFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessPacs.listWorkPlanShiftImports(organizationId, {
        name: request.name,
        take: request.pagination.take,
        skip: request.pagination.skip,
    }, jwt.userId);
    res.json(dbResult);
});
module.exports.getWorkPlanShiftImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    let shiftTemplateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "shiftTemplateId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.getWorkPlanShiftImport(organizationId, shiftTemplateId);
    res.json(dbResult);
});
module.exports.generateExcelForWorkPlanShiftImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.WorkPlanShiftImportGetTemplate(req.body, new api_validatorhelper_1.ValidatorHelper());
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    let visitorExcelResponse = await business_pacs_1.BusinessWorkPlanImport.downloadWorkPlanExcel(organizationId, authResult.jwt.userId, locale, {
        userFilterId: request.userFilterId,
        dateRange: request.dateRange,
        templateId: request.templateId,
        days: request.days,
        type: request.type,
    });
    if (!visitorExcelResponse.errorMessage) {
        if (fs_1.default.existsSync(visitorExcelResponse.filePath)) {
            res.download(visitorExcelResponse.filePath, visitorExcelResponse.filename, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(visitorExcelResponse.filePath);
            });
        }
        else {
            res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json();
        }
    }
    else {
        res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
            message: visitorExcelResponse.errorMessage,
            noReport: true,
            showAsModal: false,
        });
    }
});
module.exports.uploadExcelForWorkPlanShiftImport = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let templateId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "templateId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendancePlanAssign)(req, organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    const validation = await business_pacs_1.BusinessWorkPlanImport.validateWorkPlanExcel(organizationId, authResult.jwt.userId, {
        locale: locale,
        fileBuffer: req.swagger.params.file.originalValue.buffer,
        type: req.swagger.params.type.originalValue,
        startDate: req.swagger.params.startDateTime.originalValue,
        endDate: req.swagger.params.endDateTime.originalValue,
        templateId: templateId,
    });
    if (validation.validationSuccess) {
        res.json({});
    }
    else if (validation.errorMessage) {
        res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
            message: validation.errorMessage,
            errorCells: validation.errorCells,
            errorUsers: validation.errorUsers,
            errorDates: validation.errorDates,
            showAsModal: true,
            noReport: true,
        });
    }
    else {
        if (fs_1.default.existsSync(validation.filePath)) {
            res.download(validation.filePath, validation.filename, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(validation.filePath);
            });
        }
        else {
            res.status(app_enums_1.enums.HttpStatusCode.TIMEOUT).json();
        }
    }
});
