"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RestPacsApiV1 = void 0;
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../../../app.enums");
const api_validatorhelper_1 = require("../../api.validatorhelper");
const dal_constants_1 = require("../../../dal/dal.constants");
const api_models_common_1 = require("../../api.models.common");
var RestPacsApiV1;
(function (RestPacsApiV1) {
    class AddVacationRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateDate("startDateTime", false, raw.startDateTime);
            validator.validateDate("endDateTime", false, raw.endDateTime);
            validator.validateString({
                field: "title",
                optional: false,
                input: raw.title,
                maxLength: 255,
                minLength: 1,
            });
            validator.validateEnum("type", false, app_enums_1.enums.VacationType, raw.type);
            this.id = raw.id;
            this.startDateTime = new Date(raw.startDateTime);
            this.endDateTime = new Date(raw.endDateTime);
            if (this.startDateTime >= this.endDateTime) {
                validator.addRangeError(["startDateTime", "endDateTime"]);
            }
            this.title = raw.title;
            this.type = raw.type;
            validator.finalize();
        }
    }
    RestPacsApiV1.AddVacationRequest = AddVacationRequest;
    class UpsertWorkPlanRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateUUID("regionId", true, raw.regionId);
            validator.validateString({
                field: "colorCode",
                optional: false,
                input: raw.colorCode,
                minLength: 6,
                maxLength: 6,
            });
            validator.validateUUID("organizationUnitId", true, raw.organizationUnitId);
            validator.finalize();
            this.id = raw.id;
            this.name = raw.name;
            this.type = raw.type;
            this.regionId = raw.regionId;
            this.accessCheckType = raw.accessCheckType;
            this.offset = raw.offset;
            this.timezone = raw.timezone;
            this.ignoreHolidays = raw.ignoreHolidays;
            this.allowMobileCheckins = raw.allowMobileCheckins;
            this.mobileCheckinRequiresLocation = raw.mobileCheckinRequiresLocation;
            this.allowUnreliableCheckins = raw.allowUnreliableCheckins;
            this.permissionRequiredForExtraWorking = !raw.permissionRequiredForExtraWorking ? false : raw.permissionRequiredForExtraWorking;
            this.typedGeoLocations = raw.typedGeoLocations;
            this.colorCode = raw.colorCode;
            this.organizationUnitId = raw.organizationUnitId;
            this.ignoreUnpairedAccesses = raw.ignoreUnpairedAccesses;
        }
    }
    RestPacsApiV1.UpsertWorkPlanRequest = UpsertWorkPlanRequest;
    class UserListWithPagination extends api_models_common_1.ApiModelsCommon.PaginationRequest {
        constructor(raw, validator) {
            super(raw, 200, validator, false);
            validator.validateUUIDArray("userIds", true, raw.userIds);
            validator.validateUUIDArray("organizationUnitIds", true, raw.organizationUnitIds);
            validator.validateUUIDArray("userGroupIds", true, raw.userGroupIds);
            validator.validateIsBoolean("organizationUnitHierarchically", true, raw.organizationUnitHierarchically);
            this.userIds = raw.userIds;
            this.organizationUnitIds = raw.organizationUnitIds;
            this.organizationUnitHierarchically = raw.organizationUnitHierarchically;
            this.userGroupIds = raw.userGroupIds;
            validator.finalize();
        }
    }
    RestPacsApiV1.UserListWithPagination = UserListWithPagination;
    class DateTime {
        constructor(raw, validator) {
            validator.validateDate("dateTime", true, raw.dateTime);
            this.dateTime = raw.dateTime;
            validator.finalize();
        }
    }
    RestPacsApiV1.DateTime = DateTime;
    class WorkPlanToleranceValuesRequest {
        constructor(raw, validator) {
            validator.validateInteger({
                field: "earlyStart",
                optional: false,
                input: raw.earlyStart,
                min: 0,
                max: 1440,
            });
            validator.validateInteger({
                field: "earlyLeave",
                optional: false,
                input: raw.earlyLeave,
                min: 0,
                max: 1440,
            });
            validator.validateInteger({
                field: "lateStart",
                optional: false,
                input: raw.lateStart,
                min: 0,
                max: 1440,
            });
            validator.validateInteger({
                field: "lateLeave",
                optional: false,
                input: raw.lateLeave,
                min: 0,
                max: 1440,
            });
            validator.finalize();
            this.earlyStart = parseInt(raw.earlyStart);
            this.earlyLeave = parseInt(raw.earlyLeave);
            this.lateStart = parseInt(raw.lateStart);
            this.lateLeave = parseInt(raw.lateLeave);
        }
    }
    RestPacsApiV1.WorkPlanToleranceValuesRequest = WorkPlanToleranceValuesRequest;
    class UpsertRegularWorkPlanPeriodRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateUUID("workPlanId", false, raw.workPlanId);
            validator.validateDate("periodStartDateTime", false, raw.periodStartDateTime);
            validator.validateObjectInstance("tolerances", false, raw.tolerances);
            validator.validateEnum("routineType", false, app_enums_1.enums.PacsWorkplanPeriodRoutineTypes, raw.routineType);
            validator.validateInteger({ field: "offset", optional: true, input: raw.offset, min: -720, max: 720 });
            if (raw.tolerances) {
                try {
                    this.tolerances = new WorkPlanToleranceValuesRequest(raw.tolerances, validator);
                }
                catch {
                }
            }
            validator.validateObjectArray("dayItems", false, raw.dayItems);
            this.dayItems = [];
            for (const wh of raw.dayItems) {
                try {
                    let whr = new WorkPlanDayItemRequest(wh, raw.routineType, validator);
                    if (this.dayItems.find((t) => t.index === whr.index)) {
                        validator.addContradictionError(["index"], "Specific day of week can not be added twice");
                    }
                    this.dayItems.push(whr);
                }
                catch { }
            }
            this.id = raw.id;
            this.routineType = raw.routineType;
            this.workPlanId = raw.workPlanId;
            this.periodStartDateTime = new Date(raw.periodStartDateTime);
            this.offset = raw.offset ?? null;
        }
    }
    RestPacsApiV1.UpsertRegularWorkPlanPeriodRequest = UpsertRegularWorkPlanPeriodRequest;
    class UpsertHalfFlexibleWorkPlanPeriodRequest extends UpsertRegularWorkPlanPeriodRequest {
        constructor(raw, validator) {
            try {
                super(raw, validator);
                validator.validateInteger({
                    field: "weeklyExpectedWorkDuration",
                    optional: false,
                    input: raw.weeklyExpectedWorkDuration,
                    min: 0,
                    max: 10080,
                });
                validator.finalize();
                this.weeklyExpectedWorkDuration = parseInt(raw.weeklyExpectedWorkDuration);
            }
            catch { }
        }
    }
    RestPacsApiV1.UpsertHalfFlexibleWorkPlanPeriodRequest = UpsertHalfFlexibleWorkPlanPeriodRequest;
    class WorkingHoursRangeRequest {
        constructor(raw, validator) {
            validator.validateRegex("startTime", raw.startTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            validator.validateRegex("endTime", raw.endTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            if (raw.startTime >= raw.endTime) {
                validator.addRangeError(["startTime", "endTime"], "Start time can not be later then end time");
            }
            validator.finalize();
            this.startTime = raw.startTime;
            this.endTime = raw.endTime;
        }
    }
    RestPacsApiV1.WorkingHoursRangeRequest = WorkingHoursRangeRequest;
    class FixedBreakRequest {
        constructor(raw, validator) {
            validator.validateRegex("startTime", raw.startTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            validator.validateRegex("endTime", raw.endTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            if (raw.startTime >= raw.endTime) {
                validator.addRangeError(["startTime", "endTime"], "Start time can not be later then end time");
            }
            else {
                let tdiff = moment_1.default.duration((0, moment_1.default)(raw.endTime, "HHmm").diff((0, moment_1.default)(raw.startTime, "HHmm"))).asMinutes();
                validator.validateInteger({
                    field: "allowedMaxBreakDuration",
                    optional: true,
                    input: raw.allowedMaxBreakDuration,
                    min: 1,
                    max: tdiff,
                });
                validator.validateInteger({
                    field: "executeAutomaticallyBelowDuration",
                    optional: true,
                    input: raw.executeAutomaticallyBelowDuration,
                    min: 0,
                    max: tdiff,
                });
            }
            validator.validateEnum("type", false, app_enums_1.enums.BreakType, raw.type);
            validator.finalize();
            this.type = parseInt(raw.type);
            this.startTime = raw.startTime;
            this.endTime = raw.endTime;
            this.allowedMaxBreakDuration = raw.allowedMaxBreakDuration;
            this.executeAutomaticallyBelowDuration = raw.executeAutomaticallyBelowDuration;
        }
    }
    RestPacsApiV1.FixedBreakRequest = FixedBreakRequest;
    class ClaimBreakRequest {
        constructor(raw, validator) {
            validator.validateInteger({
                field: "workDuration",
                optional: false,
                input: raw.workDuration,
                min: 1,
                max: 1440,
            });
            validator.validateInteger({
                field: "breakDuration",
                optional: false,
                input: raw.breakDuration,
                min: 1,
                max: 1440,
            });
            validator.validateEnum("type", false, app_enums_1.enums.BreakType, raw.type);
            validator.finalize();
            this.type = parseInt(raw.type);
            this.workDuration = parseInt(raw.workDuration);
            this.breakDuration = parseInt(raw.breakDuration);
            if (this.workDuration + this.breakDuration > 1440) {
                validator.addContradictionError(["workDuration", "breakDuration"], "Work duration + break duration can not exceed one day");
                validator.finalize();
            }
            this.executeAutomaticallyBelowDuration = raw.executeAutomaticallyBelowDuration;
        }
    }
    RestPacsApiV1.ClaimBreakRequest = ClaimBreakRequest;
    class ExtraWorkBreakRequest {
        constructor(raw, validator) {
            validator.validateRegex("periodStartTime", raw.periodStartTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            validator.validateRegex("periodEndTime", raw.periodEndTime, false, /^([0-1]?[0-9]|2[0-3])[0-5][0-9]$/);
            if (raw.periodStartTime >= raw.periodEndTime) {
                validator.addRangeError(["periodStartTime", "periodEndTime"], "Start time can not be later then end time");
            }
            else {
                const tdiff = moment_1.default.duration((0, moment_1.default)(raw.periodEndTime, "HHmm").diff((0, moment_1.default)(raw.periodStartTime, "HHmm"))).asMinutes();
                validator.validateInteger({
                    field: "minWorkDuration",
                    optional: true,
                    input: raw.minWorkDuration,
                    min: 0,
                    max: tdiff,
                });
                validator.validateInteger({
                    field: "breakDuration",
                    optional: false,
                    input: raw.breakDuration,
                    min: 1,
                    max: tdiff,
                });
            }
            validator.validateEnum("type", false, app_enums_1.enums.BreakType, raw.type);
            validator.finalize();
            this.type = parseInt(raw.type);
            this.periodStartTime = raw.periodStartTime;
            this.periodEndTime = raw.periodEndTime;
            this.minWorkDuration = raw.minWorkDuration ?? 0;
            this.breakDuration = raw.breakDuration;
        }
    }
    RestPacsApiV1.ExtraWorkBreakRequest = ExtraWorkBreakRequest;
    class WorkPlanDayBreakItemRequest {
        constructor(raw, routineType, validator) {
            if (routineType === app_enums_1.enums.PacsWorkplanPeriodRoutineTypes.Week) {
                validator.validateEnum("index", false, app_enums_1.enums.DayOfWeekForWorking, raw.index);
            }
            else if (routineType === app_enums_1.enums.PacsWorkplanPeriodRoutineTypes.Day) {
                validator.validateInteger({
                    field: "index",
                    optional: false,
                    min: 1,
                    input: raw.index,
                });
            }
            validator.validateObjectArray("fixedBreaks", true, raw.fixedBreaks);
            if (raw.fixedBreaks && raw.fixedBreaks.length > 0) {
                this.fixedBreaks = [];
                for (const wh of raw.fixedBreaks) {
                    try {
                        let whr = new FixedBreakRequest(wh, validator);
                        for (const whi of this.fixedBreaks) {
                            if (whr.endTime >= whi.startTime && whr.startTime <= whi.endTime) {
                                validator.addRangeError(["startTime", "endTime"], "Overlapped fixed breaks");
                            }
                        }
                        this.fixedBreaks.push(whr);
                    }
                    catch { }
                }
            }
            validator.validateObjectInstance("claimBreak", true, raw.claimBreak);
            if (raw.claimBreak) {
                this.claimBreak = new ClaimBreakRequest(raw.claimBreak, validator);
            }
            validator.validateObjectArray("extraWorkBreaks", true, raw.extraWorkBreaks);
            if (raw.extraWorkBreaks?.length > 0) {
                this.extraWorkBreaks = [];
                for (const wh of raw.extraWorkBreaks) {
                    try {
                        const whr = new ExtraWorkBreakRequest(wh, validator);
                        for (const whi of this.extraWorkBreaks) {
                            if (whr.periodEndTime >= whi.periodStartTime && whr.periodStartTime <= whi.periodEndTime) {
                                validator.addRangeError(["periodStartTime", "periodEndTime"], "Overlapped fixed breaks");
                            }
                        }
                        this.extraWorkBreaks.push(whr);
                    }
                    catch { }
                }
            }
            validator.finalize();
            this.index = raw.index;
        }
    }
    RestPacsApiV1.WorkPlanDayBreakItemRequest = WorkPlanDayBreakItemRequest;
    class WorkPlanDayItemRequest extends WorkPlanDayBreakItemRequest {
        constructor(raw, routineType, validator) {
            try {
                super(raw, routineType, validator);
                validator.validateObjectArray("workingHours", true, raw.workingHours);
                if (raw.workingHours && raw.workingHours.length > 0) {
                    this.workingHours = [];
                    for (const wh of raw.workingHours) {
                        try {
                            let whr = new WorkingHoursRangeRequest(wh, validator);
                            for (const whi of this.workingHours) {
                                if (whr.endTime >= whi.startTime && whr.startTime <= whi.endTime) {
                                    validator.addRangeError(["startTime", "endTime"], "Overlapped working hours");
                                }
                            }
                            this.workingHours.push(whr);
                        }
                        catch { }
                    }
                }
                validator.finalize();
            }
            catch { }
        }
    }
    RestPacsApiV1.WorkPlanDayItemRequest = WorkPlanDayItemRequest;
    class UpsertFullFlexibleWorkPlanPeriodRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateUUID("workPlanId", false, raw.workPlanId);
            validator.validateDate("periodStartDateTime", false, raw.periodStartDateTime);
            validator.validateObjectArray("dayBreakItems", false, raw.dayBreakItems);
            validator.validateEnum("routineType", false, app_enums_1.enums.PacsWorkplanPeriodRoutineTypes, raw.routineType);
            validator.validateInteger({ field: "offset", optional: true, input: raw.offset, min: -720, max: 720 });
            this.dayBreakItems = [];
            for (const wh of raw.dayBreakItems) {
                try {
                    let whr = new WorkPlanDayBreakItemRequest(wh, raw.routineType, validator);
                    if (this.dayBreakItems.find((t) => t.index === whr.index)) {
                        validator.addContradictionError(["dayOfWeek"], "Specific day of week can not be added twice");
                    }
                    this.dayBreakItems.push(whr);
                }
                catch { }
            }
            validator.validateInteger({
                field: "weeklyExpectedWorkDuration",
                optional: false,
                input: raw.weeklyExpectedWorkDuration,
                min: 0,
                max: 10080,
            });
            validator.finalize();
            this.id = raw.id;
            this.routineType = raw.routineType;
            this.workPlanId = raw.workPlanId;
            this.periodStartDateTime = new Date(raw.periodStartDateTime);
            this.weeklyExpectedWorkDuration = parseInt(raw.weeklyExpectedWorkDuration);
            this.offset = raw.offset ?? null;
        }
    }
    RestPacsApiV1.UpsertFullFlexibleWorkPlanPeriodRequest = UpsertFullFlexibleWorkPlanPeriodRequest;
    class UpsertShiftWorkPlanPeriodRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateUUID("workPlanId", false, raw.workPlanId);
            validator.validateDate("periodStartDateTime", false, raw.periodStartDateTime);
            validator.validateObjectArray("dayBreakItems", false, raw.dayBreakItems);
            validator.validateEnum("routineType", false, app_enums_1.enums.PacsWorkplanPeriodRoutineTypes, raw.routineType);
            validator.validateInteger({ field: "offset", optional: true, input: raw.offset, min: -720, max: 720 });
            this.dayBreakItems = [];
            for (const wh of raw.dayBreakItems) {
                try {
                    let whr = new WorkPlanDayBreakItemRequest(wh, raw.routineType, validator);
                    if (this.dayBreakItems.find((t) => t.index === whr.index)) {
                        validator.addContradictionError(["dayOfWeek"], "Specific day of week can not be added twice");
                    }
                    this.dayBreakItems.push(whr);
                }
                catch { }
            }
            validator.validateObjectInstance("tolerances", false, raw.tolerances);
            if (raw.tolerances) {
                try {
                    this.tolerances = new WorkPlanToleranceValuesRequest(raw.tolerances, validator);
                }
                catch {
                }
            }
            validator.validateInteger({
                field: "workDuration",
                optional: false,
                input: raw.workDuration,
                min: 1,
            });
            validator.validateInteger({
                field: "restDuration",
                optional: false,
                input: raw.restDuration,
                min: 1,
            });
            validator.finalize();
            this.id = raw.id;
            this.routineType = raw.routineType;
            this.workPlanId = raw.workPlanId;
            this.periodStartDateTime = new Date(raw.periodStartDateTime);
            this.workDuration = parseInt(raw.workDuration);
            this.restDuration = parseInt(raw.restDuration);
            this.offset = raw.offset ?? null;
        }
    }
    RestPacsApiV1.UpsertShiftWorkPlanPeriodRequest = UpsertShiftWorkPlanPeriodRequest;
    class PPermissionTypeRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateIsBoolean("isPassive", false, raw.isPassive);
            validator.validateIsBoolean("hrCanInit", false, raw.hrCanInit);
            validator.validateIsBoolean("personnelCanInit", false, raw.personnelCanInit);
            validator.validateIsBoolean("unitManagerCanInit", false, raw.unitManagerCanInit);
            validator.validateIsBoolean("isDailyScheduled", false, raw.isDailyScheduled);
            validator.validateIsArrayOfString("approvementList", false, raw.approvementList);
            validator.validateIsBoolean("sendNotification", false, raw.sendNotification);
            validator.validateIsBoolean("userMustSign", false, raw.userMustSign);
            validator.validateIsBoolean("needsManagerDeputy", false, raw.needsManagerDeputy);
            validator.validateIsBoolean("needsUserDeputy", false, raw.needsUserDeputy);
            validator.validateEnum("method", false, dal_constants_1.DalConstants.PPermissionType, raw.method);
            validator.validateIsBoolean("showRemainingCount", false, raw.showRemainingCount);
            validator.validateIsBoolean("isUnpaidLeave", false, raw.isUnpaidLeave);
            validator.validateIsBoolean("notifyHR", false, raw.notifyHR);
            validator.validateIsBoolean("noteIsMandatory", false, raw.noteIsMandatory);
            validator.validateObject("dynamicForm", true, raw.dynamicForm, validator);
            validator.validateIsBoolean("applyOnHolidays", false, raw.applyOnHolidays);
            validator.validateIsBoolean("applyOnlyOnWorkingHours", false, raw.applyOnlyOnWorkingHours);
            validator.validateIsBoolean("notifyManagers", false, raw.notifyManagers);
            validator.validateIsBoolean("isNegativeBalanceAllowed", false, raw.isNegativeBalanceAllowed);
            validator.validateObjectArray("applicableDays", true, raw.applicableDays);
            this.id = raw.id;
            this.name = raw.name;
            this.isPassive = raw.isPassive;
            this.description = raw.description;
            this.hrCanInit = raw.hrCanInit;
            this.personnelCanInit = raw.personnelCanInit;
            this.unitManagerCanInit = raw.unitManagerCanInit;
            this.isDailyScheduled = raw.isDailyScheduled;
            this.minValue = raw.minValue;
            this.maxValue = raw.maxValue;
            this.maxValueBoundaryInterval = raw.maxValueBoundaryInterval;
            this.method = raw.method;
            this.approvementList = raw.approvementList;
            this.sendNotification = raw.sendNotification;
            this.userMustSign = raw.userMustSign;
            this.needsManagerDeputy = raw.needsManagerDeputy;
            this.needsUserDeputy = raw.needsUserDeputy;
            this.type = raw.type;
            this.showRemainingCount = raw.showRemainingCount;
            this.isUnpaidLeave = raw.isUnpaidLeave;
            this.notifyHR = raw.notifyHR;
            this.noteIsMandatory = raw.noteIsMandatory;
            this.dynamicForm = raw.dynamicForm;
            this.applyOnHolidays = raw.applyOnHolidays;
            this.applyOnlyOnWorkingHours = raw.applyOnlyOnWorkingHours;
            this.notifyManagers = raw.notifyManagers;
            this.isNegativeBalanceAllowed = raw.isNegativeBalanceAllowed;
            this.applicableDays = raw.applicableDays ?? [];
            validator.finalize();
        }
    }
    RestPacsApiV1.PPermissionTypeRequest = PPermissionTypeRequest;
    class PPermissionTypeFilter {
        constructor(raw, validator) {
            this.name = raw.name;
            this.isPassive = raw.isPassive;
            this.pagination = {
                take: raw.pagination.take,
                skip: raw.pagination.skip,
            };
        }
    }
    RestPacsApiV1.PPermissionTypeFilter = PPermissionTypeFilter;
    class PPermissionValidationRequest {
    }
    RestPacsApiV1.PPermissionValidationRequest = PPermissionValidationRequest;
    class PPermissionDeputyInfoRequest {
    }
    RestPacsApiV1.PPermissionDeputyInfoRequest = PPermissionDeputyInfoRequest;
    class SelfPPermissionRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateUUID("deputyUserId", true, raw.deputyUserId);
            validator.validateDate("startDateTime", false, raw.dateRange.startDateTime);
            validator.validateDate("endDateTime", false, raw.dateRange.endDateTime);
            validator.validateUUID("ppermissionTypeId", false, raw.ppermissionTypeId);
            validator.validateUUID("dynamicFormId", true, raw.dynamicFormId);
            this.id = raw.id;
            this.ppermissionTypeId = raw.ppermissionTypeId;
            this.startDateTime = raw.dateRange.startDateTime;
            this.endDateTime = raw.dateRange.endDateTime;
            this.note = raw.note;
            this.deputyUserId = raw.deputyUserId;
            this.attachmentList = raw.attachmentList;
            this.approve = raw.approve;
            this.dynamicFormId = raw.dynamicFormId;
            this.dynamicFormData = raw.dynamicFormData;
            validator.finalize();
        }
    }
    RestPacsApiV1.SelfPPermissionRequest = SelfPPermissionRequest;
    class PPermissionRequestWithUsers {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateDate("startDateTime", false, raw.dateRange.startDateTime);
            validator.validateDate("endDateTime", false, raw.dateRange.endDateTime);
            validator.validateUUID("ppermissionTypeId", false, raw.ppermissionTypeId);
            validator.validateUUID("dynamicFormId", true, raw.dynamicFormId);
            this.id = raw.id;
            this.ppermissionTypeId = raw.ppermissionTypeId;
            this.startDateTime = raw.dateRange.startDateTime;
            this.endDateTime = raw.dateRange.endDateTime;
            this.note = raw.note;
            this.userAndDeputyList = raw.userAndDeputyList;
            this.attachmentList = raw.attachmentList;
            this.approve = raw.approve;
            this.dynamicFormId = raw.dynamicFormId;
            this.dynamicFormData = raw.dynamicFormData;
            this.performValidations = raw.performValidations;
            validator.finalize();
        }
    }
    RestPacsApiV1.PPermissionRequestWithUsers = PPermissionRequestWithUsers;
    class MultiplePPermissionRequestForUser {
        constructor(raw, validator) {
            for (const range of raw.ranges) {
                validator.validateDate("startDateTime", false, range.startDateTime);
                validator.validateDate("endDateTime", false, range.endDateTime);
            }
            validator.validateUUID("ppermissionTypeId", false, raw.ppermissionTypeId);
            this.ppermissionTypeId = raw.ppermissionTypeId;
            this.ranges = [];
            for (const range of raw.ranges) {
                this.ranges.push({
                    startDateTime: new Date(range.startDateTime),
                    endDateTime: new Date(range.endDateTime),
                });
            }
            this.note = raw.note;
            this.userAndDeputy = raw.userAndDeputy;
            this.attachmentList = raw.attachmentList;
            validator.finalize();
        }
    }
    RestPacsApiV1.MultiplePPermissionRequestForUser = MultiplePPermissionRequestForUser;
    class UserDeputyListRequest {
        constructor(raw, validator) {
            this.list = raw.list;
            for (let userAndDeputy of this.list) {
                validator.validateUUID("userId", false, userAndDeputy.userId);
                if (userAndDeputy.deputyUserId) {
                    validator.validateUUID("deputyUserId", true, userAndDeputy.deputyUserId);
                }
            }
            validator.finalize();
        }
    }
    RestPacsApiV1.UserDeputyListRequest = UserDeputyListRequest;
    class PPermissionRequest extends SelfPPermissionRequest {
        constructor(raw, validator) {
            super(raw, new api_validatorhelper_1.ValidatorHelper());
            validator.finalize();
        }
    }
    RestPacsApiV1.PPermissionRequest = PPermissionRequest;
    class SelfPPermissionFilter {
        constructor(raw, validator) {
            validator.validateUUID("ppermissionTypeId", true, raw.ppermissionTypeId);
            validator.validateDate("startDateTime", true, raw.dateRange?.startDateTime);
            validator.validateDate("endDateTime", true, raw.dateRange?.endDateTime);
            validator.finalize();
            this.ppermissionTypeId = raw.ppermissionTypeId;
            this.startDateTime = raw.dateRange.startDateTime;
            this.endDateTime = raw.dateRange.endDateTime;
            this.pagination = {
                take: raw.pagination.take,
                skip: raw.pagination.skip,
            };
            this.ppermissionIds = raw.ppermissionIds;
            this.status = raw.status;
        }
    }
    RestPacsApiV1.SelfPPermissionFilter = SelfPPermissionFilter;
    class PPermissionFilter extends SelfPPermissionFilter {
        constructor(raw, validator) {
            super(raw, validator);
            validator.validateUUIDArray("userIds", true, raw.userIds);
            validator.validateRegex("sortOrder", raw.sortOrder, true, /(^ASC$)|(^DESC$)/);
            validator.validateUUIDArray("organizationUnitIds", true, raw.organizationUnitIds);
            validator.validateUUIDArray("userGroupIds", true, raw.userGroupIds);
            validator.validateIsBoolean("organizationUnitHierarchically", true, raw.organizationUnitHierarchically);
            this.organizationUnitIds = raw.organizationUnitIds;
            this.organizationUnitHierarchically = raw.organizationUnitHierarchically;
            this.userGroupIds = raw.userGroupIds;
            this.userIds = raw.userIds;
            this.sortOrder = raw.sortOrder ? raw.sortOrder : "DESC";
            this.sortType = raw.sortType ? raw.sortType : app_enums_1.enums.PPermissionFilterSortType.PPermissionRequestDate;
            validator.finalize();
        }
    }
    RestPacsApiV1.PPermissionFilter = PPermissionFilter;
    class ApprovePPermissionRequest {
        constructor(raw, validator) {
            this.note = raw.note;
            this.status = raw.status;
        }
    }
    RestPacsApiV1.ApprovePPermissionRequest = ApprovePPermissionRequest;
    class ApplyUserSelectionWorkPlanRequest {
        constructor(raw, validator) {
            this.ranges = [];
            validator.validateIsBoolean("enforce", true, raw.enforce);
            try {
                validator.validateObjectArray("ranges", false, raw.ranges);
                validator.validateArrayLength("ranges", false, 1, -1, raw.ranges);
                for (let index = 0; index < raw.ranges.length; index++) {
                    try {
                        let rng = new FullOptionalDateRange(raw.ranges[index], validator);
                        this.ranges.push(rng);
                    }
                    catch { }
                }
            }
            catch { }
            this.enforce = raw.enforce === "true" || raw.enforce === true;
            validator.finalize();
        }
    }
    RestPacsApiV1.ApplyUserSelectionWorkPlanRequest = ApplyUserSelectionWorkPlanRequest;
    class FullOptionalDateRange {
        constructor(raw, validator) {
            validator.validateDate("startDateTime", true, raw.startDateTime);
            validator.validateDate("endDateTime", true, raw.endDateTime);
            validator.finalize();
            if (raw.startDateTime) {
                this.startDateTime = new Date(raw.startDateTime);
            }
            if (raw.endDateTime) {
                this.endDateTime = new Date(raw.endDateTime);
            }
        }
    }
    RestPacsApiV1.FullOptionalDateRange = FullOptionalDateRange;
    class DateRange {
        constructor(raw, validator) {
            validator.validateDate("startDateTime", false, raw.startDateTime);
            validator.validateDate("endDateTime", false, raw.endDateTime);
            validator.finalize();
            this.startDateTime = (0, moment_1.default)(raw.startDateTime).toDate();
            this.endDateTime = (0, moment_1.default)(raw.endDateTime).toDate();
        }
    }
    RestPacsApiV1.DateRange = DateRange;
    class DateRangeWithPaging {
        constructor(raw, validator) {
            validator.validateDate("startDateTime", false, raw.dateRange.startDateTime);
            validator.validateDate("endDateTime", false, raw.dateRange.endDateTime);
            validator.finalize();
            this.startDateTime = (0, moment_1.default)(raw.dateRange.startDateTime).toDate();
            this.endDateTime = (0, moment_1.default)(raw.dateRange.endDateTime).toDate();
            this.pagination = {
                take: raw.pagination.take,
                skip: raw.pagination.skip,
            };
        }
    }
    RestPacsApiV1.DateRangeWithPaging = DateRangeWithPaging;
    class UpdateMonthlyReportOfUserRequest {
        constructor(raw, validator) {
            validator.validateUUID("userId", false, raw.userId);
            this.checked = raw.checked;
            this.note = raw.note;
            this.year = raw.year;
            this.month = raw.month;
            this.working = raw.working;
            this.userId = raw.userId;
        }
    }
    RestPacsApiV1.UpdateMonthlyReportOfUserRequest = UpdateMonthlyReportOfUserRequest;
    class UserFilter {
        constructor(raw, validator) {
            validator.validateUUIDArray("organizationUnitIds", true, raw.organizationUnitIds);
            this.organizationUnitIds = raw.organizationUnitIds;
            this.organizationUnitHierarchically = raw.organizationUnitHierarchically;
            validator.validateUUIDArray("userGroupIds", true, raw.userGroupIds);
            this.userGroupIds = raw.userGroupIds;
            validator.validateUUIDArray("workPlanIds", true, raw.workPlanIds);
            this.workPlanIds = raw.workPlanIds;
            validator.validateUUIDArray("userIds", true, raw.userIds);
            this.userIds = raw.userIds;
            if (raw.dateRange) {
                this.dateRange = raw.dateRange;
            }
        }
    }
    RestPacsApiV1.UserFilter = UserFilter;
    class DailySummaryFilter {
        constructor(raw, validator) {
            validator.validateDate("date", false, raw.date);
            this.status = raw.status;
            this.date = new Date(raw.date);
            this.pagination = raw.pagination;
            this.workStatusFilter = raw.workStatusFilter;
            this.sortingMethod = raw.sortingMethod || app_enums_1.enums.EmployeeInfoSorting.Username;
            this.hasOrganizationWide = raw.hasOrganizationWide;
            this.permittedUnitIds = raw.permittedUnitIds;
            this.userIds = raw.userIds;
            this.organizationUnitIds = raw.organizationUnitIds;
            this.userGroupIds = raw.userGroupIds;
            this.userOrganizationStatus = raw.userOrganizationStatus;
            this.applyOrganizationUnitFilterHierarchically = raw.applyOrganizationUnitFilterHierarchically;
            this.workPlanIds = raw.workPlanIds;
        }
    }
    RestPacsApiV1.DailySummaryFilter = DailySummaryFilter;
    class CurrentWorkingSummaryPPermissionsFilter {
        constructor(raw, validator) {
            validator.validateUUIDArray("organizationUnitIds", true, raw.organizationUnitIds);
            this.organizationUnitIds = raw.organizationUnitIds;
            this.pagination = raw.pagination;
            this.currentWorkingSummaryPPermissionType = raw.currentWorkingSummaryPPermissionType;
            validator.finalize();
        }
    }
    RestPacsApiV1.CurrentWorkingSummaryPPermissionsFilter = CurrentWorkingSummaryPPermissionsFilter;
    class SelfPacsNotificationUpsertRequest {
        constructor(raw, validator) {
            validator.validateEnum("type", false, app_enums_1.enums.PacsNotificationType, raw.type);
            validator.validateUUID("id", true, raw.id);
            this.id = raw.id;
            this.type = raw.type;
            this.scheduleSettings = {
                daily: true,
                weekly: false,
                monthly: false,
            };
            validator.finalize();
        }
    }
    RestPacsApiV1.SelfPacsNotificationUpsertRequest = SelfPacsNotificationUpsertRequest;
    class PacsNotificationUpsertRequest extends SelfPacsNotificationUpsertRequest {
        constructor(raw, validator) {
            super(raw, validator);
            validator.validateUUID("targetUserId", false, raw.targetUserId);
            validator.finalize();
            this.targetUserId = raw.targetUserId;
        }
    }
    RestPacsApiV1.PacsNotificationUpsertRequest = PacsNotificationUpsertRequest;
    class MobileCheckinRequest {
        constructor(raw, validator) {
            if (raw.timestamp) {
                validator.validateDate("timestamp", true, raw.timestamp);
                this.timestamp = new Date(raw.timestamp);
            }
            else {
                this.timestamp = new Date();
            }
            validator.validateUUID("accessControlPointId", true, raw.accessControlPointId);
            this.latitude = raw.latitude;
            this.longitude = raw.longitude;
            this.isLocationReliable = raw.isLocationReliable;
            this.accessControlPointId = raw.accessControlPointId;
            validator.finalize();
        }
    }
    RestPacsApiV1.MobileCheckinRequest = MobileCheckinRequest;
    class ManualAnnualPPermissionRequest {
        constructor(raw, validator) {
            this.remainingAnnualPPermission = raw.remainingAnnualPPermission;
            validator.finalize();
        }
    }
    RestPacsApiV1.ManualAnnualPPermissionRequest = ManualAnnualPPermissionRequest;
    class EmployeeStateRequest {
        constructor(raw, validator) {
            validator.validateDate("timestamp", true, raw.timestamp);
            validator.finalize();
            if (raw.timestamp) {
                this.timestamp = new Date(raw.timestamp);
            }
        }
    }
    RestPacsApiV1.EmployeeStateRequest = EmployeeStateRequest;
    class UserWorkplanRangeRequest {
        constructor(raw, validator) {
            this.items = [];
            validator.validateUUID("userId", false, raw.userId);
            this.userId = raw.userId;
            this.range = new FullOptionalDateRange(raw.range, validator);
            validator.validateObjectArray("items", false, raw.items);
            validator.validateArrayLength("items", false, 1, -1, raw.items);
            for (let index = 0; index < raw.items.length; index++) {
                validator.validateUUID("items[" + index + "]", false, raw.items[index].workplanId);
                let rng = new FullOptionalDateRange(raw.items[index].range, validator);
                this.items.push({
                    workplanId: raw.items[index].workplanId,
                    range: rng,
                });
            }
        }
    }
    RestPacsApiV1.UserWorkplanRangeRequest = UserWorkplanRangeRequest;
    class AutoShiftUpsertRequest {
        constructor(raw, validator) {
            raw.groupIds = [...new Set(raw.groupIds)];
            validator.validateUUIDArray("groupIds", false, raw.groupIds);
            validator.validateArrayLength("groupIds", false, 1, -1, raw.groupIds);
            for (let index = 0; index < raw.groupIds.length; index++) {
                validator.validateUUID("groupId", false, raw.groupIds[index]);
            }
            this.groupIds = raw.groupIds;
            raw.regionIds = [...new Set(raw.regionIds)];
            validator.validateUUIDArray("regionIds", false, raw.regionIds);
            validator.validateArrayLength("regionIds", false, 1, -1, raw.regionIds);
            for (let index = 0; index < raw.regionIds.length; index++) {
                validator.validateUUID("regionId", false, raw.regionIds[index]);
            }
            this.regionIds = raw.regionIds;
            validator.validateDate("startDateTime", false, raw.startDateTime);
            this.startDateTime = raw.startDateTime;
            if (raw.endDateTime) {
                validator.validateDate("endDateTime", raw.endDateTime);
                this.endDateTime = raw.endDateTime;
            }
            validator.validateString({ optional: false, field: "name" });
            this.name = raw.name;
            validator.validateIsBoolean("supportMobileCheckin", false, raw.supportMobileCheckin);
            this.supportMobileCheckin = raw.supportMobileCheckin;
            validator.validateIsBoolean("supportManualLog", false, raw.supportManualLog);
            this.supportManualLog = raw.supportManualLog;
            validator.validateIsBoolean("useOnlyFirstAccess", false, raw.useOnlyFirstAccess);
            this.useOnlyFirstAccess = raw.useOnlyFirstAccess;
        }
    }
    RestPacsApiV1.AutoShiftUpsertRequest = AutoShiftUpsertRequest;
    class AuthoShiftUpsertRuleRequest {
        constructor(raw, validator) {
            validator.validateObjectArray("rules", false, raw.rules);
            validator.validateArrayLength("rules", false, 1, -1, raw.rules);
            for (let index = 0; index < raw.rules.length; index++) {
                validator.validateUUID("workPlanId", false, raw.rules[index].id);
                validator.validateObject("checkRange", false, raw.rules[index].checkRange, new api_validatorhelper_1.ValidatorHelper()
                    .validateObject("to", false, raw.rules[index].checkRange.to, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, raw.rules[index].checkRange.to.days)
                    .validateObjectInstance("hours", true, raw.rules[index].checkRange.to.hours)
                    .validateObjectInstance("minutes", true, raw.rules[index].checkRange.to.minutes))
                    .validateObject("from", false, raw.rules[index].checkRange.from, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, raw.rules[index].checkRange.from.days)
                    .validateObjectInstance("hours", true, raw.rules[index].checkRange.from.hours)
                    .validateObjectInstance("minutes", true, raw.rules[index].checkRange.from.minutes)));
                validator.validateObject("membershipRange", false, raw.rules[index].membershipRange, new api_validatorhelper_1.ValidatorHelper()
                    .validateObject("to", false, raw.rules[index].membershipRange.to, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, raw.rules[index].membershipRange.to.days)
                    .validateObjectInstance("hours", true, raw.rules[index].membershipRange.to.hours)
                    .validateObjectInstance("minutes", true, raw.rules[index].membershipRange.to.minutes))
                    .validateObject("from", false, raw.rules[index].membershipRange.from, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, raw.rules[index].membershipRange.from.days)
                    .validateObjectInstance("hours", true, raw.rules[index].membershipRange.from.hours)
                    .validateObjectInstance("minutes", true, raw.rules[index].membershipRange.from.minutes)));
            }
            this.rules = raw.rules;
        }
    }
    RestPacsApiV1.AuthoShiftUpsertRuleRequest = AuthoShiftUpsertRuleRequest;
    class AutoShiftListRequest {
        constructor(raw, validator) {
            validator.validateObject("pagination", true, raw.pagination, new api_validatorhelper_1.ValidatorHelper().validateObjectInstance("skip", true, raw.pagination?.skip).validateObjectInstance("take", true, raw.pagination?.take));
            this.pagination = raw.pagination;
            validator.validateObject("filter", true, raw.filter, new api_validatorhelper_1.ValidatorHelper()
                .validateObjectInstance("startDateTime", true, raw.filter?.startDateTime)
                .validateObjectInstance("endDateTime", true, raw.filter?.endDateTime)
                .validateObjectInstance("regionIds", true, raw.filter?.regionIds)
                .validateObjectInstance("groupIds", true, raw.filter?.groupIds));
            this.filter = raw.filter;
        }
    }
    RestPacsApiV1.AutoShiftListRequest = AutoShiftListRequest;
    class ReCalculateRequest {
        constructor(raw, validator) {
            validator.validateDate("start", false, raw.start);
            this.start = raw.start;
            validator.validateDate("end", true, raw.end);
            this.end = raw.end;
        }
    }
    RestPacsApiV1.ReCalculateRequest = ReCalculateRequest;
    class UpsertWorkPlanShiftImportRequest {
        constructor(raw, validator) {
            validator.validateUUID("id", true, raw.id);
            validator.validateString({ field: "name", optional: false, input: raw.name });
            validator.validateUUID("defaultWorkPlanId", true, raw.defaultWorkPlanId);
            raw.workPlanImportRules.forEach((rule) => {
                validator.validateUUID("id", true, rule.id);
                validator.validateUUID("workPlanId", false, rule.workPlanId);
                validator.validateString({ field: "keyword", optional: true, input: rule.keyword });
                validator.validateUUID("shiftTemplateId", true, rule.shiftTemplateId);
                validator.validateObject("membershipRange", false, rule.membershipRange, new api_validatorhelper_1.ValidatorHelper()
                    .validateObject("to", false, rule.membershipRange.to, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, rule.membershipRange.to.days)
                    .validateObjectInstance("hours", true, rule.membershipRange.to.hours)
                    .validateObjectInstance("minutes", true, rule.membershipRange.to.minutes))
                    .validateObject("from", false, rule.membershipRange.from, new api_validatorhelper_1.ValidatorHelper()
                    .validateObjectInstance("days", true, rule.membershipRange.from.days)
                    .validateObjectInstance("hours", true, rule.membershipRange.from.hours)
                    .validateObjectInstance("minutes", true, rule.membershipRange.from.minutes)));
            });
            if (raw.id)
                this.id = raw.id;
            this.name = raw.name;
            this.defaultWorkPlanId = raw.defaultWorkPlanId;
            this.workPlanImportRules = raw.workPlanImportRules;
            validator.finalize();
        }
    }
    RestPacsApiV1.UpsertWorkPlanShiftImportRequest = UpsertWorkPlanShiftImportRequest;
    class WorkPlanShiftImportFilter {
        constructor(raw, validator) {
            this.name = raw.name;
            this.pagination = {
                take: raw.pagination.take,
                skip: raw.pagination.skip,
            };
            validator.validateString({
                field: "name",
                optional: true,
                input: raw.name,
                maxLength: 255,
                minLength: 1,
            });
            validator.validateInteger({
                field: "take",
                optional: false,
                input: raw.pagination.take,
            });
            validator.validateInteger({
                field: "skip",
                optional: false,
                input: raw.pagination.skip,
            });
            validator.finalize();
        }
    }
    RestPacsApiV1.WorkPlanShiftImportFilter = WorkPlanShiftImportFilter;
    class WorkPlanShiftImportGetTemplate {
        constructor(raw, validator) {
            validator.validateUUID("userFilterId", false, raw.userFilterId);
            this.userFilterId = raw.userFilterId;
            validator.validateEnum("type", false, dal_constants_1.DalConstants.WorkPlanImportTemplateType, raw.type);
            this.type = raw.type;
            if (this.type === dal_constants_1.DalConstants.WorkPlanImportTemplateType.DateRange) {
                validator.validateDateRange(["startDateTime", "endDatetime"], {
                    start: raw.dateRange?.startDateTime,
                    end: raw.dateRange?.endDateTime,
                    startOptional: false,
                    endOptional: false,
                });
                this.dateRange = raw.dateRange;
            }
            else if (this.type === dal_constants_1.DalConstants.WorkPlanImportTemplateType.Cyclical) {
                validator.validateInteger({
                    field: "days",
                    optional: false,
                    input: raw.days,
                    min: 1,
                    max: 60,
                });
                this.days = raw.days;
            }
            validator.validateUUID("templateId", false, raw.templateId);
            this.templateId = raw.templateId;
            validator.finalize();
        }
    }
    RestPacsApiV1.WorkPlanShiftImportGetTemplate = WorkPlanShiftImportGetTemplate;
})(RestPacsApiV1 = exports.RestPacsApiV1 || (exports.RestPacsApiV1 = {}));
