"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiMobileClientController = void 0;
const express_1 = __importDefault(require("express"));
const api_util_1 = require("../../api.util");
const api_auth_1 = require("../../api.auth");
const predefined_permissions_1 = require("../../../dal/db/predefined/predefined.permissions");
const api_mobile_client_model_1 = require("./api.mobile-client.model");
const api_validatorhelper_1 = require("../../api.validatorhelper");
const dal_manager_1 = require("../../../dal/dal.manager");
const messageBroker_server_to_device_pub_1 = require("../../../messageBroker/messageBroker.server-to-device.pub");
const api_securityhelper_1 = require("../../api.securityhelper");
const api_error_1 = require("../../api.error");
const bluebird_1 = require("bluebird");
const messagebroker_models_1 = require("../../../messageBroker/messagebroker.models");
const app_enums_1 = require("../../../app.enums");
const business_log_1 = require("../../../business/business.log");
const dal_access_psql_log_1 = require("../../../dal/access/psql/dal.access.psql.log");
class ApiMobileClientController {
    constructor(app) {
        this.getMobileCreditApplicationSettings = async (req, res) => {
            let organizationId = (0, api_validatorhelper_1.validateUUID)(req.organizationId, "organizationId");
            let mobileCreditSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationMobileCreditSettings(organizationId);
            res.json(mobileCreditSettings);
        };
        this.getAdminDevices = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.accessMobileClient.getDevicesOfOrganization(req.organizationId);
            let result = {
                items: dbResult,
            };
            res.json(result);
        };
        this.getAccessRights = async (req, res) => {
            let requestBody = new api_mobile_client_model_1.MobileClientApiModelsV1.AskMobileAccessRightsWithPagination(req.body, new api_validatorhelper_1.ValidatorHelper());
            let dbResult = await dal_manager_1.dbManager.accessMobileClient.getAccessRights(req.organizationId, req.auth_token.userId, requestBody.pagination, requestBody.filterAuthenticationFactors);
            let result = {
                items: dbResult.items,
                pagination: dbResult.pagination,
            };
            res.json(result);
        };
        this.remoteAccess = async (req, res) => {
            let requestBody = new api_mobile_client_model_1.MobileClientApiModelsV1.RemoteAccessCommand(req.body, new api_validatorhelper_1.ValidatorHelper());
            let remoteAccessCommandStructure = await dal_manager_1.dbManager.accessDevice.getRemoteAccessCommandStructure(req.organizationId, req.auth_token.userId, requestBody.accessControlPointId, requestBody.qr);
            if (!remoteAccessCommandStructure) {
                (0, api_securityhelper_1.throwAuthError)();
            }
            if (remoteAccessCommandStructure.deviceId) {
                const result = new Promise((resolve, reject) => {
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(remoteAccessCommandStructure.deviceId, {
                        e: messagebroker_models_1.ServerToDeviceRPCMessageType.RemoteAccess,
                        p: {
                            a: requestBody.accessControlPointId,
                            d: requestBody.direction,
                            o: req.auth_token.userId,
                            iq: requestBody.qr ? true : false,
                        },
                    }, 30000, (err, success) => {
                        if (err) {
                            res.status(404);
                            res.json((0, api_error_1.generateTimeoutApiError)(new bluebird_1.TimeoutError(err.message)));
                        }
                        else {
                            if (success.s) {
                                res.json({ result: "success" });
                                resolve(success);
                            }
                            else {
                                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR);
                                res.json((0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, "Access Point Could Not Be Opened", null, true));
                            }
                        }
                    });
                });
                await result
                    .then(async (value) => {
                    if (requestBody.qr) {
                        await (0, dal_access_psql_log_1.assignAutoShift)(req.organizationId, {
                            credentialOwnerUserId: value.o,
                            generationTime: value.u,
                            redisCache: dal_manager_1.dbManager.accessRedisCache,
                            logId: value.id,
                        });
                    }
                })
                    .catch((err) => {
                    console.log(err);
                });
            }
            else {
                if (await (0, business_log_1.insertVirtualACPAccessLog)(req.organizationId, {
                    remoteAccessCommandStructure,
                    direction: requestBody.direction ? requestBody.direction : app_enums_1.enums.libEnumsV2.AccessDirection.All,
                    userId: req.auth_token.userId,
                    qr: requestBody.qr,
                })) {
                    res.json({ result: "success" });
                }
                else {
                    res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR);
                    res.json((0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, "Access Point Could Not Be Opened", null, true));
                }
            }
        };
        this.getServerDetailes = (req, res) => {
            res.json({
                version: "0.9.1",
                message: "3rd party device adapter end point is working!",
            });
        };
        this._router = express_1.default.Router();
        this._router.use(express_1.default.json({ limit: "100mb" }));
        this._router.get("/", this.getServerDetailes);
        this._router.post("/:oid/accessrights", api_auth_1.jwtAuthorize, api_auth_1.validatePermissions, (0, api_util_1.globalRouteHandler)(this.getAccessRights));
        this._router.post("/:oid/remoteaccess", api_auth_1.jwtAuthorize, api_auth_1.validatePermissions, (0, api_util_1.globalRouteHandler)(this.remoteAccess));
        this._router.get("/:oid/mobilecreditsettings", api_auth_1.validatePermissions, (0, api_util_1.globalRouteHandler)(this.getMobileCreditApplicationSettings));
        this._router.get("/:oid/admindevices", api_auth_1.jwtAuthorize, api_auth_1.validatePermissions.bind({
            permissionsRequirements: {
                organizationWide: [predefined_permissions_1.Permissions.system.getWrite()],
            },
        }), (0, api_util_1.globalRouteHandler)(this.getAdminDevices));
        this._router.use(api_auth_1.handleException);
        app.use("/m", this._router);
    }
}
exports.ApiMobileClientController = ApiMobileClientController;
