"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiLicenceControllverV1 = void 0;
const express_1 = __importDefault(require("express"));
const dal_manager_1 = require("../../../dal/dal.manager");
const api_auth_1 = require("../../api.auth");
const api_util_1 = require("../../api.util");
const api_validatorhelper_1 = require("../../api.validatorhelper");
const terminalTokenAuthorize = async function (req, res, next) {
    let authHeader = req.headers.authorization;
    if (!authHeader) {
        return res.status(401).json({ message: "Invalid authorization" });
    }
    let token = authHeader;
    try {
        let device = await dal_manager_1.dbManager.accessDevice.getTerminalByToken(token);
        req.params.organizationId = device.organizationId;
        req.params.terminalId = device.id;
    }
    catch (error) {
        return res.status(401).json({ message: "Authorization token is not found or expired!" });
    }
    next();
};
const adapterTokenAuthorize = async function (req, res, next) {
    let authHeader = req.headers.authorization;
    if (!authHeader) {
        return res.status(401).json({ message: "Invalid authorization" });
    }
    let token = authHeader;
    try {
        let adapter = await dal_manager_1.dbManager.accessDevice.getAdapterByToken(token);
        req.params.organizationId = adapter.organizationId;
        req.params.adapterId = adapter.id;
    }
    catch (error) {
        return res.status(401).json({ message: "Authorization token is not found or expired!" });
    }
    next();
};
class ApiLicenceControllverV1 {
    constructor(app) {
        this.getTerminalToken = async (req, res) => {
            let terminalId = (0, api_validatorhelper_1.validateUUID)(req.params.tid, "terminalId");
            try {
                let challenge = await dal_manager_1.dbManager.accessDevice.generateTerminalChallenge(terminalId);
                res.json({ challenge: challenge });
            }
            catch (error) {
                return res.status(401).json({ message: "Unable to generate challenge token" });
            }
        };
        this.getAdapterToken = async (req, res) => {
            let adapterId = (0, api_validatorhelper_1.validateUUID)(req.params.aid, "adapterId");
            try {
                let challenge = await dal_manager_1.dbManager.accessDevice.generateAdapterChallenge(adapterId);
                res.json({ challenge: challenge });
            }
            catch (error) {
                return res.status(401).json({ message: "Unable to generate challenge token" });
            }
        };
        this.requestCertificatesForAdapter = async (req, res) => {
            let adapterId = (0, api_validatorhelper_1.validateUUID)(req.params.aid, "adapterId");
            let date = new Date(parseInt(req.params.date.substr(0, 4)), parseInt(req.params.date.substr(4, 2)));
            let certFiles = await dal_manager_1.dbManager.accessDevice.getAdapterCertFiles(req.organizationId, adapterId, date);
            (0, api_auth_1.sendZippedJsonResponse)(certFiles, res);
        };
        this.requestCertificatesForTerminal = async (req, res) => {
            let terminalId = (0, api_validatorhelper_1.validateUUID)(req.params.tid, "terminalId");
            let date = new Date(parseInt(req.params.date.substr(0, 4)), parseInt(req.params.date.substr(4, 2)));
            let certFiles = await dal_manager_1.dbManager.accessDevice.getTerminalCertFiles(req.organizationId, terminalId, date);
            (0, api_auth_1.sendZippedJsonResponse)(certFiles, res);
        };
        this._router = express_1.default.Router();
        this._router.get("/terminal/:tid/token", (0, api_util_1.globalRouteHandler)(this.getTerminalToken));
        this._router.get("/adapter/:aid/token", (0, api_util_1.globalRouteHandler)(this.getAdapterToken));
        this._router.get("/terminal/:tid/:yyyymm", terminalTokenAuthorize, (0, api_util_1.globalRouteHandler)(this.requestCertificatesForTerminal));
        this._router.get("/adapter/:aid/:yyyymm", adapterTokenAuthorize, (0, api_util_1.globalRouteHandler)(this.requestCertificatesForAdapter));
        app.use("/l/v1", express_1.default.json({ limit: "100mb" }), this._router);
    }
}
exports.ApiLicenceControllverV1 = ApiLicenceControllverV1;
