"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiControlPanelControllerV2 = void 0;
const express_1 = __importDefault(require("express"));
const fs_1 = __importDefault(require("fs"));
const app_logs_1 = require("../../../app.logs");
const business_device_1 = require("../../../business/business.device");
const dal_manager_1 = require("../../../dal/dal.manager");
const messagebroker_models_1 = require("../../../messageBroker/messagebroker.models");
const messageBroker_server_to_device_pub_1 = require("../../../messageBroker/messageBroker.server-to-device.pub");
const api_auth_1 = require("../../api.auth");
const api_error_1 = require("../../api.error");
const api_models_common_1 = require("../../api.models.common");
const api_util_1 = require("../../api.util");
const api_validatorhelper_1 = require("../../api.validatorhelper");
class ApiControlPanelControllerV2 {
    constructor(app) {
        this.askDeviceData = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            let request = new api_models_common_1.ApiModelsCommon.AskDeviceDataZip(req.body, validator);
            validator.finalize();
            app_logs_1.logger.info("device request " + req.deviceId);
            let dbResult = await dal_manager_1.dbManager.accessDevice.getDataWithUsersForControlPanelMSeries(req.deviceId, req.organizationId, request.configurationAndSettings, request.users, request.accessNotificationEvents);
            app_logs_1.logger.info(" db result " + req.deviceId);
            let result = {
                systemDateISO: new Date().toISOString(),
                configurationAndSettings: dbResult.configurationAndSettings,
                users: dbResult.users,
                lastSyncISO: dbResult.lastSyncISO,
                accessNotificationEvents: dbResult.accessNotificationEvents,
            };
            app_logs_1.logger.info(" zip " + req.deviceId);
            (0, api_auth_1.sendZippedJsonResponse)(result, res);
        };
        this.askUsersWithPagination = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            let request = new api_models_common_1.ApiModelsCommon.AskUsersWithPaginationRequest(req.body, validator);
            validator.finalize();
            let userIds = JSON.parse(await dal_manager_1.dbManager.accessRedisCache.getValue(request.sessionId));
            let targetUserIds = userIds.slice(request.skip, request.take + request.skip);
            let users = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return dal_manager_1.dbManager.accessDevice.getUsersOfTerminalPg(req.deviceId, targetUserIds, req.organizationId, trx);
            });
            (0, api_auth_1.sendZippedJsonResponse)({ users: users }, res);
        };
        this.askUsersWithPaginationStart = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateUUID("sessionId", false, req.body.sessionId);
            validator.finalize();
            let sessionId = req.body.sessionId;
            let userIds = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const userIdsOfTerminal = await dal_manager_1.dbManager.accessDevice.getUserIdsOfTerminal(req.organizationId, req.deviceId, trx);
                await dal_manager_1.dbManager.accessDevice.removeAllChangesForTerminal(req.organizationId, req.deviceId, trx);
                return userIdsOfTerminal;
            });
            await dal_manager_1.dbManager.accessRedisCache.setExpireValue(sessionId, JSON.stringify(userIds), 60 * 60);
            res.json({
                totalUserCount: userIds.length,
            });
        };
        this.askUsersWithPaginationEnd = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateUUID("sessionId", false, req.body.sessionId);
            validator.finalize();
            let sessionId = req.body.sessionId;
            await dal_manager_1.dbManager.accessRedisCache.delValue(sessionId);
            res.json();
        };
        this.startAskChangeSession = async (req, res) => {
            let sessionId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "sessionId");
            res.json({
                itemCount: await dal_manager_1.dbManager.accessDevice.startGetChangesSession(req.deviceId, sessionId, req.organizationId),
                systemDateISO: new Date().toISOString(),
            });
        };
        this.askChangesWhole = async (req, res) => {
            let sessionId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "sessionId");
            let items = await dal_manager_1.dbManager.accessDevice.getWholeChanges(req.deviceId, sessionId, req.organizationId);
            (0, api_auth_1.sendZippedJsonResponse)(items, res);
        };
        this.askChangesWithPagination = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateUUID("sessionId", false, req.params.sid);
            let request = new api_models_common_1.ApiModelsCommon.PaginationRequest(req.body, 1000, validator);
            validator.finalize();
            let sessionId = req.params.sid;
            let dbResult = await dal_manager_1.dbManager.accessDevice.getChanges(req.deviceId, sessionId, req.organizationId, request);
            let result = {
                items: dbResult.items,
                pagination: dbResult.pagination,
            };
            (0, api_auth_1.sendZippedJsonResponse)(result, res);
        };
        this.askChangeSessionCompleted = async (req, res) => {
            let sessionId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "sessionId");
            let remainingItemsCount = await dal_manager_1.dbManager.accessDevice.completeGetChangesSession(req.deviceId, sessionId, req.organizationId);
            res.json({
                remainingItemsCount: remainingItemsCount,
                systemDateISO: new Date().toISOString(),
            });
        };
        this.uploadOfflineLogs = async (req, res) => {
            let request = req.body;
            app_logs_1.logger.info("[ADAPTER] offline logs for deviceId:" + req.deviceId);
            let d1 = new Date();
            try {
                await dal_manager_1.dbManager.accessLog.checkUploadLogIdExits(req.organizationId, request.uploadId);
            }
            catch (error) {
                throw (0, api_error_1.generateConflictError)({ message: "Upload Id already exists", details: request.uploadId });
            }
            let count = 0;
            let organizationId = req.organizationId;
            if (request.accessLogs && request.accessLogs.length > 0) {
                await dal_manager_1.dbManager.accessDevice.uploadDeviceAccessLogs(organizationId, request.accessLogs);
                count += request.accessLogs.length;
            }
            if (request.statusSensorLogs && request.statusSensorLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadStatusSensorLogs(organizationId, request.statusSensorLogs);
                count += request.statusSensorLogs.length;
            }
            if (request.tamperSwitchLogs && request.tamperSwitchLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadTamperSwitchLogs(organizationId, request.tamperSwitchLogs);
                count += request.tamperSwitchLogs.length;
            }
            if (request.counterLogs && request.counterLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadCounterLogs({ organizationId, logs: request.counterLogs });
                count += request.counterLogs.length;
            }
            if (request.systemStatusLogs && request.systemStatusLogs.length > 0) {
                let deviceHealth = await (0, business_device_1.uploadDeviceSystemStatusLogs)(organizationId, req.deviceId, request.systemStatusLogs);
                if (!deviceHealth) {
                    throw (0, api_error_1.generateBadRequestApiError)({ message: "System status health update" });
                }
                count += request.systemStatusLogs.length;
            }
            await dal_manager_1.dbManager.accessLog.addUploadLogId(organizationId, req.deviceId, request.uploadId, count);
            res.json({
                uploadId: request.uploadId,
            });
            let deviceIds = await dal_manager_1.dbManager.accessRegion.getOrganizationRegionRelatedDeviceIds(organizationId);
            for (let deviceId of deviceIds) {
                messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(deviceId, {
                    e: messagebroker_models_1.ServerToDeviceMessageType.GetChangesRequired,
                });
            }
            let d2 = new Date();
            app_logs_1.logger.info("[ADAPTER] offline logs for:" + req.deviceId + ":" + (d2.getTime() - d1.getTime()));
        };
        this.updateControlPanelSettingsAndConfiguration = async (req, res) => {
            let request = req.body;
            try {
                let dbResult = await dal_manager_1.dbManager.accessDevice.updateControlPanelMSeriesDeviceConfigurationAndSettings(req.deviceId, req.organizationId, request);
                res.json(dbResult);
            }
            catch (error) {
                res.status(400).json({ error: error });
            }
        };
        this.failSnapshot = async (req, res) => {
            let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "snapshotId");
            try {
                let dbResult = await dal_manager_1.dbManager.accessLog.uploadSnapshot(req.organizationId, req.deviceId, snapshotId, false, null);
                if (!dbResult)
                    throw (0, api_error_1.generateNotFoundApiError)({ message: "Unable to fail snapshot" });
            }
            catch (error) {
                throw (0, api_error_1.generateConflictError)({ message: "Snapshot Id already exists", details: snapshotId });
            }
            res.json();
        };
        this.uploadSnapshot = async (req, res) => {
            let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "snapshotId");
            let snapshot = req.body.snapshot.data;
            let actionUtc = req.body.actionUtc;
            try {
                let dbResult = await dal_manager_1.dbManager.accessLog.uploadSnapshot(req.organizationId, req.deviceId, snapshotId, true, actionUtc, snapshot);
                if (!dbResult)
                    throw (0, api_error_1.generateNotFoundApiError)({ message: "Unable to upload snapshot" });
            }
            catch (error) {
                throw (0, api_error_1.generateConflictError)({ message: "Snapshot Id already exists", details: snapshotId });
            }
            res.json();
        };
        this.listUserGroups = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsForControlPanel(req.organizationId);
            res.json(dbResult);
        };
        this.listAccessRuleHistory = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return dal_manager_1.dbManager.accessLog.listAccessRuleHistory(req.organizationId, trx);
            });
            res.json(dbResult);
        };
        this.listUserRegionTicketUnitsInfo = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.accessSocial.listUserRegionTicketsForControlPanel(req.organizationId);
            res.json(dbResult);
        };
        this.getServerDetailes = (req, res) => {
            res.json({
                version: "2.0.0",
                message: "control panel api is working!",
            });
        };
        this.downloadUpdateFile = async (req, res) => {
            let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(req.organizationId, req.deviceId);
            let updateFile = await dal_manager_1.dbManager.accessDevice.getUpdateFile(req.organizationId, deviceBasic.brand, deviceBasic.model);
            if (updateFile) {
                if (fs_1.default.existsSync(updateFile.updateFilePath)) {
                    res.download(updateFile.updateFilePath);
                }
                else {
                    res.status(400).send("File does not exists in the specified location");
                }
            }
            else {
                res.status(404).send();
            }
        };
        this._router = express_1.default.Router();
        this._router.get("/", this.getServerDetailes);
        this._router.post("/askdevicedata", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askDeviceData));
        this._router.post("/askuserswithpagination", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askUsersWithPagination));
        this._router.post("/askuserswithpaginationstart", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askUsersWithPaginationStart));
        this._router.post("/askuserswithpaginationend", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askUsersWithPaginationEnd));
        this._router.get("/askchanges/:sid", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askChangesWhole));
        this._router.post("/askchanges/:sid", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askChangesWithPagination));
        this._router.get("/askchanges/:sid/new", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.startAskChangeSession));
        this._router.get("/askchanges/:sid/completed", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.askChangeSessionCompleted));
        this._router.post("/uploadofflinelogs", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.uploadOfflineLogs));
        this._router.post("/settingsandconfigurations", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.updateControlPanelSettingsAndConfiguration));
        this._router.post("/snapshot/:sid", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.uploadSnapshot));
        this._router.get("/snapshot/:sid/fail", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.failSnapshot));
        this._router.get("/usergroups", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.listUserGroups));
        this._router.get("/accessrulehistory", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.listAccessRuleHistory));
        this._router.get("/userregionticketunits", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.listUserRegionTicketUnitsInfo));
        this._router.get("/updatefile", api_auth_1.controlPanelAuthorize, (0, api_util_1.globalRouteHandler)(this.downloadUpdateFile));
        app.use("/cp/v2", express_1.default.json({ limit: "100mb" }), this._router);
    }
}
exports.ApiControlPanelControllerV2 = ApiControlPanelControllerV2;
