"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.validatePermissions = exports.handleException = exports.jwtAuthorize = exports.terminalAuthorize = exports.armonV10Authorize = exports.controlPanelAuthorize = exports.adapterAuthorize = exports.sendZippedJsonResponse = void 0;
const jwt = require("jsonwebtoken");
const util_1 = require("util");
const zlib_1 = __importDefault(require("zlib"));
const app_auth_1 = require("../app.auth");
const app_config_1 = require("../app.config");
const app_enums_1 = require("../app.enums");
const dal_manager_1 = require("../dal/dal.manager");
const app_constants_1 = require("../app.constants");
const tempArmon1 = [
    {
        deviceId: "656663cf-8ba7-41b6-9213-5483855bb599",
        amqpUsername: "a1t1",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "efcf6aa8-0cd3-4ef1-89da-0094ecb44ebc",
        amqpUsername: "a1t2",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "11b9a666-a7c8-457c-a19b-4d60641716b0",
        amqpUsername: "a1t3",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "de37550b-af9d-4353-9af4-73a83c11c3fd",
        amqpUsername: "a1t4",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "74a4158d-6a46-4055-8713-49cc30b64ead",
        amqpUsername: "a1t5",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "0b7a3b2e-95ca-489e-9b7e-fb0b5c0e4f2b",
        amqpUsername: "a1t6",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "422f71f5-eed0-4932-aaf0-42ba17fa93e3",
        amqpUsername: "a1t7",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "1201653a-e081-4363-a963-ad49b0830c18",
        amqpUsername: "a1t8",
        amqpPassword: "Aa2233*Pi!",
    },
    {
        deviceId: "5d1e41f6-3b22-4377-9d32-20f92c392f58",
        amqpUsername: "a1t9",
        amqpPassword: "Aa2233*Pi!",
    },
];
const sendZippedJsonResponse = (jsonObject, res) => {
    let bf = Buffer.from(JSON.stringify(jsonObject));
    let zippedBf = zlib_1.default.gzipSync(bf);
    res.setHeader("Content-Type", "application/json");
    res.setHeader("x-decompressed-content-length", bf.length);
    res.setHeader("Content-Length", zippedBf.length);
    res.setHeader("Content-Encoding", "gzip");
    res.send(zippedBf);
};
exports.sendZippedJsonResponse = sendZippedJsonResponse;
const adapterAuthorize = function (req, res, next) {
    let tls = req.connection;
    if (!tls.authorized) {
        res.status(401).json({ statusCode: 401, code: 2097, message: "Unauthorized client certificate!" });
        return;
    }
    let cert = tls.getPeerCertificate();
    req.adapterId = cert.subject.CN;
    req.organizationId = cert.subject.O;
    next();
};
exports.adapterAuthorize = adapterAuthorize;
const controlPanelAuthorize = function (req, res, next) {
    let tls = req.connection;
    if (!tls.authorized) {
        res.status(401).json({ statusCode: 401, code: 2097, message: "Unauthorized client certificate!" });
        return;
    }
    let cert = tls.getPeerCertificate();
    req.deviceId = cert.subject.CN;
    req.organizationId = cert.subject.O;
    next();
};
exports.controlPanelAuthorize = controlPanelAuthorize;
const armonV10Authorize = function (req, res, next) {
    let tls = req.connection;
    if (!tls.authorized) {
        res.status(401).json({ statusCode: 401, code: 2097, message: "Unauthorized client certificate!" });
        return;
    }
    let cert = tls.getPeerCertificate();
    req.deviceId = cert.subject.CN;
    req.organizationId = cert.subject.O;
    next();
};
exports.armonV10Authorize = armonV10Authorize;
const terminalAuthorize = function (req, res, next) {
    let tls = req.connection;
    if (!tls.authorized) {
        res.status(401).json({ statusCode: 401, code: 2097, message: "Unauthorized client certificate!" });
        return;
    }
    let cert = tls.getPeerCertificate();
    req.deviceId = cert.subject.CN;
    req.organizationId = cert.subject.O;
    next();
};
exports.terminalAuthorize = terminalAuthorize;
const jwtAuthorize = function (req, res, next) {
    let authHeader = req.header(app_constants_1.ArmonHeaders.Authorization);
    if (!authHeader) {
        res.status(401).json({ code: 2056, message: "Authorization required!" });
    }
    else if (authHeader.indexOf("Bearer ") === -1) {
        res.status(401).json({ code: 2057, message: "Invalid authorization header" });
    }
    else {
        let bearerToken = authHeader.substr(7);
        try {
            let decodedToken = jwt.verify(bearerToken, app_config_1.appConfig.jwtSecret);
            dal_manager_1.dbManager.accessOAuth.hasRevokedToken(decodedToken.o[0].i, decodedToken.i).then((hasRevokedToken) => {
                if (hasRevokedToken) {
                    res.status(401).json({ statusCode: 401, code: 2060, message: "The token is revoked! Login again!" });
                }
                else {
                    req.raw_auth_token = bearerToken;
                    req.auth_token = new app_auth_1.UserJwtPayload(decodedToken);
                    next();
                }
            });
        }
        catch (err) {
            if (err && err.name === "TokenExpiredError") {
                res.status(401).json({ statusCode: 401, code: 2060, message: "The token is expired! Login again!" });
            }
            else {
                res.status(401).json({ code: 2058, message: err });
            }
        }
    }
};
exports.jwtAuthorize = jwtAuthorize;
const handleException = function (err, req, res, next) {
    res.status(err.statusCode || 500).json(err.body || { message: err.message } || err || { message: "something blew up and the err object was undefined" });
};
exports.handleException = handleException;
const uuidPattern = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-4][0-9a-f]{3}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
const validatePermissions = function (req, res, next) {
    let reqs = this && this.permissionsRequirements ? this.permissionsRequirements : undefined;
    let organizationId = req.params.oid;
    if (!organizationId) {
        return res.status(400).json({
            message: "Request validation error",
            validationerrors: [
                {
                    code: app_enums_1.enums.ValidationErrorCode.REQUIRED,
                    fields: ["organization id"],
                    message: "Required field",
                },
            ],
        });
    }
    else if (!uuidPattern.test(organizationId)) {
        return res.status(400).json({
            message: "Request validation error",
            validationerrors: [
                {
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: ["organization id"],
                    message: "Invalid UUIDV4.",
                    url: "https://tools.ietf.org/html/rfc4122",
                },
            ],
        });
    }
    req.organizationId = organizationId;
    if (!reqs) {
        return next();
    }
    let results = new ArmonPermissionRequirementVerifier(req.auth_token, organizationId, reqs).getResults();
    if (results.invalid) {
        res.status(401).json({
            code: app_enums_1.enums.ArmonApiErrorCode.UNAUTHORIZED_API_ACCESS,
            message: "Your permissions are not enough!",
        });
        return;
    }
    req.armonPermissionVerificationResults = results;
    next();
};
exports.validatePermissions = validatePermissions;
class ArmonPermissionRequirementVerifier {
    constructor(auth_token, organizationId, requirements) {
        this._result = {
            validForOrganizationWide: undefined,
            validForUnitIds: undefined,
            invalid: false,
        };
        this._auth_token = auth_token;
        if (requirements.organizationWide) {
            this.handleForOrganization(organizationId, requirements.organizationWide);
        }
        if (requirements.unit) {
            this.handleForUnits(organizationId, requirements.unit);
        }
        if (this._result.validForOrganizationWide === false && this._result.validForUnitIds && this._result.validForUnitIds.length === 0) {
            this._result.invalid = true;
        }
        if (requirements.specificUnitId) {
            if (!this._result.validForUnitIds) {
                this._result.invalid = true;
            }
            else {
                this._result.invalid = this._result.validForUnitIds.indexOf(requirements.specificUnitId) === -1;
            }
        }
    }
    handleForOrganization(organizationId, permissions) {
        if ((0, util_1.isArray)(permissions)) {
            for (let p of permissions) {
                if ((0, util_1.isArray)(p)) {
                    this.handleForOrganization(organizationId, p);
                }
                else {
                    this.orOrganization(organizationId, p.split(","));
                }
            }
        }
        else {
            this.andOrganization(organizationId, permissions.split(","));
        }
    }
    handleForUnits(organizationId, permissions) {
        if ((0, util_1.isArray)(permissions)) {
            for (let p of permissions) {
                if ((0, util_1.isArray)(p)) {
                    this.handleForUnits(organizationId, p);
                }
                else {
                    this.orCollectUnits(organizationId, p.split(","));
                }
            }
        }
        else {
            this.andCollectUnits(organizationId, permissions.split(","));
        }
    }
    andOrganization(organizationId, permissions) {
        let r = this._auth_token.isPermittedForOrganization(organizationId, permissions);
        this._result.validForOrganizationWide = this._result.validForOrganizationWide === undefined ? r : this._result.validForOrganizationWide && r;
        return this;
    }
    orOrganization(organizationId, permissions) {
        let r = this._auth_token.isPermittedForOrganization(organizationId, permissions);
        this._result.validForOrganizationWide = this._result.validForOrganizationWide === undefined ? r : this._result.validForOrganizationWide || r;
        return this;
    }
    andCollectUnits(organizationId, permissions) {
        let u = this._auth_token.getPermittedUnitIdsFor(organizationId, permissions);
        this._result.validForUnitIds = this._result.validForUnitIds === undefined ? u : u.filter((ui) => this._result.validForUnitIds.indexOf(ui) !== -1);
        return this;
    }
    orCollectUnits(organizationId, permissions) {
        let u = this._auth_token.getPermittedUnitIdsFor(organizationId, permissions);
        this._result.validForUnitIds =
            this._result.validForUnitIds === undefined ? u : this._result.validForUnitIds.concat(u.filter((ui) => this._result.validForUnitIds.indexOf(ui) === -1));
        return this;
    }
    getResults() {
        return this._result;
    }
}
