"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpServerToReportSub = void 0;
const fs_1 = __importDefault(require("fs"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const worker_threads_1 = require("worker_threads");
const app_config_1 = require("../app.config");
const app_constants_1 = require("../app.constants");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const business_notification_1 = require("../business/business.notification");
const report_excel_access_control_points_1 = require("../business/report/excel/report.excel.access-control-points");
const report_excel_access_logs_1 = require("../business/report/excel/report.excel.access-logs");
const report_excel_acp_users_list_1 = require("../business/report/excel/report.excel.acp-users-list");
const report_excel_audit_access_control_points_1 = require("../business/report/excel/report.excel.audit-access-control-points");
const report_excel_audit_identity_1 = require("../business/report/excel/report.excel.audit-identity");
const report_excel_audit_terminal_1 = require("../business/report/excel/report.excel.audit-terminal");
const report_excel_credential_1 = require("../business/report/excel/report.excel.credential");
const report_excel_daily_first_entrance_last_exit_1 = require("../business/report/excel/report.excel.daily-first-entrance-last-exit");
const report_excel_daily_first_entrance_last_exit_summary_1 = require("../business/report/excel/report.excel.daily-first-entrance-last-exit-summary");
const report_excel_daily_monthly_report_1 = require("../business/report/excel/report.excel.daily-monthly-report");
const report_excel_daily_working_details_date_1 = require("../business/report/excel/report.excel.daily-working-details-date");
const report_excel_employee_instant_attendance_report_1 = require("../business/report/excel/report.excel.employee-instant-attendance-report");
const report_excel_employee_permission_usage_report_1 = require("../business/report/excel/report.excel.employee-permission-usage-report");
const report_excel_employees_permission_usage_report_1 = require("../business/report/excel/report.excel.employees-permission-usage-report");
const report_excel_feedbacks_1 = require("../business/report/excel/report.excel.feedbacks");
const report_excel_first_and_last_access_logs_1 = require("../business/report/excel/report.excel.first-and-last-access-logs");
const report_excel_first_and_last_access_logs_for_single_user_1 = require("../business/report/excel/report.excel.first-and-last-access-logs-for-single-user");
const report_excel_forbidden_users_1 = require("../business/report/excel/report.excel.forbidden-users");
const report_excel_identity_1 = require("../business/report/excel/report.excel.identity");
const report_excel_montly_work_summary_extra_work_1 = require("../business/report/excel/report.excel.montly-work-summary-extra-work");
const report_excel_pacs_planning_calendar_1 = require("../business/report/excel/report.excel.pacs-planning-calendar");
const report_excel_permission_management_1 = require("../business/report/excel/report.excel.permission-management");
const report_excel_region_state_1 = require("../business/report/excel/report.excel.region-state");
const report_excel_region_ticket_transaction_1 = require("../business/report/excel/report.excel.region-ticket-transaction");
const report_excel_region_ticket_user_units_1 = require("../business/report/excel/report.excel.region-ticket-user-units");
const report_excel_sensor_events_1 = require("../business/report/excel/report.excel.sensor-events");
const report_excel_terminal_list_1 = require("../business/report/excel/report.excel.terminal-list");
const report_excel_terminated_visits_1 = require("../business/report/excel/report.excel.terminated-visits");
const report_excel_user_ticket_usage_1 = require("../business/report/excel/report.excel.user-ticket-usage");
const report_excel_visitor_1 = require("../business/report/excel/report.excel.visitor");
const report_excel_visitor_count_1 = require("../business/report/excel/report.excel.visitor-count");
const report_excel_analysis_visitors_1 = require("../business/report/excel/report.excel.analysis-visitors");
const report_excel_arvento_access_logs_1 = require("../business/report/excel/report.excel.arvento-access-logs");
const report_excel_work_summary_employee_1 = require("../business/report/excel/report.excel.work-summary-employee");
const report_excel_duration_in_region_overview_1 = require("../business/report/excel/report.excel.duration-in-region-overview");
const generator_1 = require("../business/report/generator");
const report_pdf_access_control_points_1 = require("../business/report/pdf/report.pdf.access-control-points");
const report_pdf_access_logs_1 = require("../business/report/pdf/report.pdf.access-logs");
const report_pdf_first_in_last_out_1 = require("../business/report/pdf/report.pdf.first-in-last-out");
const report_pdf_identity_1 = require("../business/report/pdf/report.pdf.identity");
const report_pdf_permission_management_1 = require("../business/report/pdf/report.pdf.permission-management");
const report_pdf_region_state_1 = require("../business/report/pdf/report.pdf.region-state");
const report_pdf_sensor_events_1 = require("../business/report/pdf/report.pdf.sensor-events");
const report_pdf_work_summary_1 = require("../business/report/pdf/report.pdf.work-summary");
const dal_manager_1 = require("../dal/dal.manager");
const restapi_1 = require("../lib/es/models/restapi");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
const report_excel_entry_by_unit_1 = require("../business/report/excel/report.excel.entry-by-unit");
const generatorCreatorList = {
    [app_enums_1.enums.ReportCode.AccessLogs]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_access_logs_1.ExcelReportAccessLogs(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_access_logs_1.PdfReportAccessLogs(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.CustomReport]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new generator_1.CustomReportGenerator(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new generator_1.CustomReportGenerator(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return new generator_1.CustomReportGenerator(request, locale);
        },
    },
    [app_enums_1.enums.ReportCode.PacsFirstInLastOut]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_daily_first_entrance_last_exit_1.DailyFirstEntranceLastExitReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_first_in_last_out_1.PdfReportFirstInLastOut(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.RegionState]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_region_state_1.RegionStateExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_region_state_1.PdfReportRegionState(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.SensorEvents]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_sensor_events_1.SensorEventsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_sensor_events_1.PdfSensorEvents(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.Credential]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_credential_1.CredentialsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.Identity]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_identity_1.UserList(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_identity_1.PdfIdentityList(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsDailyWorkingHours]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_daily_working_details_date_1.DailyWorkingDetailsAtDateReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_work_summary_1.PdfReportWorkSummaryV2(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsEmployeeWorkingHoursSummary]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_work_summary_employee_1.ExcelReportWorkSummaryOfEmployee(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: null,
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PermissionManagement]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_permission_management_1.PermissionManagementExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_permission_management_1.PdfReportPermissionManagement(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.VisitorReport]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_visitor_1.VisitorsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.TerminatedVisits]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_terminated_visits_1.VisitRecordsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.RegionTicketUserUnits]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_region_ticket_user_units_1.RegionTicketUserUnitsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.RegionTicketTransactions]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_region_ticket_transaction_1.RegionTicketTransactionExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.UserTicketUsage]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_user_ticket_usage_1.UserTicketUsageExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.EmployeePermissionUsage]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_employee_permission_usage_report_1.EmployeePermissionUsageReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.FirstAndLastAccessLogs]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_first_and_last_access_logs_1.FirstAndLastAccessLogsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.FirstAndLastAccessLogsForSingleUser]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_first_and_last_access_logs_for_single_user_1.FirstAndLastAccessLogsForSingleUserExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.EntryByUnit]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_entry_by_unit_1.EntryByUnitReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.ForbiddenUsers]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_forbidden_users_1.ForbiddenUsersExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsMonthlyWorkingHoursSummary]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_daily_monthly_report_1.PacsMontlyWorkSummaryExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.TerminalList]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_terminal_list_1.TerminalList(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.VisitorCountReport]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_visitor_count_1.VisitorCountReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AnalysisVisitorsReport]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_analysis_visitors_1.AnalysisVisitorsReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.ArventoAccessLogsReport]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_arvento_access_logs_1.ArventoAccessLogsReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.InstantAttendanceStatus]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_employee_instant_attendance_report_1.EmployeeInstantAttendanceReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsFirstInLastOutSummaryMultipleUsers]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_daily_first_entrance_last_exit_summary_1.DailyFirstEntranceLastExitSummaryReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AuditLogsIdentity]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_audit_identity_1.AuditIdentityLogsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AuditLogsAccessControlPoint]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_audit_access_control_points_1.AuditAccessControlPointLogsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AuditLogsTerminal]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_audit_terminal_1.AuditTerminalLogsExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AccessControlPointUsers]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_acp_users_list_1.AccessControlPointUsersExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.SystemFeedback]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_feedbacks_1.SystemFeedbackExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsMonthlyWorkingHoursSummaryExtraWork]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_montly_work_summary_extra_work_1.PacsMontlyWorkSummaryExtraWorkExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PacsPlanningCalendar]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_pacs_planning_calendar_1.PacsPlanningCalendarExcelReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.AccessControlPoints]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_access_control_points_1.ExcelReportAccessControlPoints(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return new report_pdf_access_control_points_1.PdfReportAccessControlPoints(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.PermissionClaimUsagesForEmployees]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_employees_permission_usage_report_1.EmployeesPermissionUsageReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
    [app_enums_1.enums.ReportCode.DurationInRegionReportOverview]: {
        [app_enums_1.enums.ReportFormat.Excel]: (request, locale) => {
            return new report_excel_duration_in_region_overview_1.DurationInRegionOverviewReport(request, locale);
        },
        [app_enums_1.enums.ReportFormat.Pdf]: (request, locale) => {
            return undefined;
        },
        [app_enums_1.enums.ReportFormat.Csv]: (request, locale) => {
            return undefined;
        },
    },
};
class WorkerPool {
    constructor(workerPath, numberOfThreads) {
        this.workerPath = workerPath;
        this.numberOfThreads = numberOfThreads;
        this.workersById = {};
        this.activeWorkersById = {};
        this.init();
        app_logs_1.logger.debug(numberOfThreads + " worker threads created for PDF reports...");
    }
    init() {
        if (this.numberOfThreads < 1) {
            return null;
        }
        for (let i = 0; i < this.numberOfThreads; i += 1) {
            const worker = new worker_threads_1.Worker(this.workerPath);
            this.workersById[i] = worker;
            this.activeWorkersById[i] = false;
        }
    }
    async run(request, locale, onTakesLong) {
        return new Promise(async (resolve, reject) => {
            let availableWorkerId = this.getInactiveWorkerId();
            if (availableWorkerId === -1) {
                const interval = setInterval(async () => {
                    availableWorkerId = this.getInactiveWorkerId();
                    if (availableWorkerId !== -1) {
                        clearInterval(interval);
                        try {
                            const response = await this.runWorker(availableWorkerId, request, locale, onTakesLong);
                            resolve(response);
                        }
                        catch (error) {
                            reject(error);
                        }
                    }
                }, 2000);
            }
            else {
                try {
                    const response = await this.runWorker(availableWorkerId, request, locale, onTakesLong);
                    resolve(response);
                }
                catch (error) {
                    reject(error);
                }
            }
        });
    }
    getInactiveWorkerId() {
        for (let i = 0; i < this.numberOfThreads; i++) {
            if (!this.activeWorkersById[i]) {
                return i;
            }
        }
        return -1;
    }
    async runWorker(workerId, request, locale, onTakesLong) {
        return new Promise((resolve, reject) => {
            const worker = this.workersById[workerId];
            this.activeWorkersById[workerId] = true;
            let isHugeReport = false;
            const messageCallback = (result) => {
                if (result === "TAKESTOOLONG") {
                    app_logs_1.logger.debug("Takes too long message response is obtained by main thread");
                    onTakesLong();
                    isHugeReport = true;
                }
                else if (typeof result === "object") {
                    app_logs_1.logger.debug("Report response is obtained by main thread");
                    cleanUp();
                    result.notificationType = request.reportTemplateId ? restapi_1.NotificationType.ScheduledReport : isHugeReport ? restapi_1.NotificationType.HugeReport : undefined;
                    resolve(result);
                }
                else {
                    cleanUp();
                    reject(result);
                }
            };
            const errorCallback = (error) => {
                cleanUp();
                reject(error);
            };
            const cleanUp = () => {
                worker.removeAllListeners("message");
                worker.removeAllListeners("error");
                this.activeWorkersById[workerId] = false;
            };
            worker.on("message", messageCallback);
            worker.on("error", errorCallback);
            worker.postMessage({ request, locale });
        });
    }
}
class AmqpServerToReportSub {
    constructor() {
        this.onMessage = async (msg) => {
            if (msg === null || msg.fields.redelivered) {
                this._channel.nack(msg, false, false);
                return;
            }
            const triggerType = msg.properties.headers["trigger-type"];
            let request = null;
            let scheduledReportRequestMsgBody = null;
            let receiverUserFilterId = null;
            let createdByUserId;
            try {
                if (triggerType === "scheduled") {
                    scheduledReportRequestMsgBody = JSON.parse(msg.content.toString());
                    const sjp = await (0, generator_1.prepareReportArgumentsForScheduledJob)({
                        organizationId: scheduledReportRequestMsgBody.organizationId,
                        scheduledJobId: scheduledReportRequestMsgBody.scheduledJobId,
                    });
                    request = sjp.request;
                    receiverUserFilterId = sjp.receiverFilterId;
                    createdByUserId = sjp.createdByUserId;
                }
                else {
                    request = JSON.parse(msg.content.toString());
                    createdByUserId = request.requesterUserId;
                }
            }
            catch (error) {
                app_logs_1.logger.error(error);
                await (0, business_notification_1.sendReportErrorNotification)({
                    organizationId: request.organizationId,
                    receiverUserIds: [request.requesterUserId],
                    reportResponse: {
                        filePath: null,
                        notificationType: restapi_1.NotificationType.ReportServiceError,
                        sendFileInAttachments: null,
                        format: request.format,
                        reportType: request.code,
                    },
                });
                if (triggerType === "instant") {
                    this._channel.sendToQueue(msg.properties.replyTo, {
                        error: "Error",
                    }, { correlationId: msg.properties.correlationId });
                }
                throw error;
            }
            const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId: request.organizationId, userId: createdByUserId });
            const onTakesLong = () => {
                this._channel.sendToQueue(msg.properties.replyTo, {
                    error: "Notification",
                }, { correlationId: msg.properties.correlationId });
            };
            const reportStartTime = (0, moment_1.default)();
            let reportResponse = null;
            try {
                if (request.format === app_enums_1.enums.ReportFormat.Excel) {
                    const generator = generatorCreatorList[request.code][app_enums_1.enums.ReportFormat.Excel](request, locale);
                    reportResponse = await generator.generateReportAndGetResponse(onTakesLong);
                }
                else if (request.format === app_enums_1.enums.ReportFormat.Csv) {
                    const generator = generatorCreatorList[request.code][app_enums_1.enums.ReportFormat.Csv](request, locale);
                    reportResponse = await generator.generateReportAndGetResponse(onTakesLong);
                }
                else {
                    app_logs_1.logger.info("PDF report request is sent to worker thread");
                    reportResponse = await this._workerPool.run(request, locale, onTakesLong);
                }
                if (!reportResponse) {
                    return;
                }
                app_logs_1.logger.debug("Report prepared now trying to send to related queue...");
                if (reportResponse.notificationType === app_enums_1.enums.NotificationType.HugeReport || reportResponse.notificationType === app_enums_1.enums.NotificationType.ScheduledReport) {
                    const preparedReportPath = path_1.default.join(app_config_1.appConfig.preparedReportDirectory, reportResponse.preparedReportId);
                    fs_1.default.copyFileSync(reportResponse.filePath, preparedReportPath);
                    fs_1.default.unlinkSync(reportResponse.filePath);
                    await (0, business_notification_1.sendReportNotification)({
                        organizationId: request.organizationId,
                        notificationId: scheduledReportRequestMsgBody?.notificationId,
                        receiverUserFilterId: reportResponse.notificationType === app_enums_1.enums.NotificationType.ScheduledReport ? receiverUserFilterId : null,
                        receiverUserIds: reportResponse.notificationType === app_enums_1.enums.NotificationType.HugeReport ? [request.requesterUserId] : null,
                        reportResponse,
                        request,
                    });
                }
                else {
                    const rpcResponse = {
                        filePath: reportResponse.filePath,
                        duration: (0, moment_1.default)().diff(reportStartTime, "milliseconds"),
                    };
                    this._channel.sendToQueue(msg.properties.replyTo, rpcResponse, { correlationId: msg.properties.correlationId });
                }
                this._channel.ack(msg);
            }
            catch (error) {
                this._channel.nack(msg);
                app_logs_1.logger.error(`Error while creating report: ${error?.message || error}`);
                if ((error?.message || error).includes("user_filter_user_pkey")) {
                    app_logs_1.logger.error("==========================================================================");
                    app_logs_1.logger.error(JSON.stringify(scheduledReportRequestMsgBody));
                }
                if (msg.properties.replyTo) {
                    this._channel.sendToQueue(msg.properties.replyTo, { error: error.message }, { correlationId: msg.properties.correlationId });
                }
                await (0, business_notification_1.sendReportErrorNotification)({
                    organizationId: request.organizationId,
                    receiverUserIds: [request.requesterUserId],
                    reportResponse,
                });
            }
        };
        this._channel = null;
        this._queue = null;
        this.workerThreadListener();
    }
    async workerThreadListener() {
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("Worker thread is inititating...");
            app_config_1.appConfig.init(app_constants_1.ServiceNames.ReportService);
            app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
            await dal_manager_1.dbManager.init({
                main: app_config_1.appConfig.db.main,
                log: app_config_1.appConfig.db.log,
                environment: app_config_1.appConfig.nodeEnv,
                ignoreMigrations: true,
                logDirectory: app_config_1.appConfig.dbLogDirectory,
                redis: app_config_1.appConfig.db.redis,
            }, false, false, false, false);
            worker_threads_1.parentPort.on("message", async (data) => {
                app_logs_1.logger.debug("Report request is taken by worker thread...,");
                const generator = generatorCreatorList[data.request.code][data.request.format](data.request, data.locale);
                worker_threads_1.parentPort.postMessage(await generator
                    .generateReportAndGetResponse(() => {
                    worker_threads_1.parentPort.postMessage("TAKESTOOLONG");
                })
                    .catch((error) => {
                    worker_threads_1.parentPort.postMessage("ERROR " + error);
                }));
            });
        }
    }
    async init() {
        return new Promise((resolve, reject) => {
            if (worker_threads_1.isMainThread) {
                this._workerPool = new WorkerPool(__filename, 2);
            }
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "server-to-report.sub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[server-to-report.sub] channel is establishing...");
                    await Promise.all([
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.serverToReport.queue, { durable: true }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, "topic", {
                            durable: true,
                        }),
                        channel.prefetch(5),
                    ]);
                    await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.serverToReport.queue, messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, messagebroker_models_1.MessageBrokerNames.serverToReport.reportBindingKey);
                    await channel.consume(messagebroker_models_1.MessageBrokerNames.serverToReport.queue, this.onMessage, { noAck: false });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-report.sub] error while creating channel: ", err));
            this._channel.on("connect", () => app_logs_1.logger.info("[server-to-report.sub] channel established"));
            this._channel.on("close", () => app_logs_1.logger.info("[server-to-report.sub] channel closed"));
            this._channel.once("connect", () => resolve());
        });
    }
}
exports.amqpServerToReportSub = new AmqpServerToReportSub();
