"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpServerToDevicePub = void 0;
const cluster_1 = __importDefault(require("cluster"));
const lodash_1 = __importDefault(require("lodash"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../app.enums");
const app_httpserver_1 = require("../app.httpserver");
const app_logs_1 = require("../app.logs");
const business_device_1 = require("../business/business.device");
const dal_manager_1 = require("../dal/dal.manager");
const ws_usernsp_1 = require("../ws/ws.usernsp");
const messageBroker_event_sub_1 = require("./messageBroker.event.sub");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
const messageBroker_server_to_app_pub_1 = require("./messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("./messageBroker.server-to-app.sub");
const enums_1 = require("../lib/access-model/v2/enums");
const dal_access_psql_log_1 = require("../dal/access/psql/dal.access.psql.log");
const cli_queries_1 = require("../dal/access/psql/cli-queries");
class AmqpServerToDevicePub {
    constructor() {
        this._rpcCallbackRegistry = {};
        this.onRpcMessage = async (msg) => {
            if (msg === null) {
                return;
            }
            let returnValues = {};
            try {
                const data = JSON.parse(msg.content.toString());
                const organizationId = msg.properties.headers["o"];
                if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.RemoteAccess) {
                    const now = new Date();
                    let log = await (0, business_device_1.updateLogFromDevice)({ organizationId, deviceLog: data.p, logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant });
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        let insertedLog = await dal_manager_1.dbManager.accessLog.addAccessLogFromDevice(organizationId, log, trx);
                        if (insertedLog.rg && insertedLog.rg.length > 0) {
                            for (const regionLog of insertedLog.rg) {
                                let regionNotification = {
                                    i: regionLog.i,
                                    s: regionLog.s.valueOf(),
                                    ee: regionLog.ee,
                                    xe: regionLog.xe,
                                    o: insertedLog.o,
                                    t: insertedLog.u,
                                };
                                await (0, cli_queries_1.upsertRegionStates)(organizationId, [{ accessControlPointId: insertedLog.a, log: regionNotification }], trx);
                                if (regionLog.ru && regionLog.ru.c) {
                                    await (0, cli_queries_1.upsertAccessRuleHistory)(organizationId, {
                                        date: now,
                                        accessRuleId: regionLog.ru.i,
                                        userId: insertedLog.o,
                                        count: regionLog.ru.c,
                                        accessControlPointId: insertedLog.a,
                                        actionDateISO: insertedLog.u,
                                        startDateTime: regionLog.ru.sd,
                                        endDateTime: regionLog.ru.ed,
                                        weekDay: regionLog.ru.wd,
                                    }, trx);
                                }
                                if (regionLog.rbu && regionLog.rbu.length) {
                                    await dal_manager_1.dbManager.accessUserGroup.updateUserGroupLimitRules({
                                        organizationId,
                                        rulesToUpdate: regionLog.rbu.map((rbu) => {
                                            return {
                                                userGroupId: rbu.g,
                                                accessRuleId: rbu.i,
                                                count: rbu.c >= 0 ? rbu.c : 0,
                                            };
                                        }),
                                        accessControlPointId: insertedLog.a,
                                        actionDateISO: insertedLog.u,
                                        trx,
                                    });
                                }
                                if (regionLog.rti) {
                                    await (0, cli_queries_1.updateUserRegionTicketUnits)(organizationId, {
                                        userId: insertedLog.o,
                                        accessControlPointId: insertedLog.a,
                                        decrementUnit: regionLog.rti.c,
                                        regionTicketId: regionLog.rti.i,
                                    }, trx);
                                }
                            }
                        }
                        if (insertedLog.ir === undefined && insertedLog.d && insertedLog.d === enums_1.EnumsV2.AccessDirection.Entrance && insertedLog.s === true) {
                            await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, {
                                credentialOwnerUserId: insertedLog.o,
                                generationTime: insertedLog.u,
                                redisCache: dal_manager_1.dbManager.accessRedisCache,
                                logId: insertedLog.id,
                            });
                        }
                        await messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.activity + "." + (log.a ? log.a : "*"), {
                            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.newlog,
                            p: insertedLog,
                            oId: organizationId,
                        });
                    });
                    returnValues = data.p;
                }
                else if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.ChangeAccessControlPointState) {
                    let response = data.p;
                    app_logs_1.logger.debug("CHANGE STATE " + JSON.stringify(response));
                    const userId = response.o;
                    const user = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId });
                    let oldAccessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV3(organizationId, userId, response.a);
                    await dal_manager_1.dbManager.accessAccessControlPoint.updateAccessControlPointState(organizationId, response.a, response.sl.map((r) => {
                        return {
                            state: r.s,
                            direction: r.d,
                        };
                    }));
                    let newAccessControlPoint = lodash_1.default.cloneDeep(oldAccessControlPoint);
                    newAccessControlPoint.states = response.sl.map((r) => {
                        return {
                            state: r.s,
                            direction: r.d,
                        };
                    });
                    await dal_manager_1.dbManager.accessAccessControlPoint.logAcpUpdate(organizationId, userId, oldAccessControlPoint, newAccessControlPoint);
                    await app_httpserver_1.httpServer.sendRemoteStateChangedNotification(organizationId, {
                        accessControlPointId: response.a,
                        accessControlPointName: (await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, response.a)).name,
                        fullname: user.f,
                        stateList: response.sl.map((r) => {
                            return {
                                state: r.s,
                                direction: r.d,
                            };
                        }),
                    });
                    returnValues = data.p;
                }
                else if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.CredentialRegisterMode) {
                    let response = data.p;
                    returnValues = data.p;
                }
                else if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.TakeSnapshot) {
                    let response = "ok";
                    returnValues = data.p;
                }
                else if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.RemoteRelay) {
                    let response = "ok";
                    returnValues = data.p;
                }
                else if (data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.SoftwareUpdateInterfaceCommand || data.e == messagebroker_models_1.ServerToDeviceRPCMessageType.OpenVPNConnectCommand) {
                    returnValues = data.p;
                }
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (this._rpcCallbackRegistry[msg.properties.correlationId]) {
                this._rpcCallbackRegistry[msg.properties.correlationId](null, returnValues);
                this._rpcCallbackRegistry[msg.properties.correlationId] = undefined;
            }
        };
        this._channel = null;
        this._v1rpcQueue = null;
    }
    async init() {
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "server-to-device.pub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[server-to-device.pub] channel is establishing...");
                    [this._v1rpcQueue] = await Promise.all([
                        channel.assertQueue(undefined, {
                            exclusive: true,
                            autoDelete: true,
                        }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deviceToServer.exchange, "topic", {
                            durable: true,
                        }),
                    ]);
                    await channel.consume(this._v1rpcQueue.queue, this.onRpcMessage, { noAck: true });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-device.pub] error while creating channel: ", err));
            this._channel.on("connect", () => app_logs_1.logger.info("[server-to-device.pub] channel created"));
            this._channel.on("close", () => app_logs_1.logger.info("[server-to-device.pub] channel closed"));
            this._channel.once("connect", () => resolve());
            if (cluster_1.default.isMaster) {
                messageBroker_event_sub_1.amqpEventSub.registerOnRegionStateChange(async (payload) => {
                    try {
                        let dbNotificationItem = JSON.parse(payload);
                        if (dbNotificationItem.relatedDeviceIds && dbNotificationItem.relatedDeviceIds.length > 0) {
                            for (let deviceId of dbNotificationItem.relatedDeviceIds) {
                                let now = new Date();
                                let regionNotification = {
                                    i: dbNotificationItem.data.regionId,
                                    s: dbNotificationItem.data.state,
                                    ee: dbNotificationItem.data.entranceLockExpirationDateISO ? dbNotificationItem.data.entranceLockExpirationDateISO : null,
                                    xe: dbNotificationItem.data.exitLockExpirationDateISO ? dbNotificationItem.data.exitLockExpirationDateISO : null,
                                    o: dbNotificationItem.data.userId,
                                    t: dbNotificationItem.data.actionDateISO,
                                };
                                exports.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
                                    e: messagebroker_models_1.ServerToDeviceRPCMessageType.RegionStateChanged,
                                    p: regionNotification,
                                }, 30000, (err, success) => {
                                    if (err) {
                                        app_logs_1.logger.warn("Could not send regionState deviceChange from RPCQueue, saving to terminalChange table for device:" + deviceId);
                                        let changeItem = {
                                            actionDateISO: now.toISOString(),
                                            type: app_enums_1.enums.DeviceChangeItemType.RegionState,
                                            data: {
                                                actionDateISO: dbNotificationItem.data.actionDateISO,
                                                regionId: dbNotificationItem.data.regionId,
                                                state: dbNotificationItem.data.state,
                                                userId: dbNotificationItem.data.userId,
                                                entranceLockExpirationUtc: dbNotificationItem.data.entranceLockExpirationDateISO
                                                    ? dbNotificationItem.data.entranceLockExpirationDateISO
                                                    : null,
                                                exitLockExpirationUtc: dbNotificationItem.data.exitLockExpirationDateISO ? dbNotificationItem.data.exitLockExpirationDateISO : null,
                                            },
                                        };
                                        dal_manager_1.dbManager.accessDevice.addTerminalChange(deviceId, dbNotificationItem.organizationId, changeItem);
                                    }
                                });
                            }
                        }
                    }
                    catch (error) {
                        app_logs_1.logger.error(error);
                    }
                });
                messageBroker_event_sub_1.amqpEventSub.registerOnAccessRuleAppliedChange(async (payload) => {
                    try {
                        let dbNotificationItem = JSON.parse(payload);
                        if (dbNotificationItem.relatedDeviceIds && dbNotificationItem.relatedDeviceIds.length > 0) {
                            for (let deviceId of dbNotificationItem.relatedDeviceIds) {
                                let now = new Date();
                                let accessRuleAppliedNotification = {
                                    i: dbNotificationItem.data.ruleId,
                                    u: dbNotificationItem.data.userId,
                                    c: dbNotificationItem.data.count,
                                    d: dbNotificationItem.data.actionDateISO,
                                    sd: dbNotificationItem.data.options?.startDateTime,
                                    ed: dbNotificationItem.data.options?.endDateTime,
                                    wd: dbNotificationItem.data.options?.dayOfWeek,
                                };
                                exports.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
                                    e: messagebroker_models_1.ServerToDeviceRPCMessageType.AccessRuleApplied,
                                    p: accessRuleAppliedNotification,
                                }, 30000, (err, success) => {
                                    if (err) {
                                        app_logs_1.logger.warn("Could not send accessrule deviceChange from RPCQueue, saving to terminalChange table for device:" + deviceId);
                                        let changeItem = {
                                            actionDateISO: now.toISOString(),
                                            type: app_enums_1.enums.DeviceChangeItemType.RegionMaxEntranceCountRuleState,
                                            data: dbNotificationItem.data,
                                        };
                                        dal_manager_1.dbManager.accessDevice.addTerminalChange(deviceId, dbNotificationItem.organizationId, changeItem);
                                    }
                                });
                            }
                        }
                    }
                    catch (error) {
                        app_logs_1.logger.error(error);
                    }
                });
                messageBroker_event_sub_1.amqpEventSub.registerOnAccessCapacityBasedRuleAppliedChange(async (payload) => {
                    try {
                        let dbNotificationItem = JSON.parse(payload);
                        if (dbNotificationItem.relatedDeviceIds && dbNotificationItem.relatedDeviceIds.length > 0) {
                            for (let deviceId of dbNotificationItem.relatedDeviceIds) {
                                let now = new Date();
                                let accessCapacityBasedRuleAppliedNotification = {
                                    i: dbNotificationItem.data.ruleId,
                                    c: dbNotificationItem.data.current,
                                    g: dbNotificationItem.data.groupId,
                                };
                                exports.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
                                    e: messagebroker_models_1.ServerToDeviceRPCMessageType.AccessCapacityBasedRuleApplied,
                                    p: accessCapacityBasedRuleAppliedNotification,
                                }, 30000, (err, success) => {
                                    if (err) {
                                        app_logs_1.logger.warn("Could not send accessrule deviceChange from RPCQueue, saving to terminalChange table for device:" + deviceId);
                                        let changeItem = {
                                            actionDateISO: now.toISOString(),
                                            type: app_enums_1.enums.DeviceChangeItemType.AccessCapacityBasedRuleApplied,
                                            data: dbNotificationItem.data,
                                        };
                                        dal_manager_1.dbManager.accessDevice.addTerminalChange(deviceId, dbNotificationItem.organizationId, changeItem);
                                    }
                                });
                            }
                        }
                    }
                    catch (error) {
                        app_logs_1.logger.error(error);
                    }
                });
            }
        });
    }
    async sendToExchange(routingKey, packet, expiration) {
        return new Promise((resolve, reject) => {
            let sent = this._channel.publish(messagebroker_models_1.MessageBrokerNames.deviceToServer.exchange, routingKey, packet, {
                persistent: true,
                contentType: "application/json",
                expiration: expiration || 60000,
            });
            if (!sent) {
                this._channel.once("drain", () => {
                    return resolve();
                });
            }
            else {
                return resolve();
            }
        });
    }
    sendToRpcQueue(deviceId, packet, expiration, callback) {
        this._v1rpcQueue.consumerCount;
        let correlationId = uuid_1.default.v4();
        this._rpcCallbackRegistry[correlationId] = callback;
        setTimeout(() => {
            if (this._rpcCallbackRegistry[correlationId]) {
                this._rpcCallbackRegistry[correlationId](new Error("Message expired"), null);
                this._rpcCallbackRegistry[correlationId] = undefined;
            }
        }, expiration);
        this._channel.sendToQueue("rpc_" + deviceId, packet, {
            correlationId: correlationId,
            replyTo: this._v1rpcQueue.queue,
            expiration: expiration,
        });
    }
}
exports.amqpServerToDevicePub = new AmqpServerToDevicePub();
