"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.notificationNack = exports.notificationAck = exports.sendToDelayedRetryQueue = exports.initServerToNotificationSubChannel = void 0;
const app_logs_1 = require("../app.logs");
const manager_1 = require("../business/notification/manager");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
let connection;
let channel;
const appTag = "[notification.sub.ts]";
async function initServerToNotificationSubChannel() {
    return new Promise((resolve, reject) => {
        connection = messageBroker_manager_1.messageBrokerManager.connection;
        channel = connection.createChannel({
            json: true,
            name: "notification.sub",
            setup: async (channel) => {
                app_logs_1.logger.info("%schannel is creating...", appTag);
                await Promise.all([
                    channel.assertExchange(messagebroker_models_1.MessageBrokerNames.serviceToNotification.exchange, "direct", { durable: true }),
                    channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, "direct", { durable: true }),
                    channel.assertExchange(messagebroker_models_1.MessageBrokerNames.notificationReAttempt.exchange, "x-delayed-message", {
                        durable: true,
                        arguments: { "x-delayed-type": "direct" },
                    }),
                    channel.assertQueue(messagebroker_models_1.MessageBrokerNames.serviceToNotification.queue, {
                        arguments: {
                            "x-dead-letter-exchange": messagebroker_models_1.MessageBrokerNames.deadLetter.exchange,
                            "x-dead-letter-routing-key": messagebroker_models_1.MessageBrokerNames.deadLetter.notificationBindingKey,
                        },
                        maxPriority: 10,
                        durable: true,
                    }),
                    channel.assertQueue(messagebroker_models_1.MessageBrokerNames.notificationReAttempt.queue, {
                        durable: true,
                        maxPriority: 10,
                    }),
                    channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.notificationQueue, { durable: true }),
                    channel.prefetch(5),
                ]);
                await Promise.all([
                    channel.bindQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.notificationQueue, messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, messagebroker_models_1.MessageBrokerNames.deadLetter.notificationBindingKey),
                    channel.bindQueue(messagebroker_models_1.MessageBrokerNames.serviceToNotification.queue, messagebroker_models_1.MessageBrokerNames.serviceToNotification.exchange, messagebroker_models_1.MessageBrokerNames.serviceToNotification.bindingKey),
                    channel.bindQueue(messagebroker_models_1.MessageBrokerNames.notificationReAttempt.queue, messagebroker_models_1.MessageBrokerNames.notificationReAttempt.exchange, messagebroker_models_1.MessageBrokerNames.notificationReAttempt.bindingKey),
                ]);
                await channel.consume(messagebroker_models_1.MessageBrokerNames.notificationReAttempt.queue, manager_1.onNotificationMessageToSend, { noAck: false });
                await channel.consume(messagebroker_models_1.MessageBrokerNames.serviceToNotification.queue, manager_1.onNotificationMessageReceived, { noAck: false });
            },
        });
        channel.on("error", (err) => app_logs_1.logger.error("error while creating channel: ", err));
        channel.on("connect", () => app_logs_1.logger.info("%schannel has been created", appTag));
        channel.on("close", () => app_logs_1.logger.info("%schannel closed", appTag));
        channel.once("connect", () => resolve());
    });
}
exports.initServerToNotificationSubChannel = initServerToNotificationSubChannel;
async function sendToDelayedRetryQueue(msg, priority) {
    return new Promise((resolve, reject) => {
        const attemptNumber = msg.a;
        const delayAmount = 5000 * attemptNumber;
        channel.publish(messagebroker_models_1.MessageBrokerNames.notificationReAttempt.exchange, messagebroker_models_1.MessageBrokerNames.notificationReAttempt.bindingKey, msg, {
            persistent: true,
            headers: { "x-delay": delayAmount },
            priority: priority ?? undefined,
        }, (err, ok) => {
            if (err) {
                app_logs_1.logger.error("%sBroker rejected message error: " + err, appTag);
                reject();
            }
            else {
                resolve(new Date(new Date().getTime() + delayAmount));
            }
        });
    });
}
exports.sendToDelayedRetryQueue = sendToDelayedRetryQueue;
function notificationAck(message, allUpTo) {
    channel.ack(message, allUpTo);
}
exports.notificationAck = notificationAck;
function notificationNack(message, allUpTo, requeue) {
    channel.nack(message, allUpTo, requeue);
}
exports.notificationNack = notificationNack;
