"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getQueueMessages = exports.getQueueDetails = exports.purgeQueueMessages = exports.refreshTerminalConnectionStates = void 0;
const got = require("got");
const app_logs_1 = require("../app.logs");
const business_device_1 = require("../business/business.device");
const dal_constants_1 = require("../dal/dal.constants");
const dal_manager_1 = require("../dal/dal.manager");
const terminal_1 = require("../lib/es/models/terminal");
const messageBroker_event_sub_1 = require("./messageBroker.event.sub");
const messagebroker_models_1 = require("./messagebroker.models");
const messageBroker_server_to_device_pub_1 = require("./messageBroker.server-to-device.pub");
async function refreshTerminalConnectionStates(params) {
    const response = await got.get(`http://${params.hostname}:${params.port}/api/connections`, {
        json: true,
        headers: {
            Authorization: params.authorization,
        },
    });
    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        const usersInRabbitMq = response.body.map((r) => r.user);
        const devices = await dal_manager_1.dbManager.accessDevice.getTerminalDeviceIdList(trx, params.organizationId);
        await Promise.all(devices.map(async (device) => {
            await (0, business_device_1.updateDeviceConnectionStatus)({
                organizationId: device.organizationId,
                deviceId: device.id,
                isConnected: usersInRabbitMq.indexOf(device.id) >= 0,
                isRestart: true,
            });
            if (usersInRabbitMq.indexOf(device.id) >= 0) {
                messageBroker_event_sub_1.amqpEventSub.registerOnTerminalChange(device.id, (() => {
                    app_logs_1.logger.debug(`Terminal change for device ${device.id}`);
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(device.id, {
                        e: messagebroker_models_1.ServerToDeviceMessageType.GetChangesRequired,
                    });
                }).bind(this));
            }
            const rabbitConnectionName = response.body.find((r) => r.user === device.id)?.name;
            if (rabbitConnectionName) {
                await dal_manager_1.dbManager.accessRedisCache.setValue(dal_constants_1.DalConstants.NewDataCacheType.DeviceRabbitMQConnectionName + "_" + rabbitConnectionName, JSON.stringify({ id: device.id, organizationId: device.organizationId }));
            }
            const changeCount = await dal_manager_1.dbManager.accessDevice.getChangeCountForTerminal(device.id, device.organizationId, trx);
            if (changeCount) {
                messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(device.id, {
                    e: messagebroker_models_1.ServerToDeviceMessageType.GetChangesRequired,
                });
            }
        }));
    });
}
exports.refreshTerminalConnectionStates = refreshTerminalConnectionStates;
async function purgeQueueMessages(params) {
    return new Promise(async (resolve, reject) => {
        const response = await got.delete(`http://${params.hostname}:${params.port}/api/queues/%2F/${params.queueName}/contents`, {
            headers: {
                Authorization: params.authorization,
            },
        });
        if (response.statusCode === terminal_1.HttpStatusCode.NoContent) {
            resolve();
        }
        else {
            reject();
        }
    });
}
exports.purgeQueueMessages = purgeQueueMessages;
async function getQueueDetails(params) {
    return (await got.get(`http://${params.hostname}:${params.port}/api/queues/%2f/${params.queueName}`, {
        json: true,
        headers: {
            Authorization: params.authorization,
        },
    })).body;
}
exports.getQueueDetails = getQueueDetails;
async function getQueueMessages(params) {
    return (await got.post(`http://${params.hostname}:${params.port}/api/queues/%2f/${params.queueName}/get`, {
        json: true,
        body: {
            count: 200,
            ackmode: "reject_requeue_true",
            encoding: "auto",
        },
        headers: {
            Authorization: params.authorization,
        },
    })).body;
}
exports.getQueueMessages = getQueueMessages;
