"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.closeDeadLetterChannel = exports.amqpToMonitorChannelInit = void 0;
const app_logs_1 = require("../app.logs");
const messagebroker_models_1 = require("./messagebroker.models");
const cli_queries_1 = require("../dal/access/psql/cli-queries");
const monitor_config_1 = __importDefault(require("../components/monitor/monitor.config"));
const dal_access_cache_redis_1 = require("../dal/access/redis/dal.access.cache.redis");
const dal_access_cache_client_1 = __importDefault(require("../dal/access/redis/dal.access.cache.client"));
const messageBroker_amqp_to_cli_sub_1 = require("./messageBroker.amqp-to-cli.sub");
const app_enums_1 = require("../app.enums");
const appTag = "[amqp-to-monitor.sub]";
let channel = null;
let dbPool = null;
let redisCache = null;
async function amqpToMonitorChannelInit(connection, pool) {
    try {
        const client = await dal_access_cache_client_1.default.connect({
            url: monitor_config_1.default.REDIS_SERVER_URL,
        });
        redisCache = new dal_access_cache_redis_1.RedisCache(client);
        dbPool = pool;
    }
    catch (err) {
        app_logs_1.logger.error(err);
        app_logs_1.logger.error(`REDIS_SERVER_URL ${monitor_config_1.default.REDIS_SERVER_URL}`);
        app_logs_1.logger.error(`There is a error while initiating deadLetter clients`);
    }
    return new Promise((resolve, reject) => {
        channel = connection.createChannel({
            json: true,
            name: "amqp-to-monitor.sub",
            setup: async (channel) => {
                app_logs_1.logger.debug("%s channel is establishing...", appTag);
                await Promise.all([
                    channel.prefetch(1),
                    channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, "direct", { durable: true }),
                    channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, {
                        durable: true,
                    }),
                ]);
                await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, messagebroker_models_1.MessageBrokerNames.deadLetter.deviceBindingKey);
                await channel.consume(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, onDeadLetterMessage, { noAck: false });
            },
        });
        channel.on("error", (err) => app_logs_1.logger.error("[amqp-to-monitor.sub] error while creating channel: ", err));
        channel.on("connect", () => app_logs_1.logger.info("%schannel created", appTag));
        channel.on("close", () => app_logs_1.logger.info("%schannel closed", appTag));
        channel.once("connect", () => resolve());
    });
}
exports.amqpToMonitorChannelInit = amqpToMonitorChannelInit;
async function closeDeadLetterChannel() {
    await channel.close();
}
exports.closeDeadLetterChannel = closeDeadLetterChannel;
const onDeadLetterMessage = async (msg) => {
    if (!msg) {
        return;
    }
    try {
        app_logs_1.logger.debug("[amqp-to-monitor.sub] Dead Letter Message Received");
        const data = JSON.parse(msg.content.toString());
        const organizationId = msg.properties.headers.o;
        const deviceId = msg.properties.headers.d;
        if (msg === null) {
            return;
        }
        if (data.e === messagebroker_models_1.DeviceToServerMessageTypes.AccessLog && data.p === null) {
            app_logs_1.logger.debug("[amqp-to-monitor.sub] Access log payload is empty. Discarding message.", appTag);
            channel.nack(msg, false, false);
            return;
        }
        const organizationExists = await (0, cli_queries_1.systemTransaction)(dbPool, async (trx) => (await trx.query(`SELECT id FROM public."organizationList" WHERE id = $1`, [organizationId])).rowCount);
        if (!organizationExists) {
            app_logs_1.logger.warn(`[amqp-to-monitor.sub] Organization[${organizationId}] is not exist. Discarding message...`);
            channel.nack(msg, false, false);
            return;
        }
        switch (data.e) {
            case messagebroker_models_1.DeviceToServerMessageTypes.AccessLog: {
                const deadLetterResult = await (0, messageBroker_amqp_to_cli_sub_1.onDeadLetterAccessLog)({
                    msg,
                    organizationId,
                    deviceId,
                    deviceAccessLog: data.p,
                    redisClient: redisCache,
                    pool: dbPool,
                });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.CounterSensorEvent: {
                const deadLetterResult = await (0, messageBroker_amqp_to_cli_sub_1.onDeadLetterCounterSensorLog)({ msg, organizationId, packet: data.p, pool: dbPool });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.StatusSensorStateChanged: {
                const deadLetterResult = await (0, messageBroker_amqp_to_cli_sub_1.onDeadLetterStatusSensorLog)({
                    msg,
                    organizationId,
                    log: (data.p || data.pp),
                    pool: dbPool,
                });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.TamperEvent: {
                const deadLetterResult = await (0, messageBroker_amqp_to_cli_sub_1.onDeadLetterTamperSwitch)({ msg, organizationId, packet: data.p, pool: dbPool });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.SystemHealth: {
                const deadLetterResult = await (0, messageBroker_amqp_to_cli_sub_1.onDeadLetterDeviceSystemStatusLogs)({
                    msg,
                    organizationId,
                    deviceId,
                    packet: data.p,
                    pool: dbPool,
                    redisClient: redisCache,
                });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.EmergencyStateChanged:
            case messagebroker_models_1.DeviceToServerMessageTypes.WebRTC:
                app_logs_1.logger.debug("[amqp-to-monitor.sub] Discarding WebRtc or Emergency Message...");
                channel.ack(msg);
                break;
            default:
                app_logs_1.logger.error("[amqp-to-monitor.sub] unknown msg %s", data.e);
                await (0, cli_queries_1.systemTransaction)(dbPool, async (trx) => {
                    return (0, cli_queries_1.insertUnhandledDeviceLog)({
                        organizationId,
                        deviceId,
                        logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                        type: data.e,
                        note: { message: "dead letter received an unknown message type" },
                        data: data.p,
                        trx,
                    });
                });
                channel.nack(msg, false, true);
                break;
        }
        app_logs_1.logger.debug("Offline Log Processed");
    }
    catch (error) {
        app_logs_1.logger.error(error);
        app_logs_1.logger.error("[amqp-to-monitor.sub] error while consuming msg %s", error.message || error);
        channel.nack(msg, false, true);
    }
};
