"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZiraatStatusSensorEvent = exports.generateReport = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const generator_1 = require("../../../business/report/generator");
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const app_config_1 = require("../../../app.config");
const uuid_1 = __importDefault(require("uuid"));
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const moment_1 = __importDefault(require("moment"));
const luxon_1 = require("luxon");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "DD.MM.YYYY";
    Constants.DateTimeFormat = "DD.MM.YYYY HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    const report = new ZiraatStatusSensorEvent(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class ZiraatStatusSensorEvent extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating sensor events excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 2;
        this.initGeneralSheet(this._wsGeneral, this._generalRowIndex - 1, 1);
        let currentTime = luxon_1.DateTime.fromISO(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTime).value);
        const data = await dal_manager_1.dbManager.accessDevice.reportStatusSensorPendingTimeoutLogs({
            organizationId: this._request.organizationId,
            filter: {
                dateTime: currentTime,
            },
        });
        for (const record of data) {
            this.addGeneralRow(record);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Ziraat Status Sensor Timeout Access Control Points Logs",
            reportContainsEmptyData: data?.length ? false : true,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 30 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_ACCESS_CONTROL_POINT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 30 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.TIMEOUT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 45 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(record) {
        let colIndex = 1;
        let cellForDate = this.createCellDate(record.actionUtc);
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: cellForDate ?? "",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: record.acpName.split("/")[0] ?? "",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        const statusMessage = i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_TIMEOUT", locale: this._locale }, { acpName: record.acpName.split("/")[0], date: cellForDate }) + "\r\n";
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: statusMessage ?? "",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    createCellDate(actionUtc) {
        let date = (0, moment_1.default)(actionUtc);
        return (0, moment_1.default)(date).locale(this._locale).format("LL") + " " + (0, moment_1.default)(date).locale(this._locale).format("HH:mm:ss");
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
    }
}
exports.ZiraatStatusSensorEvent = ZiraatStatusSensorEvent;
