"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ZiraatAccessAndSensorLogsExcel = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const report_util_1 = require("../../../business/report/report.util");
const a_1 = require("../a");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
    Constants.ColorCodes = {
        ON_TIME: "7C9923",
        TOLERATED: "D56626",
        NOT_TOLERATED: "A11913",
        PERMISSION: "4C2AA8",
    };
})(Constants || (Constants = {}));
class ZiraatAccessAndSensorLogsExcel extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._options = request.filter;
        this._requestFilter = request.filter;
        this._options = {
            startUtc: this._requestFilter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.start,
            endUtc: this._requestFilter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.end,
        };
    }
    async generateReport() {
        app_logs_1.logger.debug("Generating daily working details custom report");
        const option = Object.assign(this._options, {
            paginationRequest: {
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        });
        this._logsAndSnapshots = await (0, a_1.getAccessAndSensorLogs)({
            organizationId: this._request.organizationId,
            withSnapshots: false,
            startUtc: this._options.startUtc,
            endUtc: this._options.endUtc,
        });
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._logsAndSnapshots === undefined : undefined,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                i18n_1.default.__({ phrase: "CUSTOM-REPORT.ACCESS-AND-SENSOR-LOGS", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.startUtc).locale(this._locale).format("LL") +
                " - " +
                (0, moment_1.default)(this._options.endUtc).locale(this._locale).format("LL") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        for (const item of this._logsAndSnapshots.accessAndStatusSensorLogs) {
            this.addGeneralRow(item);
        }
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        this._wsGeneral.pageSetup.printArea = "A6:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready-> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 35 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 30 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.CABIN", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 30 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.DESCRIPTION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.RESULT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 30 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(data) {
        let colIndex = 1;
        if (data.type === 1) {
            const accessLog = data.log;
            const user = this._logsAndSnapshots.userCaptions.find((uc) => uc.id === accessLog.o);
            if (user) {
                const captions = (0, report_util_1.parseCaption)(user.caption);
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex++,
                    value: captions.columns[1].text.toUpperCase(),
                    ws: this._wsGeneral,
                    border: Constants.Styling.Dotted,
                });
            }
            else {
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex++,
                    value: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale: this._locale }),
                    ws: this._wsGeneral,
                    border: Constants.Styling.Dotted,
                });
            }
            const accessTime = (0, moment_1.default)(accessLog.u).locale(this._locale);
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: (accessTime.format("L") + " " + accessTime.format("HH:mm:ss")),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: this._logsAndSnapshots.accessControlPoints.find((acp) => acp.id === accessLog.a).name,
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            let cellForStatus = this.createCellForAccessLogStatus(accessLog, this._locale);
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: (cellForStatus.log ?? cellForStatus.description),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: cellForStatus.statusInfo,
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
                textColorCode: cellForStatus.statusColor,
            });
        }
        else {
            const sensorLog = data.log;
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale: this._locale }),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            const sensorTime = (0, moment_1.default)(sensorLog.u).locale(this._locale);
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: (sensorTime.format("L") + " " + sensorTime.format("HH:mm:ss")),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            const sensorLogAccessPointName = this._logsAndSnapshots.accessControlPoints.find((acp) => acp.id === sensorLog.a).name;
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: sensorLogAccessPointName,
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: i18n_1.default.__({ phrase: "CUSTOM-REPORT.SENSOR_UNKNOWN_ENTRANCE", locale: this._locale }, {
                    an: sensorLogAccessPointName,
                }),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: i18n_1.default.__({
                    phrase: sensorLog.r === app_enums_1.enums.StatusSensorReason.UnknownReasonOpenAlthoughRelayIsMapped
                        ? "CUSTOM-REPORT.OPENED_WITH_KEY"
                        : report_util_1.ReportConstants.StatusSensorReasons.find((ssr) => ssr.type === sensorLog.r)?.description,
                    locale: this._locale,
                }),
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
        }
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + "\n" + i18n_1.default.__({ phrase: "CUSTOM-REPORT.ACCESS-AND-SENSOR-LOGS", locale: this._locale }), border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)(this._options.startUtc).locale(this._locale).format("LL") + " - " + (0, moment_1.default)(this._options.endUtc).locale(this._locale).format("LL"), border, { italic: true }, Constants.DateFormat);
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.textColorCode) {
            c.font = {
                color: { argb: params.textColorCode },
                bold: true,
            };
        }
    }
    createCellForAccessLogStatus(record, locale) {
        let log;
        if (record.v) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale });
        }
        else if (record.ir) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale });
        }
        else if (record.di) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale });
        }
        let translationKey = (0, report_util_1.getDescriptionAccessLogItem)(record.r, record.d);
        let isSuccess = record.r === app_enums_1.enums.AccessLogReason.Success || (record.di && record.r === app_enums_1.enums.AccessLogReason.Manual);
        let credentialDescription = "";
        if (record.cx && record.cx.length > 0) {
            let visibleCredentialData = [];
            record.cx.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.t));
            });
            record.cx.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.t);
                credentialDescription += i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.d && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.d + ")";
                }
            });
        }
        return {
            statusColor: isSuccess ? "7C9923" : "A11913",
            description: record != null && record.an != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.an.split("/")[0],
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.d, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
}
exports.ZiraatAccessAndSensorLogsExcel = ZiraatAccessAndSensorLogsExcel;
async function generateReport(request, locale) {
    const report = new ZiraatAccessAndSensorLogsExcel(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
