"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ZiraatAccessAndSensorLogsPDF = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../../../business/report/generator");
const report_util_1 = require("../../../business/report/report.util");
const a_1 = require("../a");
class ZiraatAccessAndSensorLogsPDF extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._organizationId = request.organizationId;
        const filter = request.filter;
        this._options = {
            startUtc: filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.start,
            endUtc: filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.end,
        };
        this._locale = locale;
    }
    async generateReport() {
        const data = await (0, a_1.getAccessAndSensorLogs)({ organizationId: this._request.organizationId, withSnapshots: false, startUtc: this._options.startUtc, endUtc: this._options.endUtc });
        const locale = this._locale;
        const reportLogsData = [
            [
                { text: i18n_1.default.__({ phrase: "PDF-REPORT.caption", locale }), style: "tableHeader" },
                { text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DETAILS", locale }), style: "tableHeader" },
            ],
        ];
        for (const rawLog of data.accessAndStatusSensorLogs) {
            let pdfRow = [];
            let logDetails = [];
            let userCaption = [];
            if (rawLog.type === 1) {
                const accessLog = rawLog.log;
                const user = data.userCaptions.find((uc) => uc.id === accessLog.o);
                if (user) {
                    const captions = (0, report_util_1.parseCaption)(user.caption);
                    userCaption.push({
                        columns: [
                            {
                                image: captions.columns[0].image,
                                width: report_util_1.ReportConstants.widthProfilePicture * 2,
                                height: report_util_1.ReportConstants.heightProfilePicture * 2,
                                style: "center",
                                margin: [20, 20, 20, 5],
                            },
                            {
                                text: captions.columns[1].text,
                                fontSize: 14,
                                color: "black",
                                margin: [40, 40, 0, 20],
                            },
                        ],
                    });
                }
                else {
                    userCaption.push({
                        columns: [
                            {
                                image: report_util_1.ReportConstants.emptyAvatar,
                                width: report_util_1.ReportConstants.widthProfilePicture * 2,
                                height: report_util_1.ReportConstants.heightProfilePicture * 2,
                                style: "center",
                                margin: [20, 20, 20, 5],
                            },
                            {
                                text: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale }),
                                fontSize: 14,
                                color: "black",
                                margin: [40, 40, 0, 20],
                            },
                        ],
                    });
                }
                const m = (0, moment_1.default)(accessLog.u).locale(locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.date", locale }) + ": " + m.format("L") + " " + m.format("HH:mm:ss"),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.CABIN", locale }) + ": " + data.accessControlPoints.find((acp) => acp.id === accessLog.a).name,
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                let cellForStatus = this.createCellForAccessLogStatus(accessLog, this._locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DESCRIPTION", locale }) + ": " + (cellForStatus.log ?? cellForStatus.description) + "\n",
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.RESULT", locale }) + ": " + cellForStatus.statusInfo + "\n",
                            fontSize: 14,
                            color: cellForStatus.statusColor,
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
            }
            else {
                const sensorLog = rawLog.log;
                userCaption.push({
                    columns: [
                        {
                            image: report_util_1.ReportConstants.emptyAvatar,
                            width: report_util_1.ReportConstants.widthProfilePicture * 2,
                            height: report_util_1.ReportConstants.heightProfilePicture * 2,
                            style: "center",
                            margin: [20, 20, 20, 5],
                        },
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale }),
                            fontSize: 14,
                            color: "black",
                            margin: [40, 40, 0, 20],
                        },
                    ],
                });
                const m = (0, moment_1.default)(sensorLog.u).locale(locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.date", locale }) + ": " + m.format("L") + " " + m.format("HH:mm:ss"),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                const sensorLogAccessPointName = data.accessControlPoints.find((acp) => acp.id === sensorLog.a).name;
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.CABIN", locale }) + ": " + sensorLogAccessPointName,
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DESCRIPTION", locale }) +
                                ": " +
                                i18n_1.default.__({ phrase: "CUSTOM-REPORT.SENSOR_UNKNOWN_ENTRANCE", locale: this._locale }, {
                                    an: sensorLogAccessPointName,
                                }),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.RESULT", locale }) +
                                ": " +
                                i18n_1.default.__({
                                    phrase: sensorLog.r === app_enums_1.enums.StatusSensorReason.UnknownReasonOpenAlthoughRelayIsMapped
                                        ? "CUSTOM-REPORT.OPENED_WITH_KEY"
                                        : report_util_1.ReportConstants.StatusSensorReasons.find((ssr) => ssr.type === sensorLog.r)?.description,
                                    locale: this._locale,
                                }),
                            fontSize: 14,
                            color: "#62A420",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
            }
            pdfRow.push(userCaption);
            pdfRow.push(logDetails);
            reportLogsData.push(pdfRow);
        }
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + (0, moment_1.default)().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                            alignment: "left",
                            fontSize: "10",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            pageOrientation: "landscape",
            content: [
                data.organizationInfo.n,
                { text: i18n_1.default.__({ phrase: "PDF-REPORT.access_logs", locale }) + "\n", style: "header" },
                {
                    text: [
                        { text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_DATE", locale }) + " : ", style: "filterBold" },
                        {
                            text: (0, moment_1.default)(this._options.startUtc).locale(locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(this._options.endUtc).locale(locale).format("DD/MM/YYYY HH:mm"),
                            style: "filter",
                        },
                    ],
                },
                {
                    style: "tableStyle",
                    layout: {
                        fillColor: function (rowIndex, node, columnIndex) {
                            return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                        },
                    },
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["auto", "auto"],
                        body: reportLogsData,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 24,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "center",
                },
                tableStyle: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 18,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 10,
                    color: "gray",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
                filter: {
                    fontSize: 14,
                    color: "black",
                    alignment: "left",
                },
                filterBold: {
                    fontSize: 14,
                    color: "black",
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
        const res = await this.createPdf();
        return {
            filePath: res,
            format: app_enums_1.enums.ReportFormat.Pdf,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            notificationType: undefined,
            sendFileInAttachments: false,
            customReportName: i18n_1.default.__({ phrase: "CUSTOM-REPORT.NAME", locale }),
            reportContainsEmptyData: false,
            preparedReportId: this._fileId,
        };
    }
    createPdf() {
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        return new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(this._filePath, result);
                app_logs_1.logger.debug("Report generated:" + this._filePath);
                resolve(this._filePath);
            });
        });
    }
    createCellForAccessLogStatus(record, locale) {
        let log;
        if (record.v) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale });
        }
        else if (record.ir) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale });
        }
        else if (record.di) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale });
        }
        let translationKey = (0, report_util_1.getDescriptionAccessLogItem)(record.r, record.d);
        let isSuccess = record.r === app_enums_1.enums.AccessLogReason.Success || (record.di && record.r === app_enums_1.enums.AccessLogReason.Manual);
        let credentialDescription = "";
        if (record.cx && record.cx.length > 0) {
            let visibleCredentialData = [];
            record.cx.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.t));
            });
            record.cx.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.t);
                credentialDescription += i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.d && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.d + ")";
                }
            });
        }
        return {
            statusColor: isSuccess ? "#62A420" : "#E62700",
            description: record != null && record.an != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.an.split("/")[0],
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.d, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
}
exports.ZiraatAccessAndSensorLogsPDF = ZiraatAccessAndSensorLogsPDF;
async function generateReport(request, locale) {
    let report = new ZiraatAccessAndSensorLogsPDF(request, locale);
    return await report.generateReport();
}
exports.generateReport = generateReport;
