"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TemsanIsbasiYapanPersonelCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_access_error_1 = require("../../../dal/access/dal.access.error");
const api_error_1 = require("../../../api/api.error");
const report_util_1 = require("../../../business/report/report.util");
var Constants;
(function (Constants) {
    Constants.DateFormat = "dd.mm.yyyy";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    const report = new TemsanIsbasiYapanPersonelCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class TemsanIsbasiYapanPersonelCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._request = request;
        this._filter = request.filter;
        if (!(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit)?.value?.length > 0)) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("Organization unit must be defined for this report");
        }
        this._options = {
            date: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value).startOf("day"),
            organizationUnitId: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value[0],
            applyOrganizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
    }
    async generateReport() {
        app_logs_1.logger.info("Generating custom report");
        this._filePath = await this.generate();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "TEMSAN İşbaşı Yapan Personel Listesi",
        };
    }
    async generate() {
        this._organizationName = (await dal_manager_1.dbManager.accessRedisCache.getOrganizationCache({ organizationId: this._request.organizationId })).n;
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 8;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationName +
                " " +
                " İşbaşı Yapan Personel Listesi " +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                this._options.date.clone().locale(this._locale).format("LL") +
                +"\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Requester user not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        this.initGeneralSheet(this._wsGeneral, 7, 1);
        await this.addRows();
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.info("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Organizasyon Birimi",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 30 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Sicil No",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 20 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Adı Soyadı",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 30 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Tarih",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 20 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Saat",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 15 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Neden",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 10 });
        this.addInitCell({
            ws: wsRaw,
            row: startRowIndex,
            col: startColIndex++,
            value: "Kapı",
            border: Constants.Styling.AllThin,
        });
        columns.push({ width: 30 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    async addRows() {
        let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
            organizationId: this._request.organizationId,
            userId: this._request.requesterUserId,
            organizationUnitIds: [this._options.organizationUnitId],
        });
        if (!authResultForFilter.result) {
            throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
        }
        let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: this._request.organizationId,
            organizationUnitIds: [this._options.organizationUnitId],
            filterOrganizationUnitMembersHierarchically: this._options.applyOrganizationUnitHierarchically,
        });
        let filterUserIds = (filterUsers && filterUsers.length) > 0 ? filterUsers.map((f) => f.userId) : [];
        const option = {
            sortDateDesc: true,
            take: filterUsers.length * 50,
            authorizeForFilter: true,
            endUtc: this._options.date.clone().endOf("day").toDate(),
            startUtc: this._options.date.toDate(),
            organizationUnitIds: [this._options.organizationUnitId],
            filterOrganizationUnitMembersHierarchically: this._options.applyOrganizationUnitHierarchically,
            userIds: filterUserIds,
        };
        await dal_manager_1.dbManager.accessAccessLog.getAccessLogsReportNew(this._request.organizationId, option, async (rows) => {
            app_logs_1.logger.debug("[AccessLogsReport]Fetch data %s rows", rows.items.length);
            this.constructRows(rows);
        });
    }
    constructRows(rows) {
        rows.items.forEach((record) => {
            let cellForIdentity = this.createCellForIdentity(record, this._locale);
            let cellForAccessTime = this.createCellForAccessTime(record, this._locale);
            let cellForAccessPoint = this.createCellForAccessPoint(record, this._locale);
            let cellForStatus = this.createCellForStatus(record, this._locale);
            let cellForUserGroups = this.createCellForUserGroups(record);
            let cellForOrganizationUnits = this.createCellForOrganizationUnits(record);
            this.addRowToGeneralWorksheet(this._wsGeneral, cellForIdentity, cellForAccessTime, cellForAccessPoint, cellForStatus, cellForUserGroups, cellForOrganizationUnits);
        });
    }
    addRowToGeneralWorksheet(worksheet, cellForIdentity, cellForAccessTime, cellForAccessPoint, cellForStatus, cellForUserGroups, cellForOrganizationUnits) {
        let identity = cellForIdentity.caption != null ? cellForIdentity.caption : cellForIdentity.result;
        let accessTime = cellForAccessTime;
        let acp = cellForAccessPoint.an;
        let result = cellForStatus.statusInfo;
        let description = cellForStatus.description + "\r\n" + cellForStatus.log;
        let credentialDescription = cellForStatus.credentialDescription;
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: this.checkNull(cellForOrganizationUnits),
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: this.checkNull(identity.split("\r\n")[1]),
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: this.checkNull(identity.split("\r\n")[0]),
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: accessTime.date,
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: accessTime.time,
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: "",
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: this.checkNull(acp),
            ws: worksheet,
            border: Constants.Styling.Dotted,
        });
        worksheet.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    checkNull(value) {
        if (value) {
            return value;
        }
        else {
            return "";
        }
    }
    createCellForIdentity(record, locale) {
        let result = "";
        let caption;
        if (record.userCaptions != null && record.userCaptions.length > 0)
            caption = (0, report_util_1.parseCaptionForExcel)(record.userCaptions);
        else {
            if (record.identity && !record.userCaptions) {
                result += record.identity.organizationUnits;
            }
            else {
                result += "";
            }
            if (!record.identity && !record.isExitButton) {
                result += i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale });
            }
            if (!record.identity && record.isExitButton) {
                result += i18n_1.default.__({ phrase: "PDF-REPORT.identity_exit_button", locale });
            }
            if (!record.identity && !record.isExitButton && record.unknownCredentialData) {
                result += "-" + record.unknownCredentialData;
            }
        }
        return {
            result: result,
            caption: caption,
        };
    }
    createCellForAccessTime(record, locale) {
        const result = {
            date: "",
            time: "",
        };
        if (record && record.utc) {
            let date = (0, moment_1.default)(record.utc);
            result.date = (0, moment_1.default)(date).locale(this._locale).format("LL");
            result.time = (0, moment_1.default)(date).locale(this._locale).format("HH:mm:ss");
        }
        return result;
    }
    createCellForAccessPoint(record, locale) {
        let visual = "";
        let counterSensorData = "";
        let statusSensorData = "";
        return {
            an: record.accessControlPoint != null && record.accessControlPoint.name != null ? record.accessControlPoint.name : "",
            visual: visual,
            counterSensorData: counterSensorData,
            statusSensorData: statusSensorData,
        };
    }
    createCellForUserGroups(record) {
        let result = "";
        if (record.identity == null)
            return result;
        let userGroups = record.identity.userGroups;
        if (userGroups == null || userGroups.length == 0)
            return result;
        userGroups.forEach((userGroup) => {
            result += userGroup.name + "\r\n";
        });
        return result;
    }
    createCellForOrganizationUnits(record) {
        let result = "";
        if (record.identity == null)
            return result;
        let organizationUnits = record.identity.organizationUnits;
        if (organizationUnits == null || organizationUnits.length == 0)
            return result;
        organizationUnits.forEach((org) => {
            result += org.name + "\r\n";
        });
        return result;
    }
    createCellForStatus(record, locale) {
        let log = "";
        if (record.isVisitor) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale }) + "\r\n";
        }
        if (record.isRemoteAccess) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale }) + "\r\n";
        }
        if (record.isManuallyLogged) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_manual", locale }, { manualCreator: record.manualRecordCreatorCaptions[0].text[0] }) + "\r\n";
        }
        if (record.isExitButton) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale }) + "\r\n";
        }
        let translationKey = this.getStatusDescription(record.result, record.direction);
        let isSuccess = this.isSuccess(record);
        let credentialDescription = "";
        if (record.credentials && record.credentials.length > 0) {
            let visibleCredentialData = [];
            record.credentials.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.type));
            });
            record.credentials.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.type);
                credentialDescription += "\r\n" + i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.data && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.data + ")";
                }
                if (record.mobileAuthenticated) {
                    credentialDescription += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.MOBILE_AUTHENTICATED", locale });
                }
            });
        }
        return {
            statusColor: this.getStatusColor(isSuccess),
            description: record.accessControlPoint != null && record.accessControlPoint.name != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.accessControlPoint.name,
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.direction, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
    isSuccess(record) {
        return record.result === app_enums_1.enums.AccessLogReason.Success || (record.isManuallyLogged && record.result === app_enums_1.enums.AccessLogReason.Manual);
    }
    getStatusColor(isSuccess) {
        return isSuccess ? "#62A420" : "#E62700";
    }
    getStatusDescription(result, direction) {
        return (0, report_util_1.getDescriptionAccessLogItem)(result, direction);
    }
    async addFilterToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        let filterResult = await this.createFilters();
        let filtersArr = [];
        if (filterResult.organizationUnits) {
            filtersArr.push(i18n_1.default.__({ phrase: "PDF-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + filterResult.organizationUnits);
        }
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS", locale: this._locale }), border, {
            bold: true,
        });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, filterResult.now, border, { italic: false });
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, border);
        this.addFilterCell(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE_REPORT_BELONGS_TO", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 3, startColIndex + 1, this._options.date.locale(this._locale).format("LL") +
            " " +
            this._options.date.locale(this._locale).format("HH:mm:ss") +
            " - " +
            this._options.date.clone().endOf("day").locale(this._locale).format("LL") +
            " " +
            this._options.date.clone().endOf("day").locale(this._locale).format("HH:mm:ss"), border);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
        ws.mergeCells(startRowIndex + 4, startColIndex + 1, startRowIndex + 4, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 4, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 4, startColIndex + 1, filtersArr.join("\r\n"), border);
        let filterlength = filtersArr.length;
        let row = ws.getRow(startRowIndex + 4);
        row.height = filterlength * 20;
    }
    async createFilters() {
        let date = (0, moment_1.default)(this._options.date).locale(this._locale).format("LL") +
            " " +
            (0, moment_1.default)(this._options.date).locale(this._locale).format("HH:mm:ss") +
            " - " +
            (0, moment_1.default)(this._options.date).clone().endOf("day").locale(this._locale).format("LL") +
            " " +
            (0, moment_1.default)(this._options.date).clone().endOf("day").locale(this._locale).format("HH:mm:ss");
        let filterResult = await dal_manager_1.dbManager.accessReport.collectAccessLogReportFilterInfo([], [], [this._options.organizationUnitId], [], [], this._request.organizationId);
        let now = (0, moment_1.default)().locale(this._locale).format("LL") + " " + (0, moment_1.default)().locale(this._locale).format("HH:mm:ss");
        let organizationUnits;
        if (filterResult.organizationUnits != null && filterResult.organizationUnits.length > 0) {
            let res = "";
            filterResult.organizationUnits.forEach((org, index) => {
                res += org.name + (index < filterResult.organizationUnits.length - 1 ? "\r\n" : " ");
            });
            organizationUnits = res;
        }
        return {
            date: date,
            now: now,
            filterResult: filterResult,
            organizationUnits: organizationUnits,
        };
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
    addInitCell(params) {
        let c = params.ws.getCell(params.row, params.col);
        c.value = params.value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.textColor) {
            c.font = {
                color: { argb: params.textColor },
            };
        }
        if (params.backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        }
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value ?? "";
        c.alignment = {
            horizontal: params.inMiddle ? "center" : "left",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: params.isBold || false,
        };
        if (params.border) {
            c.border = Object.assign(c.border ?? {}, params.border);
        }
        if (params.numFmt) {
            c.numFmt = params.numFmt;
        }
        if (params.backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        }
        if (params.textColor) {
            c.font = {
                color: { argb: params.textColor },
            };
        }
    }
}
exports.TemsanIsbasiYapanPersonelCustomReport = TemsanIsbasiYapanPersonelCustomReport;
