"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.PENTIARGECustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const fs_1 = __importDefault(require("fs"));
class PENTIARGECustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._request = request;
        this._options = request.filter;
        this._currentDate = (0, moment_1.default)();
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._hsFmt = generator_1.ReportGenerator.Constants.hsFmtTr;
        this.begin = (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay)
            .startOf("day")
            .toDate();
        this.end = (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay)
            .endOf("day")
            .toDate();
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating Penti Arge excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: "tr" }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        let userFilters = {
            organizationUnitIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            applyOrganizationUnitFilterHierarchically: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userGroupIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            userIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            workPlanIds: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
        };
        if (userFilters?.workPlanIds) {
            this._workplans = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._request.organizationId, userFilters?.workPlanIds);
        }
        let filters = [];
        if (userFilters?.organizationUnitIds && userFilters?.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, userFilters?.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: "tr" }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (userFilters?.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: "tr" }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: "tr" }));
            }
        }
        if (userFilters?.userGroupIds && userFilters?.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, userFilters?.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: "tr" }) + " : " + userGroupNames.map((ugn) => ugn.name).join(", "));
        }
        if (userFilters?.workPlanIds && userFilters?.workPlanIds.length > 0) {
            let workplanNames = this._workplans.items.filter((wp) => userFilters?.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: "tr" }) + " : " + workplanNames.join(", "));
        }
        if (userFilters?.userIds && userFilters?.userIds.length > 0) {
            let identities = await dal_manager_1.dbManager.accessIdentity.listIdentity(this._request.organizationId, userFilters?.userIds);
            let userNames = identities.items.filter((i) => userFilters?.userIds.includes(i.id)).map((i) => i.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: "tr" }) + " : " + userNames.join(", "));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        this._data = await dal_manager_1.dbManager.accessPacs2.getMonthlyReport(this._request.organizationId, this._request.requesterUserId, {
            userFilter: userFilters,
            dateFilter: {
                group: 4,
                range: {
                    startDateTime: this.begin,
                    endDateTime: this.end
                }
            }
        });
        for (const item of this._data.items) {
            this.addGeneralRow(item);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "PENTİ ARGE RAPORU",
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.items.length === 0 : undefined,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 20
            },
            {
                phrase: "EXCEL-REPORT.EMPLOYEE",
                cellWidth: 35
            },
            {
                value: "İçeride Geçirilen Süre",
                cellWidth: 20
            },
            {
                value: "Dışarıda Geçirilen Süre",
                cellWidth: 20
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    numberToDurationFormat(minute) {
        return Math.floor(minute / 1440) + (minute % 1440) / 1440;
    }
    addGeneralRow(item) {
        let colIndex = 1;
        const rowData = [
            {
                value: item.user.uniqueId ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
            },
            {
                value: item.user.fullname.toUpperCase() ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
            },
            {
                value: this.numberToDurationFormat(item.editedWorkingDurations.normalWorkDuration),
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                numFmt: this._hsFmt,
            },
            {
                value: this.numberToDurationFormat(item.editedWorkingDurations.missedWorkDuration),
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                numFmt: this._hsFmt,
            },
        ];
        colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + "ARGE RAPORU", generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale("tr").format("LL") + " " + this._currentDate.locale("tr").format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE_REPORT_BELONGS_TO", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        let startOfMonth = (0, moment_1.default)(this.begin);
        let endOfMonth = (0, moment_1.default)(this.end);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, startOfMonth.locale("tr").format("LL") + " - " + endOfMonth.locale("tr").format("LL"), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        ws.mergeCells(startRowIndex + 4, startColIndex + 1, startRowIndex + 4, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 4);
        row.height = filterlength * 25;
    }
}
exports.PENTIARGECustomReport = PENTIARGECustomReport;
async function generateReport(request, locale) {
    let report = new PENTIARGECustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
