"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.PENTIDailyWorkingDetailsCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const business_pacs_report_1 = require("../../../business/pacs/business.pacs.report");
const dal_constants_1 = require("../../../dal/dal.constants");
const report_util_1 = require("../../../business/report/report.util");
const fs_1 = __importDefault(require("fs"));
class PENTIDailyWorkingDetailsCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._request = request;
        this._options = request.filter;
        this._currentDate = (0, moment_1.default)();
        this._organizationUnitsOfUsers = {};
        this._positionsOfUsers = {};
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        if (this._request.reportTemplateId) {
            this.date = (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value)
                .subtract(1, "day")
                .toDate();
        }
        else {
            this.date = (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value).toDate();
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating Penti Günlük Çalışma Detayı excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: "tr" }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        let userFilters = {
            organizationUnitIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            applyOrganizationUnitFilterHierarchically: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userGroupIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            userIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            workPlanIds: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
        };
        if (userFilters?.workPlanIds) {
            this._workplansFilter = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._request.organizationId, userFilters?.workPlanIds);
        }
        let filters = [];
        if (userFilters?.organizationUnitIds && userFilters?.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, userFilters?.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: "tr" }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (userFilters?.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: "tr" }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: "tr" }));
            }
        }
        if (userFilters?.userGroupIds && userFilters?.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, userFilters?.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: "tr" }) + " : " + userGroupNames.map((ugn) => ugn.name).join(", "));
        }
        if (userFilters?.workPlanIds && userFilters?.workPlanIds.length > 0) {
            let workplanNames = this._workplansFilter.items.filter((wp) => userFilters?.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: "tr" }) + " : " + workplanNames.join(", "));
        }
        if (userFilters?.userIds && userFilters?.userIds.length > 0) {
            let identities = await dal_manager_1.dbManager.accessIdentity.listIdentity(this._request.organizationId, userFilters?.userIds);
            let userNames = identities.items.filter((i) => userFilters?.userIds.includes(i.id)).map((i) => i.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: "tr" }) + " : " + userNames.join(", "));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        const option = {
            userFilter: userFilters,
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
            date: this.date,
            paginationRequest: {
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        };
        this._data = await dal_manager_1.dbManager.accessPacs2.workSummaryOfFilteredEmployeesAtDate(this._request.requesterUserId, this._request.organizationId, option);
        const userIds = this._data.items.map((i) => i.userId);
        const firstLastData = await (0, business_pacs_report_1.getDailySummaryReport)(this._request.organizationId, {
            date: new Date(this.date),
            userIds: userIds,
            status: dal_constants_1.DalConstants.IdentityStatusType.All,
            sortingMethod: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            pagination: {
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
            },
            hasOrganizationWide: undefined,
            permittedUnitIds: undefined,
        }, this._request.requesterUserId);
        this._items = firstLastData.items;
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            for (const userId of userIds) {
                this._organizationUnitsOfUsers[userId] = await dal_manager_1.dbManager.accessUser.listUserUnitsHierarchicallyOrdered(this._request.organizationId, userId, trx);
                this._positionsOfUsers[userId] = (await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileExtensionFields(this._request.organizationId, userId))?.position;
            }
        });
        for (const item of this._data.items) {
            this.addGeneralRow(item);
        }
        if (this._isDummyAccessLogAppeared) {
            this.addInitCell11(this._wsGeneral, this._generalRowIndex + 2, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.WARNING_FOR_ACCESS_LOGS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Günlük Çalışma Detayı Raporu",
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.items.length === 0 : undefined,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.name_surname",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.DEPARTMENT",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.SECTION",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.SUBSECTION",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.POSITION",
                cellWidth: 35,
            },
            {
                phrase: "EXCEL-REPORT.FIRST_START",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.LAST_LEAVE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.STATUS",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.WORKPLAN",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.WORKING_DURATION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.MISSING_WORK",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(data) {
        const row = this._items.find((i) => i.user.id === data.userId);
        const userOrganizationUnits = this._organizationUnitsOfUsers[row.user.id] ?? [];
        const position = this._positionsOfUsers[row.user.id];
        let firstIn = "";
        let lastOut = "";
        if (!row) {
            return;
        }
        let colIndex = 1;
        let isDummyAccessLogForFirstIn = false;
        if (row.entranceDateTime) {
            if (row.entranceAccessLogId) {
                firstIn = (0, report_util_1.formatDateForExcel)(row.entranceDateTime);
            }
            else {
                isDummyAccessLogForFirstIn = true;
                this._isDummyAccessLogAppeared = true;
            }
        }
        let isDummyAccessLogForLastOut = false;
        if (row.exitDateTime) {
            if (row.exitAccessLogId) {
                lastOut = (0, report_util_1.formatDateForExcel)(row.exitDateTime);
            }
            else {
                isDummyAccessLogForLastOut = true;
                this._isDummyAccessLogAppeared = true;
            }
        }
        let rowData = [
            {
                value: row.user.fullname.toUpperCase(),
            },
            {
                value: userOrganizationUnits[2]?.name ?? "",
            },
            {
                value: userOrganizationUnits[3]?.name ?? "",
            },
            {
                value: userOrganizationUnits[4]?.name ?? "",
            },
            {
                value: position ?? "",
            },
            {
                value: isDummyAccessLogForFirstIn ? "__:__" : firstIn,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                backgroundColorCode: isDummyAccessLogForFirstIn ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForFirstIn ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForFirstIn ? "center" : "left",
            },
            {
                value: isDummyAccessLogForLastOut ? "__:__" : lastOut,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                backgroundColorCode: isDummyAccessLogForLastOut ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForLastOut ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForLastOut ? "center" : "left",
            },
        ];
        let statusText;
        let permissionArr = [];
        let permissionText;
        if (row.entranceDateTime) {
            statusText = i18n_1.default.__({ phrase: "EXCEL-REPORT.ARRIVED", locale: this._locale });
        }
        else if (row.missingWorkInMinutes > 0) {
            statusText = i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: this._locale });
        }
        else {
            statusText = "";
        }
        let colorText = undefined;
        let hasPermissionInDay = row.ppermissions && row.ppermissions.length > 0;
        let hasHolidayInDay = row.holidays && row.holidays.length > 0;
        if (hasPermissionInDay || hasHolidayInDay) {
            colorText = generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION;
            row.ppermissions.forEach((permission) => {
                switch (permission?.status) {
                    case app_enums_1.enums.PPermissionStatus.Approved: {
                        permissionArr.push(permission.ppermissionTypeName);
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Rejected: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED_INFO", locale: this._locale }));
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Waiting: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING_INFO", locale: this._locale }));
                        break;
                    }
                }
            });
            row.holidays.forEach((holiday) => {
                permissionArr.push(holiday.holidayName);
            });
        }
        else {
            colorText = this.findColorCodeByPerformance(data);
        }
        permissionText = permissionArr.join("\n");
        if (statusText.length > 0 && permissionText.length > 0) {
            statusText = statusText + " / " + permissionText;
        }
        else if (permissionText.length > 0) {
            statusText = permissionText;
        }
        rowData = [
            ...rowData,
            {
                value: statusText,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: null,
                backgroundColorCode: colorText,
                isTextWhite: true,
            },
            {
                value: row.workPlan.name ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            },
            {
                value: data.physicallyInRegionDuration ? (data.physicallyInRegionDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: data.physicallyInRegionDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: row.missingWorkInMinutes
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
        ];
        colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._generalRowIndex++;
    }
    findColorCodeByPerformance(employeeData) {
        let colorCode = null;
        if (employeeData.expectedWorkDuration == 0) {
            colorCode = null;
        }
        else if (!employeeData.firstStart || employeeData.lateStart > 0) {
            colorCode = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED;
        }
        else {
            if (employeeData.toleratedLateStart && employeeData.toleratedLateStart > 0) {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.TOLERATED;
            }
            else {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.ON_TIME;
            }
        }
        return colorCode;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + "GÜNLÜK ÇALIŞMA DETAYI RAPORU", generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale("tr").format("LL") + " " + this._currentDate.locale("tr").format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE_REPORT_BELONGS_TO", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        let reportDate = (0, moment_1.default)(this.date);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, reportDate.locale("tr").format("LL"), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        ws.mergeCells(startRowIndex + 4, startColIndex + 1, startRowIndex + 4, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 4);
        row.height = filterlength * 25;
    }
}
exports.PENTIDailyWorkingDetailsCustomReport = PENTIDailyWorkingDetailsCustomReport;
async function generateReport(request, locale) {
    let report = new PENTIDailyWorkingDetailsCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
