"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomEndPoints = void 0;
const api_securityhelper_1 = require("../../../api/api.securityhelper");
const api_validatorhelper_1 = require("../../../api/api.validatorhelper");
const app_auth_1 = require("../../../app.auth");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
var jwt = require("jsonwebtoken");
const MKEOrganizationId = "8837fe51-6e89-4d67-abcd-2a576adae750";
function addCustomEndPoints(app) {
    app.post("/custom/mke/accesslogs/sync", async (req, res, next) => {
        const authHeader = req.header("Authorization");
        if (!authHeader) {
            res.status(400).send({
                message: `Authorization header is missing!`,
            });
            return;
        }
        try {
            let decodedToken = jwt.verify(authHeader.substring(7), app_config_1.appConfig.jwtSecret);
            req["auth_token"] = new app_auth_1.UserJwtPayload(decodedToken);
        }
        catch (error) {
            app_logs_1.logger.error(`[mke server-hook.ts] JWT validation error! ${error}`);
            if (error.name === "TokenExpiredError") {
                res.status(401).send({ message: "JWT timed out, please refresh access token!" });
            }
            else if (error.name === "JsonWebTokenError") {
                res.status(400).send({
                    message: `JWT could not be validated, probably malformed, please refresh access token or login again!`,
                });
            }
            else {
                res.status(500).send(error);
            }
            return;
        }
        let reqBody = req.body;
        const validator = new api_validatorhelper_1.ValidatorHelper();
        try {
            validator.validateDate("syncStart", true, reqBody.syncStart);
            validator.validateUUIDArray("accessPointIds", true, reqBody.accessPointIds);
            validator.validateUUIDArray("userIds", true, reqBody.userIds);
            validator.validateUUIDArray("logIds", true, reqBody.logIds);
            validator.validatePaginationRequest("pagination", { take: reqBody.pagination?.take, skip: reqBody.pagination?.skip }, false);
            validator.finalize();
        }
        catch (error) {
            res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                message: "Request validation error",
                validationerrors: validator.errors,
            });
            return;
        }
        let userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
        const accessLogs = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            let bindings = [];
            let query = `
				WITH filtered_logs AS (
					SELECT * FROM "${MKEOrganizationId}"."access_log_sync"
					`;
            let whereConditions = [];
            if (reqBody.logIds && reqBody.logIds.length > 0) {
                bindings.push(reqBody.logIds);
                whereConditions.push(`id = ANY($${bindings.length})`);
            }
            else {
                if (reqBody.accessPointIds && reqBody.accessPointIds.length > 0) {
                    bindings.push(reqBody.accessPointIds);
                    whereConditions.push(`"accessPointId" = ANY($${bindings.length})`);
                }
                if (reqBody.userIds && reqBody.userIds.length > 0) {
                    bindings.push(reqBody.userIds);
                    whereConditions.push(`"userId" = ANY($${bindings.length})`);
                }
                if (reqBody.syncStart) {
                    bindings.push(reqBody.syncStart);
                    whereConditions.push(`"insertion" >= $${bindings.length}`);
                }
            }
            if (whereConditions.length > 0) {
                query += `WHERE (` + whereConditions.join(`) AND (`) + `)`;
            }
            query += `
				)
				SELECT 
					COUNT(*) as total,
					(
						SELECT array_agg(jsonb_build_object(
							'id', fls.id,
							'kullanici', jsonb_build_object(
								'id', uop."userId",
								'sicil', uop."uniqueId",
								'isim', uop."name" || ' ' || uop."surname"
							),
							'nokta', jsonb_build_object(
								'id', fls."accessPointId",
								'isim', acp.name
							),
							'zaman', fls.timestamp,
							'yön', fls.direction
						) ORDER BY fls.insertion ASC)
						FROM (
							SELECT id, "timestamp", "userId", "accessPointId", direction, insertion
							FROM filtered_logs fl`;
            if (reqBody.pagination.skip) {
                bindings.push(reqBody.pagination.skip);
                query += `
							OFFSET $${bindings.length}`;
            }
            if (reqBody.pagination.take) {
                bindings.push(reqBody.pagination.take);
                query += `
							LIMIT $${bindings.length}`;
            }
            query += `
						) fls
						INNER JOIN "${MKEOrganizationId}"."userOrganizationProfiles" uop
							ON uop."userId" = fls."userId"
						INNER JOIN "${MKEOrganizationId}"."accessControlPoints" acp
							ON acp."id" = fls."accessPointId"
					) as items
				FROM filtered_logs;`;
            const logsResult = (await trx.query(query, bindings)).rows[0];
            logsResult.items = logsResult.items ?? [];
            if (logsResult.items.length > 0) {
                await trx.query(`INSERT INTO "${MKEOrganizationId}"."access_log_sync_operations"
						(id, timestamp, operation)
						SELECT ids, now(), $1 FROM unnest($2::uuid[]) ids;`, [app_enums_1.enums.MKESyncOperation.Sent, logsResult.items.map((r) => r.id)]);
            }
            return logsResult;
        }, userId, MKEOrganizationId);
        res.status(200).json({
            pagination: {
                total: accessLogs.total,
                take: reqBody.pagination.take,
                skip: reqBody.pagination.skip,
            },
            items: accessLogs.items,
        });
    });
    app.post("/custom/mke/accesslogs/mark", async (req, res, next) => {
        const authHeader = req.header("Authorization");
        if (!authHeader) {
            res.status(400).send({
                message: `Authorization header is missing!`,
            });
            return;
        }
        try {
            let decodedToken = jwt.verify(authHeader.substring(7), app_config_1.appConfig.jwtSecret);
            req["auth_token"] = new app_auth_1.UserJwtPayload(decodedToken);
        }
        catch (error) {
            app_logs_1.logger.error(`[mke server-hook.ts] JWT validation error! ${error}`);
            if (error.name === "TokenExpiredError") {
                res.status(401).send({ message: "JWT timed out, please refresh access token!" });
            }
            else if (error.name === "JsonWebTokenError") {
                res.status(400).send({
                    message: `JWT could not be validated, probably malformed, please refresh access token or login again!`,
                });
            }
            else {
                res.status(500).send(error);
            }
            return;
        }
        let reqBody = req.body;
        const validator = new api_validatorhelper_1.ValidatorHelper();
        try {
            validator.validateUUIDArray("logIds", true, reqBody.logIds);
            validator.finalize();
        }
        catch (error) {
            res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                message: "Request validation error",
                validationerrors: validator.errors,
            });
            return;
        }
        let userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            const actuallyDeletedIs = await trx.query(`DELETE FROM "${MKEOrganizationId}"."access_log_sync"
					WHERE id = ANY($1::uuid[])
					RETURNING id;`, [reqBody.logIds]);
            await trx.query(`INSERT INTO "${MKEOrganizationId}"."access_log_sync_operations"
					(id, timestamp, operation)
					SELECT ids, now(), $1 FROM unnest($2::uuid[]) ids;`, [app_enums_1.enums.MKESyncOperation.MarkedAsRead, actuallyDeletedIs.rows.map((r) => r.id)]);
        }, userId, MKEOrganizationId);
        res.status(200).send();
    });
}
exports.addCustomEndPoints = addCustomEndPoints;
