"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.initSyncOp = exports.statusInterval = exports.initDb = exports.bidbDBPool = exports.armonDBPool = void 0;
const pg_1 = require("pg");
const app_logs_1 = require("../../../../app.logs");
const dal_manager_1 = require("../../../../dal/dal.manager");
const metu_synchronization_config_1 = __importDefault(require("./metu.synchronization.config"));
const TAG = `[metu-sync][utils.ts]`;
exports.armonDBPool = new pg_1.Pool({
    user: metu_synchronization_config_1.default.PGUSER,
    host: metu_synchronization_config_1.default.PGHOST,
    port: metu_synchronization_config_1.default.PGPORT,
    password: metu_synchronization_config_1.default.PGUSERPASSWORD,
    database: metu_synchronization_config_1.default.PGDATABASE,
    idleTimeoutMillis: 10000,
    max: 100,
});
exports.bidbDBPool = new pg_1.Pool({
    database: metu_synchronization_config_1.default.BIDB_DB_NAME,
    host: metu_synchronization_config_1.default.BIDB_DB_HOST,
    port: metu_synchronization_config_1.default.BIDB_DB_PORT,
    user: metu_synchronization_config_1.default.BIDB_DB_USERNAME,
    password: metu_synchronization_config_1.default.BIDB_DB_PASSWORD,
    idleTimeoutMillis: 10000,
    max: 100,
});
const initDb = async () => {
    const armonClient = await exports.armonDBPool.connect();
    console.table((await armonClient.query(`SELECT count(*) AS migrationcount from public.migration`)).rows);
    armonClient.release();
    const bidbClient = await exports.bidbDBPool.connect();
    let attempt = 0;
    const validityPromise = new Promise(async (resolve, reject) => {
        let validityInterval = null;
        const validityCheckCallback = async () => {
            const validityResult = (await bidbClient.query(`SELECT gecerli_mi AS "validity", guncellenme_tarihi AS "updateTime" from metubidb.gecerlilik`)).rows;
            console.table(validityResult);
            if (validityResult && validityResult[0] && validityResult[0].validity) {
                clearInterval(validityInterval);
                resolve();
                return;
            }
            else {
                attempt++;
                if (attempt > 60) {
                    app_logs_1.logger.error(` METU BIDB database has not been valid for 10 minutes, aborting sync`);
                    clearInterval(validityInterval);
                    reject();
                }
            }
            return validityCheckCallback;
        };
        validityInterval = setInterval(validityCheckCallback, 10000);
    });
    try {
        await validityPromise;
        bidbClient.release();
    }
    catch (error) {
        bidbClient.release();
        throw new Error(`Error while checking if BIDB database is ready for sync`);
    }
};
exports.initDb = initDb;
exports.statusInterval = setInterval(() => {
    for (const [key, value] of Object.entries(process.memoryUsage())) {
        app_logs_1.logger.info(`Memory usage by ${key}, ${value / 1000000}MB `);
    }
}, 30000);
const initSyncOp = async () => {
    try {
        app_logs_1.logger.info(`${TAG}Trying to connect both databases!`);
        await (0, exports.initDb)();
        await dal_manager_1.dbManager.init({
            environment: "development",
            log: null,
            main: {
                backup: null,
                host: metu_synchronization_config_1.default.PGHOST,
                name: metu_synchronization_config_1.default.PGDATABASE,
                password: metu_synchronization_config_1.default.PGUSERPASSWORD,
                port: metu_synchronization_config_1.default.PGPORT,
                username: metu_synchronization_config_1.default.PGUSER,
            },
            redis: {
                host: metu_synchronization_config_1.default.REDIS_HOST,
                port: metu_synchronization_config_1.default.REDIS_PORT,
            },
            logDirectory: metu_synchronization_config_1.default.LOG_DIRECTORY,
        }, true, false, false, false);
        app_logs_1.logger.info(`${TAG}Both database connections are successful!`);
    }
    catch (err) {
        app_logs_1.logger.error(`${TAG}Error while initializing db connections`);
        app_logs_1.logger.error(err);
    }
};
exports.initSyncOp = initSyncOp;
