"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTemporaryGraduates = exports.checkArmonUserExistance = exports.getArmonUserIdByUniqueId = exports.clearUserUpdateOperationsFromBIDBDb = exports.getBidbUsersWhomHasDeleteOperations = exports.getBidbUsersWhomHasInsertOrUpdateOperations = exports.getArmonOrganizationUnitIdNamePairs = exports.getArmonUserGroupIdNamePairs = exports.getBIDBUnitNames = exports.getBIDBGroupIds = exports.syncOrganizationUnits = void 0;
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const bidb_1 = require("../models/bidb");
const constants_1 = require("../utils/constants");
const utils_1 = require("../utils/utils");
const syncOrganizationUnits = async (organizationUnitNames) => {
    const bidbUnitNames = organizationUnitNames;
    return await (0, cli_queries_1.systemTransaction)(utils_1.armonDBPool, async (trx) => {
        const { rows: armonUnitsRows, rowCount: armonUnitsRowCount } = await trx.query(`
            SELECT id, name 
            FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
            WHERE "deletedAt" IS NULL;
        `);
        const { rows: organizationUnitTypes, rowCount: organizationUnitTypeRowCount } = await trx.query(`
			SELECT oudlv.id, oudlv."value" FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" AS oudlv
			INNER JOIN "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedLists}" AS oudl
				ON oudl.id = oudlv."userDefinedListId"
			WHERE oudl.type = $1
		`, [app_enums_1.enums.OrganizationUserDefinedListType.OrganizationUnitType]);
        let targetOrganizationUnitType = organizationUnitTypes.find((f) => JSON.parse(f.value).caption_tr === "Birim");
        if (!targetOrganizationUnitType) {
            app_logs_1.logger.warn(`Organization Unit Type "Birim" is not found in user defined list! "${JSON.parse(organizationUnitTypes[0].value).caption_tr}" will be used for new units!`);
        }
        const bidbUnitsMissingFromArmon = bidbUnitNames.filter((f) => !armonUnitsRows.map((au) => au.name).includes(f));
        app_logs_1.logger.info(` There are ${bidbUnitsMissingFromArmon.length} organization units missing from armon.\nThese units will be created at "Universite -> Birimler ve Bölümler" hierarchy! List can be seen below!`);
        app_logs_1.logger.info(JSON.stringify(bidbUnitsMissingFromArmon, null, 4));
        const { rows: armonOrganizationUnitIdNamePairs, rowCount } = await trx.query(`
			INSERT INTO "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			(id, "createdAt", "updatedAt", "deletedAt", "organizationId", name, "parentId", "ancestorIds", "shortCode", "managerUserId", "managerUserOrganizationId", "typeId", settings)
			SELECT uuid_generate_v4(), now(), now(), null, $1, u.nameval, $3, $4, null, null, null, $5, null
			FROM UNNEST ($2::text[]) AS u (nameval)
			RETURNING id, name, "parentId", "ancestorIds"
		`, [constants_1.METU_ORGANIZATION_ID, bidbUnitsMissingFromArmon, constants_1.METU_UNIT_DEFAULT_PARENT_ID, constants_1.METU_UNIT_DEFAULT_ANCESTOR_IDS, targetOrganizationUnitType.id]);
        app_logs_1.logger.info(`Units with info below added to Armon database : ${JSON.stringify(armonOrganizationUnitIdNamePairs, null, 4)}`);
        return armonOrganizationUnitIdNamePairs;
    });
};
exports.syncOrganizationUnits = syncOrganizationUnits;
const getBIDBGroupIds = async () => {
    const { rows: userGroupNames, rowCount } = await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx) => {
        return trx.query(`
			SELECT DISTINCT(id)
			FROM metubidb.${bidb_1.metuSyncTablenames.T_Group}
		`);
    });
    return userGroupNames.map((m) => m.id);
};
exports.getBIDBGroupIds = getBIDBGroupIds;
const getBIDBUnitNames = async () => {
    const { rows: organizationUnitNames, rowCount } = await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx) => {
        return trx.query(`
			SELECT DISTINCT (birim)
			FROM metubidb.${bidb_1.metuSyncTablenames.T_UserDetail}
			WHERE birim IS NOT NULL
		`);
    });
    return organizationUnitNames.map((m) => m.birim);
};
exports.getBIDBUnitNames = getBIDBUnitNames;
const getArmonUserGroupIdNamePairs = async (userGroupNames, trx) => {
    const { rows: userGroupIdNamePairs, rowCount } = await trx.query(`
			SELECT id, name 
			FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
			WHERE "deletedAt" IS NULL
				AND name = ANY($1::text[])
		`, [userGroupNames]);
    if (userGroupNames.length !== userGroupIdNamePairs.length) {
        app_logs_1.logger.error(`Users groups not mapped across metu bidb and armon`);
    }
    return userGroupIdNamePairs;
};
exports.getArmonUserGroupIdNamePairs = getArmonUserGroupIdNamePairs;
const getArmonOrganizationUnitIdNamePairs = async (organizationUnitNames, trx) => {
    const { rows: organizationUnitIdNamePairs, rowCount } = await trx.query(`
			SELECT id, name, "parentId", "ancestorIds"
			FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			WHERE "deletedAt" IS NULL
				AND name = ANY($1::text[])
		`, [organizationUnitNames]);
    if (organizationUnitNames.length !== organizationUnitIdNamePairs.length) {
        app_logs_1.logger.error(`Organization units not mapped across metu bidb and armon`);
        const newlyAddedOrganizationUnitsIdNamePairs = await (0, exports.syncOrganizationUnits)(organizationUnitNames);
        if (organizationUnitNames.length !== organizationUnitIdNamePairs.length + newlyAddedOrganizationUnitsIdNamePairs.length) {
            app_logs_1.logger.error(`Unable to sync organization units between METU BIDB and ARMON database!`);
        }
        else {
            return organizationUnitIdNamePairs.concat(newlyAddedOrganizationUnitsIdNamePairs);
        }
    }
    return organizationUnitIdNamePairs;
};
exports.getArmonOrganizationUnitIdNamePairs = getArmonOrganizationUnitIdNamePairs;
const getBidbUsersWhomHasInsertOrUpdateOperations = async (trx) => {
    const userResult = await trx.query(`
		SELECT k.id, k.ad, k.soyad, k.eposta,
		JSONB_AGG(
			DISTINCT(
				JSONB_BUILD_OBJECT(
					'groupId', kr.rol_id,
					'number', kr.rol_data,
					'username', kr.kullanici_kodu
				)
			)
		) as "groups",
		COALESCE( 
			JSONB_AGG (
				DISTINCT(
					JSONB_BUILD_OBJECT (
						'credential', ky.yetki_degeri,
						'unit', ky.birim,
						'number', ky.rol_data,
						'groupId', ky.rol_id
					)
				)
			) FILTER (WHERE ky.kisi_id IS NOT NULL), '[]'
		) AS "details",
		COALESCE (jsonb_agg (
			distinct(kd.*)
		) FILTER (WHERE kd.id IS NOT NULL), '[]') AS "user_updates",
		COALESCE (jsonb_agg (
			distinct(krd.*)
		) FILTER (WHERE krd.kisi_id IS NOT NULL), '[]') AS "role_updates",
		COALESCE (jsonb_agg (
			distinct(kyd.*)
		) FILTER (WHERE kyd.kisi_id IS NOT NULL), '[]') AS "detail_updates",
		FALSE AS "updateStatus" 
		FROM metubidb.kisi k
		INNER JOIN metubidb.kisi_rol kr
			ON k.id = kr.kisi_id
		LEFT JOIN metubidb.kisi_yetki ky
			ON ky.kisi_id = kr.kisi_id AND ky.rol_id = kr.rol_id
		LEFT JOIN metubidb.kisi_degisiklik kd
			ON k.id = kd.id
		LEFT JOIN metubidb.kisi_rol_degisiklik krd
			ON k.id = krd.kisi_id
		LEFT JOIN metubidb.kisi_yetki_degisiklik kyd
			ON k.id = kyd.kisi_id
		WHERE (kd.guncellenme_tarihi IS NOT NULL OR krd.guncellenme_tarihi IS NOT NULL OR kyd.guncellenme_tarihi IS NOT NULL)
		GROUP BY k.id, k.ad, k.soyad, k.eposta
		ORDER BY k.id ASC
	`);
    return userResult.rows;
};
exports.getBidbUsersWhomHasInsertOrUpdateOperations = getBidbUsersWhomHasInsertOrUpdateOperations;
const getBidbUsersWhomHasDeleteOperations = async (trx) => {
    const deletedUserResult = await trx.query(`
	SELECT COALESCE (jsonb_agg (
			distinct(kd.*)
		) FILTER (WHERE kd.id IS NOT NULL), '[]') AS user_updates,
		COALESCE (jsonb_agg (
			distinct(krd.*)
		) FILTER (WHERE krd.kisi_id IS NOT NULL), '[]') AS role_updates,
		COALESCE (jsonb_agg (
			distinct(kyd.*)
		) FILTER (WHERE kyd.kisi_id IS NOT NULL), '[]') AS detail_updates,
		FALSE AS "updateStatus"
	FROM metubidb.kisi_degisiklik kd
	LEFT JOIN metubidb.kisi_rol_degisiklik krd
		ON krd.kisi_id = kd.id
	LEFT JOIN metubidb.kisi_yetki_degisiklik kyd
		ON krd.kisi_id = kyd.kisi_id AND krd.rol_id = kyd.rol_id
	WHERE kd.islem_durumu = 0
	GROUP BY kd.id, kd.ad, kd.soyad, kd.eposta
	ORDER BY kd.id
	`);
    return deletedUserResult.rows;
};
exports.getBidbUsersWhomHasDeleteOperations = getBidbUsersWhomHasDeleteOperations;
const clearUserUpdateOperationsFromBIDBDb = async (trx, users) => {
    const user_update_params = [];
    const role_update_params = [];
    const detail_update_params = [];
    for (const user of users) {
        for (const user_update of user.user_updates) {
            user_update_params.push(user_update);
        }
        for (const role_update of user.role_updates) {
            role_update_params.push(role_update);
        }
        for (const detail_update of user.detail_updates) {
            detail_update_params.push(detail_update);
        }
    }
    await trx.query(`
		DELETE FROM metubidb.kisi_degisiklik
		WHERE (id, islem_durumu, guncellenme_tarihi) IN (
			SELECT *
			FROM UNNEST ($1::TEXT[], $2::INTEGER[], $3::TIMESTAMP WITHOUT TIME ZONE[])
		)
	`, [user_update_params.map((m) => m.id), user_update_params.map((m) => m.islem_durumu), user_update_params.map((m) => m.guncellenme_tarihi)]);
    await trx.query(`
		DELETE FROM metubidb.kisi_rol_degisiklik
		WHERE (kisi_id, rol_id, islem_durumu, guncellenme_tarihi) IN (
			SELECT *
			FROM UNNEST ($1::TEXT[], $2::INTEGER[], $3::INTEGER[], $4::TIMESTAMP WITHOUT TIME ZONE[])
		)
	`, [
        role_update_params.map((m) => m.kisi_id),
        role_update_params.map((m) => m.rol_id),
        role_update_params.map((m) => m.islem_durumu),
        role_update_params.map((m) => m.guncellenme_tarihi),
    ]);
    await trx.query(`
		DELETE FROM metubidb.kisi_yetki_degisiklik
		WHERE (kisi_id, rol_id, islem_durumu, guncellenme_tarihi) IN (
			SELECT *
			FROM UNNEST ($1::TEXT[], $2::INTEGER[], $3::INTEGER[], $4::TIMESTAMP WITHOUT TIME ZONE[])
		)
	`, [
        detail_update_params.map((m) => m.kisi_id),
        detail_update_params.map((m) => m.rol_id),
        detail_update_params.map((m) => m.islem_durumu),
        detail_update_params.map((m) => m.guncellenme_tarihi),
    ]);
    return;
};
exports.clearUserUpdateOperationsFromBIDBDb = clearUserUpdateOperationsFromBIDBDb;
const getArmonUserIdByUniqueId = async (trx, uniqueId) => {
    const { rows: userIdResultRows, rowCount: userIdResultRowCount } = await trx.query(`
		SELECT "userId" FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
		WHERE "uniqueId" = $1
			AND "deletedAt" IS NULL
	`);
    if (userIdResultRowCount !== 1) {
        app_logs_1.logger.error(`There are ${userIdResultRowCount} for uniqueId ${uniqueId}! Must be checked manually, aborting...`);
        return null;
    }
    else {
        return userIdResultRows[0].userId;
    }
};
exports.getArmonUserIdByUniqueId = getArmonUserIdByUniqueId;
const checkArmonUserExistance = async (trx, uniqueId) => {
    const dbCheckResult = await trx.query(`
		SELECT uop."userId", uop."uniqueId", uo."isDisabled"
		FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
		INNER JOIN "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			ON uo."userId" = uop."userId" AND uo.id = uop."userOrganizationId"
		WHERE uop."uniqueId" = $1
	`, [uniqueId]);
    if (dbCheckResult.rowCount) {
        if (dbCheckResult.rowCount === 1) {
            return dbCheckResult.rows[0];
        }
        else {
            app_logs_1.logger.error(`There are ${dbCheckResult.rowCount} entries for uniqueId ${uniqueId} at Armon database!`);
            app_logs_1.logger.error(`This is not expected and must be checked by development team! Skipping user`);
            throw new Error(`There are ${dbCheckResult.rowCount} entries for uniqueId ${uniqueId} at Armon database!`);
        }
    }
    return null;
};
exports.checkArmonUserExistance = checkArmonUserExistance;
const getTemporaryGraduates = async (trx) => {
    const temporaryGraduateDbResult = await trx.query(`
		SELECT uguo."createdAt", uguo."updatedAt", uo."userId", CURRENT_DATE - DATE(uguo."createdAt") AS "dateDiff"
		FROM "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
		INNER JOIN "${constants_1.METU_ORGANIZATION_ID}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			ON uguo."userOrganizationId" = uo.id AND uguo."deletedAt" IS NULL AND uo."isDisabled" IS FALSE
		WHERE uguo."userGroupId" = $1
	`, [constants_1.ARMON_USER_GROUP_IDS.YeniMezun]);
    return temporaryGraduateDbResult.rows;
};
exports.getTemporaryGraduates = getTemporaryGraduates;
