"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disableTemporaryGraduates = exports.updateExistingBidbUser = exports.importNewBidbUser = exports.getArmonApplicationUsernameFromBidbInfo = exports.convertHexadecimalCardDatatoDecimal = exports.generateCredentialNote = exports.findBidbUserRoleId = void 0;
const app_logs_1 = require("../../../../app.logs");
const dal_manager_1 = require("../../../../dal/dal.manager");
const terminal_1 = require("../../../../lib/es/models/terminal");
const bidb_1 = require("../models/bidb");
const constants_1 = require("../utils/constants");
const dal_constants_1 = require("../../../../dal/dal.constants");
const app_enums_1 = require("../../../../app.enums");
const findBidbUserRoleId = (bidbUser) => {
    if (bidbUser.groups.find((f) => f.groupId === bidb_1.BIDBGroupId.RETIRED_PERSONNEL) &&
        !bidbUser.groups.find((f) => (f.groupId === bidb_1.BIDBGroupId.PERSONNEL && f.username) ||
            (f.groupId === bidb_1.BIDBGroupId.TEMPORARY_PERSONNEL && f.username) ||
            (f.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL && f.username) ||
            (f.groupId === bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL && f.username))) {
        return constants_1.METU_ORGANIZATION_WIDE_RETIRED_PERSONNEL_ROLE_ID;
    }
    else {
        return constants_1.METU_ORGANIZATION_WIDE_STANDARD_USER_ROLE_ID;
    }
};
exports.findBidbUserRoleId = findBidbUserRoleId;
const generateCredentialNote = (groupId, unit, number) => {
    return `Grup:${bidb_1.BIDBGroupIdArmonGroupNameMapping[groupId] ?? "N/A"},\nBirim:${unit ?? "N/A"},\n(Personel/Öğrenci) No:${number ?? "N/A"}`;
};
exports.generateCredentialNote = generateCredentialNote;
const convertHexadecimalCardDatatoDecimal = (hexData) => {
    const splittedCardData = hexData.match(/.{1,2}/g);
    const reversedCardData = splittedCardData.reverse().join("");
    return parseInt(reversedCardData, 16).toString();
};
exports.convertHexadecimalCardDatatoDecimal = convertHexadecimalCardDatatoDecimal;
const getArmonApplicationUsernameFromBidbInfo = (bidbGroups, uniqueId, currentUsername) => {
    const result = {
        username: null,
        password: null,
    };
    if (bidbGroups.some((g) => g.groupId === bidb_1.BIDBGroupId.RETIRED_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.TEMPORARY_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL)) {
        let targetUserName = bidbGroups.find((g) => g.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL && g.username)?.username ??
            bidbGroups.find((g) => g.groupId === bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL && g.username)?.username ??
            bidbGroups.find((g) => g.groupId === bidb_1.BIDBGroupId.PERSONNEL && g.username)?.username ??
            bidbGroups.find((g) => g.groupId === bidb_1.BIDBGroupId.TEMPORARY_PERSONNEL && g.username)?.username ??
            bidbGroups.find((g) => g.groupId === bidb_1.BIDBGroupId.RETIRED_PERSONNEL && g.username)?.username;
        if (targetUserName) {
            if (!currentUsername) {
                app_logs_1.logger.info(`Bidb user with unique id ${uniqueId} is missing account, new account is created!`);
                result.username = targetUserName;
                result.password = Math.random().toString(36).slice(-8);
            }
            else {
                if (!bidbGroups.map((g) => g.username).includes(currentUsername)) {
                    app_logs_1.logger.info(`Bidb user with unique id ${uniqueId}'s account old username ${currentUsername} will be updated as ${targetUserName}!`);
                    result.username = targetUserName;
                    result.password = Math.random().toString(36).slice(-8);
                }
                else {
                    app_logs_1.logger.info(`Bidb user with unique id ${uniqueId} already has username ${currentUsername}, no change needed`);
                    return null;
                }
            }
        }
        else if (!currentUsername) {
            app_logs_1.logger.info(`Bidb user with unique id ${uniqueId} does not have an account, no change needed`);
            return null;
        }
    }
    if (currentUsername && !result.username) {
        app_logs_1.logger.info(`Bidb user with unique id ${uniqueId}'s account with username ${currentUsername} will be disabled`);
    }
    return result;
};
exports.getArmonApplicationUsernameFromBidbInfo = getArmonApplicationUsernameFromBidbInfo;
const importNewBidbUser = async (bidbUser, armonOrganizationUnits, armonUserGroups, trx) => {
    let organizationWideRoleId = null;
    let userAccount = null;
    if (!bidbUser.groups.map((m) => m.groupId).some((s) => [bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL, bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL, bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL].includes(s)) &&
        bidbUser.groups.map((m) => m.groupId).includes(bidb_1.BIDBGroupId.RETIRED_PERSONNEL)) {
        organizationWideRoleId = constants_1.METU_ORGANIZATION_WIDE_RETIRED_PERSONNEL_ROLE_ID;
    }
    else {
        organizationWideRoleId = constants_1.METU_ORGANIZATION_WIDE_STANDARD_USER_ROLE_ID;
    }
    if (bidbUser.groups.some((g) => (g.groupId === bidb_1.BIDBGroupId.PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.RETIRED_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.TEMPORARY_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL ||
        g.groupId === bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL) &&
        g.username)) {
        userAccount = (0, exports.getArmonApplicationUsernameFromBidbInfo)(bidbUser.groups, bidbUser.id, null);
    }
    if (bidbUser.groups.map((m) => m.groupId).includes(bidb_1.BIDBGroupId.GRADUATE) && bidbUser.groups.filter((f) => f.groupId !== bidb_1.BIDBGroupId.GRADUATE)?.length) {
        app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} removed from graduate user group implicitly in order to bypass access rules!`);
        bidbUser.groups = bidbUser.groups.filter((f) => f.groupId !== bidb_1.BIDBGroupId.GRADUATE);
    }
    try {
        let accessRights = constants_1.METU_ACCESS_CONTROL_POINT_IDS_WITH_MIFARE_AUTHENTICATION_METHODS.map((m) => {
            return {
                accessControlPointId: m,
                access: true,
                config: false,
                grant: false,
                read: false,
                remoteAccess: false,
                snapshot: false,
            };
        });
        if (bidbUser.groups.find((f) => f.groupId === bidb_1.BIDBGroupId.PERSONNEL || f.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL || bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL)) {
            app_logs_1.logger.info(`Mobile KOS device access rights added for user with unique id ${bidbUser.id}`);
            accessRights.push(...constants_1.METU_ACCESS_CONTROL_POINT_IDS_FOR_PERSONNEL_SERVICE.map((m) => {
                return {
                    accessControlPointId: m,
                    access: true,
                    config: false,
                    grant: false,
                    read: false,
                    remoteAccess: false,
                    snapshot: false,
                };
            }));
        }
        await dal_manager_1.dbManager.accessIdentity.insertIdentity(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, {
            insertIdentity: {
                organizationProfile: {
                    isDisabled: false,
                    locale: "tr",
                    name: bidbUser.ad,
                    roleId: organizationWideRoleId,
                    surname: bidbUser.soyad,
                    uniqueId: bidbUser.id,
                    email: bidbUser.eposta,
                    extensionFields: {
                        tcPass: bidbUser.id,
                        numbers: bidbUser.groups.filter((f) => f.number !== null && f.number !== undefined).map((r) => parseInt(r.number)) ?? [],
                    },
                },
                accessRights,
                credentials: bidbUser.details.map((m) => {
                    return {
                        type: terminal_1.CredentialType.MiFare,
                        data: (0, exports.convertHexadecimalCardDatatoDecimal)(m.credential),
                        note: (0, exports.generateCredentialNote)(m.groupId, m.unit, m.number),
                    };
                }) ?? [],
                organizationUnits: bidbUser.details
                    .filter((f) => f.unit !== null && f.unit !== undefined)
                    .map((m) => {
                    if (m.unit) {
                        return {
                            extraPermissions: null,
                            organizationUnitId: armonOrganizationUnits.find((f) => f.name === m.unit).id,
                            roleId: constants_1.METU_UNIT_STANDARD_USER_ROLE_ID,
                        };
                    }
                }) ?? [],
                userGroupIds: bidbUser.groups.map((m) => armonUserGroups.find((f) => f.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[m.groupId]).id) ?? [],
                userAccount,
            },
            notifyUsersForChange: false,
        }, trx);
        bidbUser.updateStatus = true;
        app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} imported successfully into Armon database!`);
    }
    catch (error) {
        app_logs_1.logger.error(`Error while importing bidb user with unique id ${bidbUser.id}!`);
        app_logs_1.logger.error(error);
    }
};
exports.importNewBidbUser = importNewBidbUser;
const updateExistingBidbUser = async (bidbUser, armonOrganizationUnits, armonUserGroups, armonUserId, trx) => {
    const armonUser = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, {
        userId: armonUserId,
        hasIdentityFullWrite: true,
        hasOrganizationWideRight: true,
        hasOrganizationWideRightToSeePassiveUsers: true,
        trx,
    });
    app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} needs to be updated with following information ${JSON.stringify(bidbUser, null, 4)}`);
    try {
        const updateIdentity = {
            id: armonUserId,
            organizationProfile: armonUser.organizationProfile,
            addedUserGroupIds: [],
            removedUserGroupIds: [],
            addedOrganizationUnits: [],
            updatedOrganizationUnits: [],
            removedOrganizationUnitIds: [],
            addedCredentials: [],
            updatedCredentials: [],
            removedCredentialIds: [],
            addedAccessRights: [],
            removedAccessControlPointIds: [],
            updatedAccessRights: [],
            userAccount: null,
        };
        updateIdentity.organizationProfile.name = bidbUser.ad;
        updateIdentity.organizationProfile.surname = bidbUser.soyad;
        if (bidbUser.eposta) {
            updateIdentity.organizationProfile.email = bidbUser.eposta;
        }
        if (!updateIdentity.organizationProfile.extensionFields) {
            updateIdentity.organizationProfile.extensionFields = {
                tcPass: null,
                numbers: [],
            };
        }
        else {
            if (!updateIdentity.organizationProfile.extensionFields.numbers) {
                updateIdentity.organizationProfile.extensionFields.numbers = [];
            }
        }
        updateIdentity.organizationProfile.extensionFields.tcPass = bidbUser.id;
        if (bidbUser.isDisabled) {
            const armonUserRemainingGroups = armonUser.userGroups
                .filter((f) => !Object.values(bidb_1.BIDBGroupIdArmonGroupNameMapping).includes(f.name))
                .filter((f) => !constants_1.METU_DEPENDENT_GROUPS_TO_IGNORE.includes(f.id));
            const armonUserRemainingCredentials = armonUser.credentials.filter((f) => f.type === terminal_1.CredentialType.MiFare && (!f.note || f.note.search("Grup:") < 0));
            const armonUserRemainingUnits = armonUser.organizationUnits.filter((f) => !armonOrganizationUnits.map((m) => m.id).includes(f.organizationUnitId));
            if (armonUserRemainingGroups.length || armonUserRemainingCredentials.length || armonUserRemainingUnits.length) {
                app_logs_1.logger.info(`BIDB User with unique id ${bidbUser.id} is not going to be disabled!`);
                app_logs_1.logger.info(`Remaining groups: ${JSON.stringify(armonUserRemainingGroups, null, 4)}`);
                app_logs_1.logger.info(`Remaining credentials: ${JSON.stringify(armonUserRemainingCredentials, null, 4)}`);
                app_logs_1.logger.info(`Remaining units: ${JSON.stringify(armonUserRemainingUnits, null, 4)}`);
                if (armonUser.userGroups.find((f) => f.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.STUDENT]) &&
                    (armonUser.userGroups.find((f) => f.id === constants_1.ARMON_USER_GROUP_IDS.YerelSari) || armonUser.userGroups.find((f) => f.id === constants_1.ARMON_USER_GROUP_IDS.UyduKahverengi)) &&
                    armonUser.credentials.find((f) => f.type === app_enums_1.enums.CredentialType.UHFRfid)) {
                    updateIdentity.addedUserGroupIds.push(constants_1.ARMON_USER_GROUP_IDS.YeniMezun);
                }
            }
            else {
                updateIdentity.organizationProfile.isDisabled = true;
                updateIdentity.organizationProfile.extensionFields.numbers = [];
                updateIdentity.organizationProfile.email = null;
                await dal_manager_1.dbManager.accessIdentity.updateIdentity(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, trx, updateIdentity, false);
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} is disabled!`);
                return;
            }
        }
        updateIdentity.organizationProfile.isDisabled = false;
        if (armonUser.organizationProfile.roleId === constants_1.METU_ORGANIZATION_WIDE_STANDARD_USER_ROLE_ID ||
            armonUser.organizationProfile.roleId === constants_1.METU_ORGANIZATION_WIDE_RETIRED_PERSONNEL_ROLE_ID) {
            const targetRoleId = (0, exports.findBidbUserRoleId)(bidbUser);
            if (armonUser.organizationProfile.roleId !== targetRoleId) {
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} organization wide role is changed! New role id : ${targetRoleId}`);
                updateIdentity.organizationProfile.roleId = targetRoleId;
            }
        }
        const userAccountInfo = (0, exports.getArmonApplicationUsernameFromBidbInfo)(bidbUser.groups, bidbUser.id, armonUser.username);
        if (userAccountInfo) {
            updateIdentity.userAccount = userAccountInfo;
        }
        if (bidbUser.groups.map((m) => m.groupId).includes(bidb_1.BIDBGroupId.GRADUATE) && bidbUser.groups.filter((f) => f.groupId !== bidb_1.BIDBGroupId.GRADUATE)?.length) {
            app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} removed from graduate user group implicitly in order to bypass access rules!`);
            bidbUser.groups = bidbUser.groups.filter((f) => f.groupId !== bidb_1.BIDBGroupId.GRADUATE);
        }
        for (const bidbUserGroup of bidbUser.groups) {
            if (armonUser.userGroups.find((f) => f.id === armonUserGroups.find((g) => g.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidbUserGroup.groupId]).id) ||
                updateIdentity.addedUserGroupIds.includes(armonUserGroups.find((g) => g.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidbUserGroup.groupId]).id)) {
            }
            else {
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} added to user group ${bidb_1.BIDBGroupIdArmonGroupNameMapping[bidbUserGroup.groupId]}!`);
                updateIdentity.addedUserGroupIds.push(armonUserGroups.find((g) => g.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidbUserGroup.groupId]).id);
            }
        }
        let userRemovedFromPersonnelGroups = false;
        for (const armonUserGroup of armonUser.userGroups.filter((f) => Object.values(bidb_1.BIDBGroupIdArmonGroupNameMapping).includes(f.name))) {
            if (!bidbUser.groups.find((f) => bidb_1.BIDBGroupIdArmonGroupNameMapping[f.groupId] === armonUserGroup.name) && !updateIdentity.removedUserGroupIds.includes(armonUserGroup.id)) {
                updateIdentity.removedUserGroupIds.push(armonUserGroup.id);
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} removed from user group ${armonUserGroup.name}!`);
                if (armonUserGroup.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.PERSONNEL] ||
                    armonUserGroup.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL] ||
                    armonUserGroup.name === bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL]) {
                    userRemovedFromPersonnelGroups = true;
                }
            }
            else {
            }
        }
        if (userRemovedFromPersonnelGroups &&
            !bidbUser.groups
                .map((m) => bidb_1.BIDBGroupIdArmonGroupNameMapping[m.groupId])
                .some((s) => [
                bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.PERSONNEL],
                bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL],
                bidb_1.BIDBGroupIdArmonGroupNameMapping[bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL],
            ].includes(s))) {
            app_logs_1.logger.info(`Mobile KOS device access rights removed from user with unique id ${bidbUser.id}`);
            updateIdentity.removedAccessControlPointIds.push(...constants_1.METU_ACCESS_CONTROL_POINT_IDS_FOR_PERSONNEL_SERVICE.filter((f) => armonUser.accessRights.map((m) => m.accessControlPointId).includes(f)));
        }
        for (const bidbUserGroup of bidbUser.groups) {
            if (bidbUserGroup.number && !updateIdentity.organizationProfile.extensionFields.numbers.includes(parseInt(bidbUserGroup.number))) {
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id}'s student / personnel number ${bidbUserGroup.number} is added to profile!`);
                updateIdentity.organizationProfile.extensionFields.numbers.push(parseInt(bidbUserGroup.number));
            }
        }
        for (const armonUserNumber of armonUser.organizationProfile.extensionFields.numbers) {
            if (!bidbUser.groups.find((f) => {
                try {
                    if (parseInt(f.number) === armonUserNumber) {
                        return true;
                    }
                    else {
                        return false;
                    }
                }
                catch (err) {
                    app_logs_1.logger.error(`Error while converting bidb user with unique id ${bidbUser.id}'s student / personnel number ${f.number} to integer`);
                }
            })) {
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id}'s old student / personnel number ${armonUserNumber} is removed from profile!`);
                updateIdentity.organizationProfile.extensionFields.numbers.splice(updateIdentity.organizationProfile.extensionFields.numbers.indexOf(armonUserNumber), 1);
            }
        }
        for (const bidbUserDetail of bidbUser.details) {
            if (bidbUserDetail.unit) {
                if (armonUser.organizationUnits.find((f) => f.organizationUnitId === armonOrganizationUnits.find((g) => g.name === bidbUserDetail.unit).id)) {
                }
                else {
                    app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} added to organization unit ${bidbUserDetail.unit}!`);
                    updateIdentity.addedOrganizationUnits.push({
                        extraPermissions: [],
                        organizationUnitId: armonOrganizationUnits.find((g) => g.name === bidbUserDetail.unit).id,
                        roleId: constants_1.METU_UNIT_STANDARD_USER_ROLE_ID,
                    });
                }
            }
        }
        if (armonUser.organizationProfile.roleId === constants_1.METU_ORGANIZATION_WIDE_STANDARD_USER_ROLE_ID ||
            armonUser.organizationProfile.roleId === constants_1.METU_ORGANIZATION_WIDE_RETIRED_PERSONNEL_ROLE_ID) {
            for (const armonOrganizationUnitInHierarchy of armonUser.organizationUnits.filter((f) => armonOrganizationUnits.find((aouf) => aouf.id === f.organizationUnitId))) {
                if (!bidbUser.details.find((f) => f.unit === armonOrganizationUnitInHierarchy.name)) {
                    updateIdentity.removedOrganizationUnitIds.push(armonOrganizationUnitInHierarchy.organizationUnitId);
                    app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} removed from organization unit ${armonOrganizationUnitInHierarchy.name}!`);
                }
                else {
                }
            }
        }
        for (const bidbUserDetail of bidbUser.details) {
            if (bidbUserDetail.credential) {
                const convertedCredentialData = (0, exports.convertHexadecimalCardDatatoDecimal)(bidbUserDetail.credential);
                let userCredential = armonUser.credentials.find((f) => f.data === convertedCredentialData && f.type === terminal_1.CredentialType.MiFare);
                if (userCredential) {
                    let credentialNote = (0, exports.generateCredentialNote)(bidbUserDetail.groupId, bidbUserDetail.unit, bidbUserDetail.number);
                    if (userCredential.note !== credentialNote) {
                        app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id}'s credential ${userCredential.data} is updated with note ${credentialNote}!`);
                        updateIdentity.updatedCredentials.push({
                            id: userCredential.id,
                            credentialNumber: userCredential.credentialNumber,
                            expirationUtc: userCredential.expirationUtc,
                            note: credentialNote,
                            customFields: userCredential.customFields,
                            extensionFields: userCredential.extensionFields,
                            groupNumber: userCredential.groupNumber,
                        });
                    }
                }
                else {
                    app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id}'s new credential ${convertedCredentialData} is added!`);
                    updateIdentity.addedCredentials.push({
                        data: convertedCredentialData,
                        type: terminal_1.CredentialType.MiFare,
                        credentialNumber: null,
                        customFields: null,
                        expirationUtc: null,
                        extensionFields: null,
                        groupNumber: null,
                        note: (0, exports.generateCredentialNote)(bidbUserDetail.groupId, bidbUserDetail.unit, bidbUserDetail.number),
                    });
                }
            }
        }
        for (const armonUserCredential of armonUser.credentials.filter((f) => f.type === dal_constants_1.DalConstants.CredentialType.MiFare && f.note?.includes("Grup:"))) {
            if (!bidbUser.details.find((f) => (0, exports.convertHexadecimalCardDatatoDecimal)(f.credential) === armonUserCredential.data)) {
                updateIdentity.removedCredentialIds.push(armonUserCredential.id);
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id}'s old credential ${armonUserCredential.data} is removed!`);
            }
            else {
            }
        }
        const accessRightMissingAccessControlPointIds = constants_1.METU_ACCESS_CONTROL_POINT_IDS_WITH_MIFARE_AUTHENTICATION_METHODS.filter((f) => !armonUser.accessRights.map((m) => m.accessControlPointId).includes(f));
        if (bidbUser.groups.find((f) => f.groupId === bidb_1.BIDBGroupId.PERSONNEL || f.groupId === bidb_1.BIDBGroupId.ACADEMIC_PERSONNEL || f.groupId === bidb_1.BIDBGroupId.ADMINISTRATIVE_PERSONNEL)) {
            accessRightMissingAccessControlPointIds.push(...constants_1.METU_ACCESS_CONTROL_POINT_IDS_FOR_PERSONNEL_SERVICE.filter((f) => !armonUser.accessRights.map((m) => m.accessControlPointId).includes(f)));
            app_logs_1.logger.info(`Missing Mobile KOS device access rights added for user with unique id ${bidbUser.id}`);
        }
        for (const acpId of accessRightMissingAccessControlPointIds) {
            app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} missing access right for point ${acpId}. New access right is added!`);
            updateIdentity.addedAccessRights.push({
                accessControlPointId: acpId,
                access: true,
                config: false,
                grant: false,
                read: false,
                remoteAccess: false,
                snapshot: false,
            });
        }
        app_logs_1.logger.debug(`Bidb user with information below will be updated! ${JSON.stringify(updateIdentity, null, 4)}`);
        await dal_manager_1.dbManager.accessIdentity.updateIdentity(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, trx, updateIdentity, false);
        app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} updated successfully!`);
        return;
    }
    catch (error) {
        app_logs_1.logger.error(`Error while updating bidb user with unique id ${bidbUser.id}!`);
        app_logs_1.logger.error(error);
    }
};
exports.updateExistingBidbUser = updateExistingBidbUser;
const disableTemporaryGraduates = async (trx, armonUserId) => {
    const armonUser = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, {
        userId: armonUserId,
        hasIdentityFullWrite: true,
        hasOrganizationWideRight: true,
        hasOrganizationWideRightToSeePassiveUsers: true,
        trx,
    });
    const updateIdentity = {
        id: armonUserId,
        organizationProfile: armonUser?.organizationProfile,
        addedUserGroupIds: [],
        removedUserGroupIds: [],
        addedOrganizationUnits: [],
        updatedOrganizationUnits: [],
        removedOrganizationUnitIds: [],
        addedCredentials: [],
        updatedCredentials: [],
        removedCredentialIds: [],
        addedAccessRights: [],
        removedAccessControlPointIds: [],
        updatedAccessRights: [],
        userAccount: null,
    };
    if (armonUser?.userGroups.find((f) => ![constants_1.ARMON_USER_GROUP_IDS.UyduKahverengi, constants_1.ARMON_USER_GROUP_IDS.YerelSari, constants_1.ARMON_USER_GROUP_IDS.YeniMezun].includes(f.id))) {
        updateIdentity.removedUserGroupIds.push(constants_1.ARMON_USER_GROUP_IDS.YeniMezun);
        await dal_manager_1.dbManager.accessIdentity.updateIdentity(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, trx, updateIdentity, false);
        app_logs_1.logger.info(`Graduate with temporary access with unique id ${armonUser.organizationProfile.uniqueId} is removed from related user group!
		User will not be disabled because they are in [${armonUser.userGroups
            .filter((f) => ![constants_1.ARMON_USER_GROUP_IDS.UyduKahverengi, constants_1.ARMON_USER_GROUP_IDS.YerelSari, constants_1.ARMON_USER_GROUP_IDS.YeniMezun].includes(f.id))
            .map((m) => m.name)
            .join(", ")}] user groups`);
    }
    else {
        updateIdentity.organizationProfile.isDisabled = true;
        updateIdentity.organizationProfile.extensionFields.numbers = [];
        updateIdentity.organizationProfile.email = null;
        await dal_manager_1.dbManager.accessIdentity.updateIdentity(constants_1.METU_ORGANIZATION_ID, constants_1.METU_SYNC_USER_ID, trx, updateIdentity, false);
        app_logs_1.logger.info(`Graduate with temporary access with unique id ${armonUser?.organizationProfile?.uniqueId} is disabled!`);
        return;
    }
};
exports.disableTemporaryGraduates = disableTemporaryGraduates;
