"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.syncUsers = exports.syncAllUsers = void 0;
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const bidb_1 = require("../models/bidb");
const utils_1 = require("../utils/utils");
const db_functions_1 = require("./db-functions");
const duplicate_org_unit_remover_1 = require("./duplicate-org-unit-remover");
const business_utils_1 = require("./business.utils");
let armonOrganizationUnits = [];
let armonUserGroups = [];
const TAG = "[metu-sync][business.sync-full.ts] ";
const syncAllUsers = async () => {
    await (0, cli_queries_1.systemTransaction)(utils_1.armonDBPool, async (trx) => {
        await (0, duplicate_org_unit_remover_1.duplicateOrganizationUnitRemover)(trx);
        const bidbGroupIds = await (0, db_functions_1.getBIDBGroupIds)();
        const bidbUnitNames = await (0, db_functions_1.getBIDBUnitNames)();
        armonUserGroups = await (0, db_functions_1.getArmonUserGroupIdNamePairs)(bidbGroupIds.map((m) => bidb_1.BIDBGroupIdArmonGroupNameMapping[m]), trx);
        armonOrganizationUnits = await (0, db_functions_1.getArmonOrganizationUnitIdNamePairs)(bidbUnitNames, trx);
        app_logs_1.logger.info(`${TAG}${armonOrganizationUnits.length} unit and ${armonUserGroups.length} group info loaded to memory from Armon DB !`);
        await (0, exports.syncUsers)(trx);
        app_logs_1.logger.info(`Sync operation completed without an error!`);
    });
};
exports.syncAllUsers = syncAllUsers;
const syncUsers = async (trx) => {
    let bidbUsertoProcessCount = 0;
    const limit = 100;
    let offset = 0;
    do {
        const { rows: bidbUserRows, rowCount: bidbUsersRowCount } = await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx) => {
            return trx.query(`     
            SELECT k.*,
            JSONB_AGG(
                DISTINCT(
                    JSONB_BUILD_OBJECT(
                        'groupId', kr.rol_id,
                        'number', kr.rol_data,
                        'username', kr.kullanici_kodu
                    )
                )
            ) as "groups",
            COALESCE( 
                JSONB_AGG (
                    DISTINCT(
                        JSONB_BUILD_OBJECT (
                            'credential', ky.yetki_degeri,
                            'unit', ky.birim,
                            'number', ky.rol_data,
                            'groupId', ky.rol_id
                        )
                    )
                ) FILTER (WHERE ky.kisi_id IS NOT NULL AND ky.yetki_degeri IS NOT NULL), '[]'
            ) AS details,
			FALSE AS "updateStatus"
            FROM
            metubidb.kisi k
            INNER JOIN metubidb.kisi_rol kr
                ON k.id = kr.kisi_id
            LEFT JOIN metubidb.kisi_yetki ky
                ON kr.kisi_id = ky.kisi_id AND kr.rol_id = ky.rol_id
            GROUP BY k.id, k.ad, k.soyad, k.eposta
            ORDER BY k.id, k.ad, k.soyad, k.eposta ASC
            OFFSET ${offset} LIMIT ${limit}
            `);
        });
        bidbUsertoProcessCount = bidbUsersRowCount;
        app_logs_1.logger.info(`${TAG} Processing ${bidbUsersRowCount} users }`);
        await (0, cli_queries_1.systemTransaction)(utils_1.armonDBPool, async (trx) => {
            for (const bidbUser of bidbUserRows) {
                app_logs_1.logger.info(`Processing bidb user with unique id ${bidbUser.id}`);
                const bidbUserExistanceCheck = await (0, db_functions_1.checkArmonUserExistance)(trx, bidbUser.id);
                if (bidbUserExistanceCheck) {
                    if (bidbUserExistanceCheck.isDisabled) {
                        app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} exists in armon database and currently disabled. User will be re-enabled.`);
                    }
                    await (0, business_utils_1.updateExistingBidbUser)(bidbUser, armonOrganizationUnits, armonUserGroups, bidbUserExistanceCheck.userId, trx);
                }
                else {
                    app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} is not present in Armon database!`);
                    await (0, business_utils_1.importNewBidbUser)(bidbUser, armonOrganizationUnits, armonUserGroups, trx);
                }
                app_logs_1.logger.info(`Bidb user with unique id ${bidbUser.id} is processed!`);
                app_logs_1.logger.info(`*****************************************************`);
            }
        });
        offset += bidbUsertoProcessCount;
        console.log(`${TAG} ${offset} users processed in total}`);
    } while (bidbUsertoProcessCount === limit);
    return;
};
exports.syncUsers = syncUsers;
