"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.EYASWeekBreak = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const i18n_1 = __importDefault(require("i18n"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../../../business/report/generator");
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
class EYASWeekBreak extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._workPlansWeekBreaks = {};
        this._workSheetRowIndex = 2;
        this._workSheetColIndex = 1;
    }
    async generateReport() {
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.WEEKLY_BREAK_REPORT", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, 1, 1);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const userWeeklyHolidays = (await trx.query(`
				SELECT uop."uniqueId", uop.name || ' ' || uop.surname as fullname, ed.date
				FROM "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" as ed
				INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
					on uop."userId" = ed."userId"
				INNER JOIN (${userFilter.query}) AS T3
					ON uop."userId" = T3."userId"
				where  "date" >= $${userFilter.bindingKeys.length + 1} 
						AND "date" <= $${userFilter.bindingKeys.length + 2}
					    AND (data->'s'->>'wd')::int <=0 
				GROUP BY ed."userId", "date" , uop."uniqueId", uop."name", uop."surname"
				ORDER BY uop."uniqueId", ed."date" ASC
				`, [...userFilter.bindingKeys, this._options.startDate.toSQL(), this._options.endDate.toSQL()])).rows;
            this.addGeneralRow(userWeeklyHolidays);
        });
        this._wsGeneral.views = [{ state: "frozen", ySplit: 1, activeCell: "A1", showGridLines: false }];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "EYAS Hafta Tatili",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.EMPLOYEE_NUMBER",
                cellWidth: 30,
            },
            { value: "İsim Soyisim" },
            {
                phrase: "REPORT.START_DATE",
                cellWidth: 40,
            },
            {
                phrase: "REPORT.END_DATE",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.GEDIZ.DEPUTY_TYPE",
                cellWidth: 60,
            },
            {
                phrase: "CUSTOM-REPORT.GEDIZ.WORKING_PROGRAM",
                cellWidth: 60,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(userWeeklyHolidays) {
        for (const weekHoliday of userWeeklyHolidays) {
            const date = luxon_1.DateTime.fromJSDate(weekHoliday.date).toFormat("dd.MM.yyyy");
            const data = [weekHoliday.uniqueId, weekHoliday.fullname, date, date, "02", "FREI"];
            this.addCellsToRow(data, this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral);
            this._workSheetRowIndex++;
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell11(ws, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.WEEKLY_BREAK_REPORT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.MONTH_AND_YEAR", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), generator_1.ReportGenerator.Constants.Styling.AllThin, undefined, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "ORGANIZATIONUNITTYPE.COMPANY", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, "EYAS", generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
            this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
}
exports.EYASWeekBreak = EYASWeekBreak;
async function generateReport(request, locale) {
    const report = new EYASWeekBreak(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
