"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.EYASWeekBreak = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const i18n_1 = __importDefault(require("i18n"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../../../business/report/generator");
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const EYASPermissionSAPNumbers = {
    "69e0fc4e-55ce-48d4-a8a8-be7424d00196": "0100",
    "ad04abc5-43d0-45dd-9a58-fa20fc92dbca": "0131",
    "251d35dd-4f79-4896-8a9f-8f55a081f06f": "0135",
    "1752150a-8d41-4215-a094-dc2d766205c9": "0150",
    "67099cba-1017-4e65-9e88-37d056d94d90": "0155",
    "94f91480-6931-4cec-b811-8b7e5d6fd473": "0015",
    "415b5c9b-74fb-47ea-815f-7c6e95b3b085": "0160",
    "93e00aaa-d50a-430d-9d06-527afe0d3ac7": "0200",
    "6128f4d5-7c29-482e-af4e-018af86b92a3": "0210",
    "11ac1771-64cc-4639-992c-67c856d1d0b3": "0240",
    "30eb6ac9-fc7b-4fe8-82e8-2467f43cd6ed": "0025",
    "e4285652-fa34-4802-9b92-8ca745117bac": "0270",
    "502b364e-5296-405a-91be-851539ecf66e": "0280",
    "8235ed06-7f7c-4bd1-bf1e-a311c2ae8b26": "0310",
    "0c358827-d982-4825-9feb-44fda1dd4078": "0380",
    "65f6d83d-09a0-45e2-afb7-1c9012303f96": "0390",
    "cf98eb30-63ee-4231-b66b-482bfbd7e069": "0400",
    "c1fda513-f1f8-4be6-8520-796ac583e64b": "0045",
    "7e8426ca-3a82-4b3b-a0da-7c5eee1e3e78": "0470",
    "56792138-91b5-4786-8649-edfcb2e81cb8": "0480",
    "8814db56-2fcc-408f-bd80-5b38fe57fedd": "0490",
    "22f9e505-ff95-44ee-b050-5ecf43984867": "0500",
    "71d61075-9280-4fa6-985a-196009e6a67c": "0510",
    "fceda345-c772-4cd3-8d53-9855cf73c046": "0055",
    "b4fb7d5f-88e1-4d25-b3c8-141db8dcaed3": "0056",
    "fbcd7045-4156-4fae-bbbd-44fd19654204": "0080",
};
class EYASWeekBreak extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._workSheetRowIndex = 2;
        this._workSheetColIndex = 1;
    }
    async generateReport() {
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.ABSENCE_REPORT", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, 1, 1);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const userPermissions = (await trx.query(`
                SELECT  upp."userId",
                        uop."uniqueId",
						uop.name || ' ' || uop.surname as "fullname",
                        jsonb_agg(
                            jsonb_build_object(
                            'typeId', pt.id,
                            'startDateTime', pp."startDateTime",
                            'endDateTime', pp."endDateTime",
							'requestDateTime', pp."requestDateTime"
                            )
                        ) as permissions
                FROM "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" as pp
                INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" as upp
                    ON pp.id = upp."ppermissionId"
                INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as pt
                        ON  pt.id = pp."ppermissionTypeId"
                INNER JOIN (${userFilter.query}) AS T3
                    ON T3."userId" = upp."userId"
                INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					ON uop."userId" = upp."userId"
                WHERE	pp.status = 1
						AND (tstzrange(pp."startDateTime", pp."endDateTime") && tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 2}))
                GROUP BY upp."userId", uop."uniqueId",uop.name, uop.surname
                `, [...userFilter.bindingKeys, this._options.startDate.minus({ month: 1 }).set({ day: 26 }).toSQL(), this._options.endDate.toSQL()])).rows;
            this.addGeneralRow(userPermissions);
        });
        this._wsGeneral.views = [{ state: "frozen", ySplit: 1, activeCell: "A1", showGridLines: false }];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "EYAS Hafta Tatili",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.EMPLOYEE_NUMBER",
                cellWidth: 30,
            },
            { value: "İsim Soyisim" },
            {
                phrase: "CUSTOM-REPORT.GEDIZ.ABSENCE_TYPE",
                cellWidth: 30,
            },
            {
                phrase: "REPORT.START_DATE",
                cellWidth: 40,
            },
            {
                phrase: "REPORT.END_DATE",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.GEDIZ.START_TIME",
                cellWidth: 60,
            },
            {
                phrase: "CUSTOM-REPORT.GEDIZ.END_TIME",
                cellWidth: 60,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(userPermissions) {
        for (const user of userPermissions) {
            for (const permission of user.permissions) {
                const absenceType = EYASPermissionSAPNumbers[permission.typeId];
                if (absenceType) {
                    const startDateTime = luxon_1.DateTime.fromISO(permission.startDateTime);
                    const endDateTime = luxon_1.DateTime.fromISO(permission.endDateTime);
                    const dayBoundaries = luxon_1.Interval.fromDateTimes(this._options.startDate.minus({ month: 1 }).set({ day: 26 }), this._options.endDate);
                    const offsetBoundaries = luxon_1.Interval.fromDateTimes(this._options.startDate.minus({ month: 1 }).set({ day: 26 }), this._options.startDate.set({ day: 26 }));
                    const permissionInterval = luxon_1.Interval.fromDateTimes(startDateTime, endDateTime);
                    const requestDateTime = luxon_1.DateTime.fromISO(permission.requestDateTime);
                    if (startDateTime >= this._options.startDate && endDateTime <= this._options.startDate.set({ day: 26 })) {
                        const data = [user.uniqueId, user.fullname, absenceType, startDateTime.toFormat("dd.MM.yyyy"), endDateTime.toFormat("dd.MM.yyyy"), "", ""];
                        this.addCellsToRow(data, this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral);
                        this._workSheetRowIndex++;
                    }
                    else if (dayBoundaries.intersection(permissionInterval) &&
                        requestDateTime >= this._options.startDate.minus({ month: 1 }).set({ day: 26 }) &&
                        requestDateTime < this._options.startDate.set({ day: 26 })) {
                        const intersection = dayBoundaries.intersection(permissionInterval);
                        const data = [user.uniqueId, user.fullname, absenceType, intersection.start.toFormat("dd.MM.yyyy"), intersection.end.toFormat("dd.MM.yyyy"), "", ""];
                        this.addCellsToRow(data, this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral);
                        this._workSheetRowIndex++;
                    }
                    else if (offsetBoundaries.intersection(permissionInterval) && requestDateTime >= this._options.startDate.set({ day: 26 })) {
                        const intersection = offsetBoundaries.intersection(permissionInterval);
                        const data = [user.uniqueId, user.fullname, absenceType, intersection.start.toFormat("dd.MM.yyyy"), intersection.end.toFormat("dd.MM.yyyy"), "", ""];
                        this.addCellsToRow(data, this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral);
                        this._workSheetRowIndex++;
                    }
                }
            }
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell11(ws, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.WEEKLY_BREAK_REPORT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.MONTH_AND_YEAR", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), generator_1.ReportGenerator.Constants.Styling.AllThin, undefined, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "ORGANIZATIONUNITTYPE.COMPANY", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, "EYAS", generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
            this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
}
exports.EYASWeekBreak = EYASWeekBreak;
async function generateReport(request, locale) {
    const report = new EYASWeekBreak(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
