"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.GEDIZExtraWorkCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const constants_1 = require("./models/constants");
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const moment_1 = __importDefault(require("moment"));
const extra_work_1 = require("./utilities/extra.work");
class GEDIZExtraWorkCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._rowIndex = 1;
        this._offsetDayCount = this._options.startDate.minus({ days: 1 }).day - 25;
    }
    async generateReport() {
        this._vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._options.organizationId);
        this._filePath = await this.generateGEDIZExtraWorkCustomReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "GEDIZ SAP Ek Ödemeler",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateGEDIZExtraWorkCustomReport() {
        this._workSheetGeneral = this._wb.addWorksheet("Sayfa 1", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._filterSheet = this._wb.addWorksheet("Uygulanan Filtreler", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        this.addFilterToSheet(filters.join("\r\n"));
        this.initGeneralSheet(1);
        this._workSheetGeneral.columns.forEach((elem) => {
            elem.width = 16;
        });
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            let { query, bindings } = (0, constants_1.generateEmployeeDataQuery)(this._options.organizationId, userFilter, this._options.startDate.toSQL(), this._options.startDate.minus({ days: this._offsetDayCount }).toSQL(), this._options.endDate.toSQL(), this._options.endDate.set({ day: 26 }).toSQL());
            await trx.query(query, bindings);
            let { rows } = await trx.query("FETCH 100 FROM employee_day_cursor");
            while (rows.length > 0) {
                this.addGeneralRow(rows);
                rows = (await trx.query("FETCH 100 FROM employee_day_cursor")).rows;
            }
            await trx.query(`CLOSE employee_day_cursor;`);
        });
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(startColIndex) {
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Tarih", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Sicil Numarası", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Çalışmayı Yapan", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Departman Bilgisi", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "İş Birimi", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Görevi", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "Çalışma Saati", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex, startColIndex + 3);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex, "Hak Edilen Fazla Mesai Süresi (SAAT)", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addInitCell(this._workSheetGeneral, this._rowIndex + 1, startColIndex++, "Hafta İçi", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addInitCell(this._workSheetGeneral, this._rowIndex + 1, startColIndex++, "Hafta Tatili", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addInitCell(this._workSheetGeneral, this._rowIndex + 1, startColIndex++, "Resmi Tatil", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addInitCell(this._workSheetGeneral, this._rowIndex + 1, startColIndex++, "Kısa Dönüş", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._workSheetGeneral.mergeCells(this._rowIndex, startColIndex, this._rowIndex + 1, startColIndex);
        this.addInitCell(this._workSheetGeneral, this._rowIndex, startColIndex++, "ÇALIŞMANIN GEREKÇESİ", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._rowIndex += 2;
        this._workSheetGeneral.views = [{ state: "frozen", ySplit: 2, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    async addGeneralRow(userInfo) {
        userInfo.forEach((data) => {
            data.monthlyData.forEach((dailyData, index) => {
                const extraWorkPermissions = dailyData.data.p.filter((perm) => perm.pt === constants_1.ExtraLeaveTypeId);
                if (extraWorkPermissions.length > 0) {
                    extraWorkPermissions.forEach((ewp) => {
                        const permissionUsageCount = (0, extra_work_1.calculatePermissionTime)(ewp);
                        let startColIndex = 1;
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: (0, moment_1.default)(dailyData.data.d).format("DD.MM.YYYY"),
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: data.uniqueId,
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: data.fullname,
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: data.extensionFields?.org_department ?? "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: data.extensionFields?.org_business_unit ?? "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: (0, moment_1.default)(ewp.r.s).format("HH:mm") + "-" + (0, moment_1.default)(ewp.r.e).format("HH:mm"),
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        let result = {
                            [constants_1.ExtraWorkTypes.HAFTA_ICI_UCRET_TURU]: 0,
                            [constants_1.ExtraWorkTypes.HAFTA_SONU_UCRET_TURU]: 0,
                            [constants_1.ExtraWorkTypes.RESMI_TATIL_UCRET_TURU]: 0,
                            [constants_1.ExtraWorkTypes.KISA_DONUS_UCRET_TURU]: 0,
                        };
                        for (const [key, value] of Object.entries(constants_1.EmployeeUnionConditions)) {
                            const conditionalValue = data.extensionFields?.[value.key];
                            if (value.values.includes(conditionalValue)) {
                                switch (key) {
                                    case constants_1.EmployeeSubGroups.BEYAZ_YAKA:
                                        (0, extra_work_1.applyRuleForWhiteCollar)(dailyData.data, permissionUsageCount, result, this._vacations);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKALI_VARDIYALI:
                                        (0, extra_work_1.applyRuleForBlueCollarShiftUnion)(dailyData.data, permissionUsageCount, result, this._vacations, data.monthlyData[index - 1]?.data);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKALI_VARDIYASIZ:
                                        (0, extra_work_1.applyRuleForBlueCollarUnion)(dailyData.data, permissionUsageCount, result, this._vacations);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKASIZ_VARDIYALI:
                                        (0, extra_work_1.applyRuleForBlueCollarShift)(dailyData.data, permissionUsageCount, result, this._vacations, data.monthlyData[index - 1]?.data);
                                        break;
                                    default:
                                        break;
                                }
                            }
                        }
                        if (ewp.u <= 0) {
                            this._workSheetGeneral.getRow(this._rowIndex).fill = {
                                type: "pattern",
                                pattern: "solid",
                                fgColor: { argb: "FFFFFF00" },
                            };
                        }
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: result[constants_1.ExtraWorkTypes.HAFTA_ICI_UCRET_TURU] > 0 ? result[constants_1.ExtraWorkTypes.HAFTA_ICI_UCRET_TURU].toFixed(2) : "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                            alignment: {
                                horizontal: "center",
                                vertical: "middle",
                                wrapText: true,
                            },
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: result[constants_1.ExtraWorkTypes.HAFTA_SONU_UCRET_TURU] > 0 ? result[constants_1.ExtraWorkTypes.HAFTA_SONU_UCRET_TURU].toFixed(2) : "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                            alignment: {
                                horizontal: "center",
                                vertical: "middle",
                                wrapText: true,
                            },
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: result[constants_1.ExtraWorkTypes.RESMI_TATIL_UCRET_TURU] > 0 ? result[constants_1.ExtraWorkTypes.RESMI_TATIL_UCRET_TURU].toFixed(2) : "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                            alignment: {
                                horizontal: "center",
                                vertical: "middle",
                                wrapText: true,
                            },
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: result[constants_1.ExtraWorkTypes.KISA_DONUS_UCRET_TURU] > 0 ? result[constants_1.ExtraWorkTypes.KISA_DONUS_UCRET_TURU].toFixed(2) : "",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                            alignment: {
                                horizontal: "center",
                                vertical: "middle",
                                wrapText: true,
                            },
                        });
                        this.setDataCell({
                            rowIndex: this._rowIndex,
                            colIndex: startColIndex++,
                            value: " ",
                            ws: this._workSheetGeneral,
                            border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                        });
                        this._rowIndex += 1;
                    });
                }
            });
        });
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        if (params.formula) {
            c.value = {
                formula: params.formula,
            };
        }
        else {
            c.value = params.value;
        }
        if (params.alignment) {
            c.alignment = params.alignment;
        }
        else {
            c.alignment = {
                horizontal: "left",
                vertical: "middle",
                wrapText: true,
            };
        }
        if (params.border) {
            c.border = params.border;
        }
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
    }
    addFilterToSheet(filter) {
        const startColIndex = 1;
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell(this._filterSheet, 1, startColIndex, "FAZLA MESAİ RAPORU", constants_1.HeadersStyling.v1.border, constants_1.HeadersStyling.v1.font);
        this._filterSheet.mergeCells(1, startColIndex, 1, startColIndex + 2);
        this.addFilterCell(this._filterSheet, 2, startColIndex, "AİT OLDUĞU AY VE YIL", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
        this.addFilterCell(this._filterSheet, 2, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font, generator_1.ReportGenerator.Constants.DateFormat);
        this._filterSheet.mergeCells(2, startColIndex + 1, 2, startColIndex + 2);
        this.addFilterCell(this._filterSheet, 3, startColIndex, "KURUM", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
        this.addFilterCell(this._filterSheet, 3, startColIndex + 1, "GEDIZ EYAS", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font);
        this._filterSheet.mergeCells(3, startColIndex + 1, 3, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell(this._filterSheet, 4, startColIndex, "UYGULANAN FILTRELER", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
            this.addFilterCell(this._filterSheet, 4, startColIndex + 1, filter, constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font);
            this._filterSheet.mergeCells(4, startColIndex + 1, 4, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = this._filterSheet.getRow(4);
            row.height = filterlength * 30;
        }
        this._filterSheet.columns = columns;
        this._filterSheet.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.GEDIZExtraWorkCustomReport = GEDIZExtraWorkCustomReport;
async function generateReport(request, locale) {
    const report = new GEDIZExtraWorkCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
