"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.GEDIZEYASTallyCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const i18n_1 = __importDefault(require("i18n"));
const sharp_1 = __importDefault(require("sharp"));
const uuid_1 = __importDefault(require("uuid"));
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const app_logs_1 = require("../../../app.logs");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const constants_1 = require("./models/constants");
const extra_work_1 = require("./utilities/extra.work");
const employee_day_1 = require("../../../business/pacs2/employee.day");
const ContentHeader = {
    alignment: {
        horizontal: "center",
        vertical: "bottom",
        wrapText: true,
        textRotation: 90,
    },
    fill: {
        type: "pattern",
        pattern: "solid",
        fgColor: { argb: "FFD8E4BC" },
    },
    font: { name: "Calibri", size: 11, bold: true },
};
class GEDIZEYASTallyCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._daysTotal = Math.round(this._options.endDate.diff(this._options.startDate, "days").toObject().days);
        this._offsetDayCount = this._options.startDate.minus({ days: 1 }).day - 25;
    }
    async generateReport() {
        this._vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._options.organizationId);
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "EYAS Aylık Puantaj Cetveli",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._wsGeneral = this._wb.addWorksheet("Aylık Puantaj Cetveli", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet("Uygulanan Filtreler", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        const logo = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(this._request.organizationId);
        if (logo) {
            const EYASLogoBase64 = "data:image/png;base64," + (await (0, sharp_1.default)(Buffer.from(logo), { failOnError: false }).toFormat("png").toBuffer()).toString("base64");
            const imageId1 = this._wb.addImage({
                base64: EYASLogoBase64,
                extension: "png",
            });
            this._wsGeneral.addImage(imageId1, {
                tl: { col: 1, row: 0 },
                ext: { width: 120, height: 60 },
            });
        }
        const filters = [];
        let organizationUnitNames = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, organizationUnitNames[0]);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const workPlanTypeAbbrevations = (await trx.query(`
					
					SELECT jsonb_object_agg(id,name) as abbrev
					from "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
					where "deletedAt" IS null
					`)).rows[0].abbrev;
            let { query, bindings } = (0, constants_1.generateEmployeeDataQuery)(this._options.organizationId, userFilter, this._options.startDate.toSQL(), this._options.startDate.minus({ days: this._offsetDayCount }).toSQL(), this._options.endDate.toSQL(), this._options.endDate.set({ day: 26 }).toSQL());
            await trx.query(query, bindings);
            let { rows } = await trx.query("FETCH 100 FROM employee_day_cursor");
            while (rows.length > 0) {
                await this.addGeneralRow(rows, workPlanTypeAbbrevations);
                rows = (await trx.query("FETCH 100 FROM employee_day_cursor")).rows;
            }
            await trx.query(`CLOSE employee_day_cursor;`);
        });
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._wsGeneral.eachRow((row, rowNumber) => {
            if (rowNumber > 6)
                row.height = 30;
        });
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, organizationUnitIdNamePair) {
        let font = { name: "Calibri", size: 14, bold: true, color: { argb: "FF000000" } };
        wsRaw.mergeCells("A1:C3");
        wsRaw.getCell("C3").border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
        wsRaw.mergeCells("A4:C4");
        wsRaw.getCell("A4").border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
        wsRaw.mergeCells("D4:V4");
        wsRaw.getCell("D4").border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
        wsRaw.mergeCells("W4:AH4");
        this.addInitCell(wsRaw, "W4", i18n_1.default.__({ phrase: "PUANTAJ CETVELİ", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("AI4:AJ4");
        this.addInitCell(wsRaw, "AI4", i18n_1.default.__({ phrase: "REPORT.MONTH", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("AK4:AO4");
        this.addInitCell(wsRaw, "AK4", this._options.startDate.month, generator_1.CustomReportGenerator.Constants.Styling.AllThin, font);
        this.addInitCell(wsRaw, "A6", "NO.", generator_1.CustomReportGenerator.Constants.Styling.AllThin, undefined, {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
            textRotation: 90,
        });
        this.setCell(wsRaw, 6, 2, "Sicil Numarası", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        this.setCell(wsRaw, 6, 3, "ADI VE SOYADI", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        this.setCell(wsRaw, 6, 4, "İşe Giriş Tarihi", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        this.setCell(wsRaw, 6, 5, "Lokasyon (Mim)", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        this.setCell(wsRaw, 6, 6, "Çalışan grubu / Çalışma Şekli", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        this.setCell(wsRaw, 6, 7, "Dahil Edilmeyen İzinler", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFD8E4BC" },
        };
        wsRaw.getCell("G6").border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
        this._workSheetColIndex = 8;
        this._workSheetRowIndex = 6;
        this._prevMonthStartColumnLetter = this.getColumnLetter(this._workSheetColIndex);
        for (let x = 26; x <= this._options.startDate.minus({ day: 1 }).day; x++) {
            this.setCell(wsRaw, this._workSheetRowIndex, this._workSheetColIndex++, x, generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFDCE6F1" },
            };
        }
        this._thisMonthStartColumnIndex = this._workSheetColIndex;
        this._thisMonthStartColumnLetter = this.getColumnLetter(this._workSheetColIndex);
        for (let i = 1; i <= this._daysTotal; i++) {
            this.setCell(wsRaw, this._workSheetRowIndex, this._workSheetColIndex++, i, generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFD8E4BC" },
            };
        }
        wsRaw.mergeCells("AP4:" + this.getColumnLetter(this._workSheetColIndex) + 4);
        this.addInitCell(wsRaw, "AP4", this._options.startDate.year, generator_1.CustomReportGenerator.Constants.Styling.AllThin, font);
        this.setCell(wsRaw, this._workSheetRowIndex, this._workSheetColIndex++, "", generator_1.CustomReportGenerator.Constants.Styling.AllThin, font);
        for (const content of constants_1.TallyReportContent) {
            wsRaw.mergeCells(this._workSheetRowIndex - 2, this._workSheetColIndex, this._workSheetRowIndex, this._workSheetColIndex);
            this.setCell(wsRaw, this._workSheetRowIndex, this._workSheetColIndex, content, generator_1.CustomReportGenerator.Constants.Styling.AllThin, ContentHeader.font, ContentHeader.alignment).fill = ContentHeader.fill;
            this._workSheetColIndex += 1;
        }
        wsRaw.mergeCells(this._workSheetRowIndex - 2, this._workSheetColIndex, this._workSheetRowIndex, this._workSheetColIndex);
        this.setCell(wsRaw, this._workSheetRowIndex, this._workSheetColIndex, "İMZA", generator_1.CustomReportGenerator.Constants.Styling.AllThin, ContentHeader.font, ContentHeader.alignment).fill = ContentHeader.fill;
        wsRaw.mergeCells("D1:" + this.getColumnLetter(this._workSheetColIndex) + 2);
        wsRaw.getCell("D1").fill = ContentHeader.fill;
        this.addInitCell(wsRaw, "D1", "Aylık Puantaj Cetveli", generator_1.CustomReportGenerator.Constants.Styling.AllThin, {
            name: "Calibri",
            size: 16,
            bold: true,
            color: { argb: "FF000000" },
        });
        wsRaw.mergeCells("D3:" + this.getColumnLetter(this._workSheetColIndex) + 3);
        this.addInitCell(wsRaw, "D3", organizationUnitIdNamePair?.name ? "GDZ ENERJİ YATIRIMLARI A.Ş ARIZA ONARIM MÜDÜRLÜĞÜ / " + organizationUnitIdNamePair.name : "GDZ ENERJİ YATIRIMLARI A.Ş ARIZA ONARIM MÜDÜRLÜĞÜ", generator_1.CustomReportGenerator.Constants.Styling.AllThin, {
            name: "Calibri",
            size: 16,
            bold: true,
            color: { argb: "FF000000" },
        });
        this._workSheetColIndex += 1;
        this._workSheetRowIndex += 1;
        wsRaw.findRow(4).height = 44.5;
        wsRaw.findRow(5).height = 25;
        wsRaw.findRow(6).height = 48.76;
        wsRaw.columns.forEach((column, index) => {
            if (index === 0) {
                column.width = 2.86;
            }
            else if (index > 0 && index <= 5) {
                column.width = 22.14;
            }
        });
        wsRaw.views = [{ state: "frozen", ySplit: 6 }];
    }
    getColumnLetter(index) {
        index -= 1;
        const ACode = "A".charCodeAt(0);
        const letters = [];
        while (index >= 0) {
            const remainder = index % 26;
            letters.unshift(String.fromCharCode(ACode + remainder));
            index = Math.floor(index / 26) - 1;
        }
        return letters.join("");
    }
    async addGeneralRow(userInfo, workPlanAbbrevations) {
        this._workSheetColIndex = 1;
        for (const user of userInfo) {
            let extraWorks = {
                [constants_1.ExtraWorkTypes.HAFTA_ICI_UCRET_TURU]: 0,
                [constants_1.ExtraWorkTypes.HAFTA_SONU_UCRET_TURU]: 0,
                [constants_1.ExtraWorkTypes.RESMI_TATIL_UCRET_TURU]: 0,
                [constants_1.ExtraWorkTypes.KISA_DONUS_UCRET_TURU]: 0,
            };
            this._workSheetColIndex = this.addCellsToRow([this._workSheetRowIndex - 6, user.uniqueId, user.fullname], this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            if (user.employmentStartUtc && luxon_1.Interval.fromDateTimes(this._options.startDate, this._options.endDate).contains(luxon_1.DateTime.fromJSDate(user.employmentStartUtc))) {
                this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, luxon_1.DateTime.fromJSDate(user.employmentStartUtc).toFormat("dd.MM.yyyy"), generator_1.CustomReportGenerator.Constants.Styling.AllThin, { bold: false });
            }
            else {
                this._wsGeneral.getCell(this._workSheetRowIndex, this._workSheetColIndex++).border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
            }
            const subGroup = user.extensionFields?.org_employee_subgroup ? constants_1.PartOfOrganizationForm.org_employee_subgroup[user.extensionFields?.org_employee_subgroup] : "";
            const workPlanRule = user.extensionFields?.org_workplan_rule ? constants_1.PartOfOrganizationForm.org_workplan_rule[user.extensionFields?.org_workplan_rule] : "";
            this._workSheetColIndex = this.addCellsToRow([user.extensionFields?.org_loc_text ?? "", (subGroup ?? "") + "/" + (workPlanRule ?? ""), ""], this._workSheetColIndex, this._workSheetRowIndex, this._wsGeneral, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            let normalWorkingDay = user.countOfWorkingDays;
            const processTallyReportDate = (dailyData, previousDayData) => {
                const workPlanAbbr = workPlanAbbrevations[Object.keys(workPlanAbbrevations).find((elem) => dailyData.data.ws
                    .filter((elem) => elem.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours)
                    .map((elem) => elem.wp)
                    .includes(elem))];
                const hasUserAccessTime = dailyData.data.ex.s || dailyData.data.ex.e;
                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromSQL(dailyData.date), luxon_1.DateTime.fromSQL(dailyData.date).endOf("day"))));
                const weekHoliday = dailyData.data.s.wd <= 0;
                let permissionTypeId = undefined;
                let extraLeaveHours = 0;
                if (dailyData.data.p?.length > 0 && dailyData.data.p.map((elem) => elem.p).find((elem) => user.permissions?.map((perm) => perm.id).includes(elem))) {
                    permissionTypeId = dailyData.data.p.find((elem) => elem.pt === constants_1.ExtraLeaveTypeId)
                        ? constants_1.ExtraLeaveTypeId
                        : dailyData.data.p.find((elem) => constants_1.EYASPermissionTypeAbbrevations[elem.pt])?.pt ?? dailyData.data.p[dailyData.data.p.length - 1].pt;
                    extraLeaveHours = dailyData.data.p
                        .filter((elem) => elem.pt === constants_1.ExtraLeaveTypeId)
                        .map((item) => (0, moment_1.default)(item.r.e).diff((0, moment_1.default)(item.r.s), "hour"))
                        .reduce((prev, current) => (prev += current), 0);
                }
                else if (dailyData.data.p?.length > 0 && !dailyData.data.p.map((elem) => elem.p).find((elem) => user.permissions?.map((perm) => perm.id).includes(elem))) {
                    normalWorkingDay += 1;
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, `Kullanılan izin gelecek ayın raporuna yansıtılmıştır.(${constants_1.EYASPermissionTypeAbbrevations[dailyData.data.p[dailyData.data.p.length - 1].pt] ?? ""})`, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    return;
                }
                if (vacationExistency) {
                    this.applyHolidays(dailyData.data.s.wd, extraLeaveHours, vacationExistency);
                }
                else if (weekHoliday) {
                    this.applyHolidays(dailyData.data.s.wd, extraLeaveHours, vacationExistency);
                }
                else if (permissionTypeId) {
                    this.applyPermissions(dailyData.data.s.wd, permissionTypeId, workPlanAbbr);
                }
                else {
                    if (hasUserAccessTime) {
                        this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (workPlanAbbr ?? ""), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    }
                    else {
                        this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (workPlanAbbr ?? ""), generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
                            type: "pattern",
                            pattern: "solid",
                            fgColor: { argb: "FFFD0F0F" },
                        };
                    }
                }
                const extraWorkPermissions = dailyData.data.p.filter((perm) => perm.pt === constants_1.ExtraLeaveTypeId);
                if (extraWorkPermissions.length > 0) {
                    extraWorkPermissions.forEach((ewp) => {
                        const permissionUsageCount = (0, extra_work_1.calculatePermissionTime)(ewp);
                        for (const [key, value] of Object.entries(constants_1.EmployeeUnionConditions)) {
                            const subGroup = user.extensionFields?.[value.key];
                            if (value.values.includes(subGroup)) {
                                switch (key) {
                                    case constants_1.EmployeeSubGroups.BEYAZ_YAKA:
                                        (0, extra_work_1.applyRuleForWhiteCollar)(dailyData.data, permissionUsageCount, extraWorks, this._vacations);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKALI_VARDIYALI:
                                        (0, extra_work_1.applyRuleForBlueCollarShiftUnion)(dailyData.data, permissionUsageCount, extraWorks, this._vacations, previousDayData?.data);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKALI_VARDIYASIZ:
                                        (0, extra_work_1.applyRuleForBlueCollarUnion)(dailyData.data, permissionUsageCount, extraWorks, this._vacations);
                                        break;
                                    case constants_1.EmployeeSubGroups.MAVI_YAKA_SENDIKASIZ_VARDIYALI:
                                        (0, extra_work_1.applyRuleForBlueCollarShift)(dailyData.data, permissionUsageCount, extraWorks, this._vacations, previousDayData?.data);
                                        break;
                                    default:
                                        break;
                                }
                            }
                        }
                    });
                }
            };
            user.monthlyData.forEach((dailyData, index) => {
                let dayDiff = 0;
                if (index === 0) {
                    dayDiff =
                        luxon_1.DateTime.fromSQL(dailyData.date)
                            .diff(this._options.startDate.minus({ days: this._offsetDayCount }), "days")
                            .toObject().days + 1;
                }
                else {
                    dayDiff = luxon_1.DateTime.fromSQL(dailyData.date)
                        .diff(luxon_1.DateTime.fromSQL(user.monthlyData[index - 1].date), "days")
                        .toObject().days;
                }
                if (dayDiff > 1) {
                    this.fillTheSpaces(dayDiff);
                }
                if (dayDiff === 1) {
                    processTallyReportDate(dailyData, user.monthlyData[index - 1]);
                }
                else {
                    processTallyReportDate(dailyData);
                }
            });
            for (let x = 0; x < this._options.endDate.diff(luxon_1.DateTime.fromSQL(user.monthlyData[user.monthlyData.length - 1].date), "days").toObject().days - 1; x++) {
                const date = luxon_1.DateTime.fromSQL(user.monthlyData[user.monthlyData.length - 1].date)
                    .plus({ days: x + 1 })
                    .startOf("day")
                    .toFormat("yyyy-MM-dd");
                let employeeDay = await employee_day_1.EmployeeDay.createEmployeeDay(this._options.organizationId, user.userId, date);
                if (employeeDay) {
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex, "", generator_1.CustomReportGenerator.Constants.Styling.AllThin).fill = {
                        type: "pattern",
                        pattern: "solid",
                        fgColor: { argb: "FF0DDED5" },
                    };
                    employeeDay.process();
                    let serialized = employeeDay.getSerialized();
                    processTallyReportDate({
                        date: date,
                        data: serialized,
                    });
                }
                else {
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, "", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                }
            }
            if (user.employmentEndUtc && luxon_1.Interval.fromDateTimes(this._options.startDate, this._options.startDate).contains(luxon_1.DateTime.fromJSDate(user.employmentEndUtc))) {
                this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, luxon_1.DateTime.fromJSDate(user.employmentEndUtc).toFormat("dd.MM.yyyy"), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            }
            else {
                this._wsGeneral.getCell(this._workSheetRowIndex, this._workSheetColIndex++).border = generator_1.CustomReportGenerator.Constants.Styling.AllThin;
            }
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, normalWorkingDay, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            const generateCellCondition = function (cell, conditions) {
                const generatedCondition = [];
                for (const condition of conditions) {
                    generatedCondition.push("COUNTIF(" + cell + `,"${condition}")`);
                }
                return generatedCondition.join("+");
            };
            this.setCell(this._wsGeneral, this._workSheetRowIndex, 7, {
                formula: generateCellCondition(this._prevMonthStartColumnLetter + this._workSheetRowIndex + ":" + this.getColumnLetter(this._thisMonthStartColumnIndex - 1) + this._workSheetRowIndex, constants_1.TallyReportPreviousMonthPermissionConditionsContent),
            }, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            constants_1.TallyReportPermissionConditionsContent.forEach((elem, index) => {
                this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, {
                    formula: generateCellCondition(this._thisMonthStartColumnLetter + this._workSheetRowIndex + ":" + this.getColumnLetter(this._workSheetColIndex - 4 - index) + this._workSheetRowIndex, elem.conditions),
                }, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            });
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, this._daysTotal, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, extraWorks[constants_1.ExtraWorkTypes.HAFTA_ICI_UCRET_TURU] ?? 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, extraWorks[constants_1.ExtraWorkTypes.HAFTA_SONU_UCRET_TURU] ?? 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, extraWorks[constants_1.ExtraWorkTypes.RESMI_TATIL_UCRET_TURU] ?? 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, extraWorks[constants_1.ExtraWorkTypes.KISA_DONUS_UCRET_TURU] ?? 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            Object.entries(constants_1.SecondReportFeeTypeRules).forEach(([typeId, feeTypeRule]) => {
                let feeCount = this._daysTotal;
                if (feeTypeRule.rules.find((rule) => {
                    if (rule.extensionFieldRules.every((extField) => {
                        return extField.validValues.find((elem) => elem.toLocaleLowerCase("tr") === user.extensionFields[extField.key]?.toLocaleLowerCase("tr"));
                    })) {
                        rule.decreasingPermissions.forEach((permType) => {
                            const permUsage = (0, extra_work_1.getPermissionUsageCount)(user.permissions, permType);
                            feeCount -= permUsage;
                        });
                        return true;
                    }
                })) {
                    switch (typeId) {
                        case constants_1.AdditionalPaymentSecondReportFeeTypes.ARAC_KULLANIM_TAZMINATI: {
                            const permUsage = (0, extra_work_1.getPermissionUsageCount)(user.permissions, constants_1.EYASPermissionTypes.ARAC_KULLANMA);
                            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, permUsage, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                            break;
                        }
                        case constants_1.AdditionalPaymentSecondReportFeeTypes.SEYYAR_GOREV_TAZMINATI: {
                            const permUsage = (0, extra_work_1.getPermissionUsageCount)(user.permissions, constants_1.EYASPermissionTypes.SEYYAR_GOREV);
                            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, permUsage, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                            break;
                        }
                        default:
                            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, feeCount > 0 ? feeCount : 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                            break;
                    }
                }
                else {
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, " ", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                }
            });
            const sap200permUsage = (0, extra_work_1.getPermissionUsageCount)(user.permissions, constants_1.EYASPermissionTypes.SAP200);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, sap200permUsage && sap200permUsage >= this._daysTotal ? 0 : this._daysTotal, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, user.extensionFields?.org_region_diff_comp ?? "", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, sap200permUsage && sap200permUsage >= this._daysTotal ? 0 : this._daysTotal, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            Object.entries(constants_1.FirstReportFeeTypeRules).forEach(([typeId, feeTypeRule]) => {
                let feeCount = normalWorkingDay;
                if (feeTypeRule.rules.find((rule) => {
                    if (rule.extensionFieldRules.every((extField) => {
                        return extField.validValues.find((elem) => elem.toLocaleLowerCase("tr") === user.extensionFields[extField.key]?.toLocaleLowerCase("tr"));
                    })) {
                        rule.decreasingPermissions.forEach((permType) => {
                            const permUsage = (0, extra_work_1.getPermissionUsageCount)(user.permissions, permType);
                            feeCount -= permUsage ?? 0;
                        });
                        return true;
                    }
                })) {
                    if (typeId === constants_1.AdditionalPaymentFirstReportFeeTypes.YEMEK_ODEME_SENDIKA) {
                        this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, user.mealFee, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    }
                    else {
                        this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, feeCount > 0 ? feeCount : 0, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    }
                }
                else {
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, " ", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                }
            });
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, " ", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
            this._workSheetColIndex = 1;
            this._workSheetRowIndex += 1;
        }
    }
    fillTheSpaces(dayDiff) {
        for (let x = 0; x < dayDiff - 1; x++) {
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, "", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        }
    }
    applyHolidays(expectedWork, totalWorkingHours, vacationExistency) {
        if (vacationExistency) {
            switch (vacationExistency.type) {
                case app_enums_1.enums.VacationType.National:
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (totalWorkingHours > 0 ? "ÇGT" : "GT"), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    break;
                case app_enums_1.enums.VacationType.Religious:
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (totalWorkingHours > 0 ? "ÇBT" : "BT"), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    break;
                default:
                    this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (totalWorkingHours > 0 ? "ÇA" : "A"), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
                    break;
            }
        }
        else if (expectedWork === 0) {
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (totalWorkingHours > 0 ? "ÇHT" : "HT"), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        }
    }
    applyPermissions(expectedWork, typeId, workPlanAbbr) {
        if (typeId === constants_1.ExtraLeaveTypeId && expectedWork > 0) {
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, workPlanAbbr, generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        }
        else {
            this.setCell(this._wsGeneral, this._workSheetRowIndex, this._workSheetColIndex++, (constants_1.EYASPermissionTypeAbbrevations[typeId] ?? ""), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        }
    }
    setCell(ws, row, col, value, border, font, alignment) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = alignment ?? {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = font ?? { bold: true };
        c.border = border;
        return c;
    }
    addInitCell(ws, addressStr, value, border, font, alignment) {
        const c = ws.getCell(addressStr);
        c.value = value;
        c.alignment = alignment ?? {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = font ?? { bold: true };
        c.border = border;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell(ws, startRowIndex, startColIndex, "AYLIK PUANTAJ CETVELİ", constants_1.HeadersStyling.v1.border, constants_1.HeadersStyling.v1.font);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, "AİT OLDUĞU AY VE YIL", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font, generator_1.CustomReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, "KURUM", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, "GEDIZ EYAS", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell(ws, startRowIndex + 3, startColIndex, "UYGULANAN FILTRELER", constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v2.font);
            this.addFilterCell(ws, startRowIndex + 3, startColIndex + 1, filter, constants_1.HeadersStyling.v2.border, constants_1.HeadersStyling.v3.font);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.GEDIZEYASTallyCustomReport = GEDIZEYASTallyCustomReport;
async function generateReport(request, locale) {
    const report = new GEDIZEYASTallyCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
