"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ADMSAPExtraWorkCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const constants_1 = require("./constants");
const app_logs_1 = require("../../../app.logs");
const lodash_1 = __importDefault(require("lodash"));
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.Header = {
            border: {
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thick",
                },
            },
            font: { name: "Calibri", size: 16, bold: true, color: { argb: "FFC00000" } },
        };
        Styling.HeaderTwo = {
            border: {
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thick",
                },
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
            },
            font: { name: "Calibri", size: 14, bold: true },
        };
        Styling.HeaderThird = {
            border: {
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thick",
                },
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
            },
            font: { name: "Calibri", size: 14, bold: false },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
var UnionType;
(function (UnionType) {
    UnionType[UnionType["New"] = 1] = "New";
    UnionType[UnionType["Old"] = 2] = "Old";
    UnionType[UnionType["WithoutUnionNetSalary"] = 3] = "WithoutUnionNetSalary";
    UnionType[UnionType["WithoutUnionGrossSalary"] = 4] = "WithoutUnionGrossSalary";
})(UnionType || (UnionType = {}));
var WorkType;
(function (WorkType) {
    WorkType[WorkType["SHIFT"] = 1] = "SHIFT";
    WorkType[WorkType["NON_SHIFT"] = 2] = "NON_SHIFT";
})(WorkType || (WorkType = {}));
const NetSalaryValues = ["1005", "1015"];
const GrossSalaryValues = ["1000"];
class ADMSAPExtraWorkCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._userInfoMapping = new Map();
        this._monthInterval = luxon_1.Interval.fromDateTimes(this._options.startDate, this._options.endDate);
    }
    async generateReport() {
        this._vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._options.organizationId);
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "ADM SAP FM Custom Report",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._wsGeneral = this._wb.addWorksheet("Ek Ödemeler Raporu", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet("Uygulanan Filtreler", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        this._workSheetRowIndex = 1;
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        if (this._options.userIds && this._options.userIds.length > 0) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let userNames = await dal_manager_1.dbManager.accessUser.getBasicUserInfoPg({ organizationId: this._request.organizationId, userIds: this._options.userIds, trx });
                filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.map((elem) => elem.fullname).join(","));
            });
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, 1);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        let query = `
        DECLARE employee_day_cursor
		CURSOR FOR
		WITH employee_day_cte AS(
			SELECT	"userId",
					"uniqueId",
					fullname,			
					COALESCE("extensionFields","extensionFields",'{}') as "extensionFields",
					"employmentStartUtc",
					"employmentEndUtc",
					jsonb_agg("data" ORDER BY data->>'d' ASC) "monthlyData"
			FROM
			(
				SELECT  ed."userId",
						T3."uniqueId",
						T3."name" || ' ' ||T3."surname" as fullname,
						uop."extensionFields"::jsonb,
						uop."employmentStartUtc",
						uop."employmentEndUtc",
						ed."data"
				FROM "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop
					ON uo.id = uop."userOrganizationId"
				INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS ed
					ON ed."userId" = uop."userId"
				INNER JOIN
					(${userFilter.query}) AS T3
				ON T3."userId" = ed."userId" AND ed."data" IS NOT NULL
				WHERE ed.date >= $${userFilter.bindingKeys.length + 1} AND ed.date <= $${userFilter.bindingKeys.length + 2}
				ORDER BY ed.date ASC
			) as ordered
			GROUP BY "userId", "uniqueId", fullname, "extensionFields"::jsonb, "employmentStartUtc", "employmentEndUtc"
		)
		SELECT * 
		FROM employee_day_cte as edc
		LEFT JOIN (
					SELECT 	
						upp."userId",
						jsonb_agg(
							jsonb_build_object(
							'id', pp.id,
							'dynamicFormData', "dynamicFormData",
							'typeId', pt.id,
							'startDateTime', pp."startDateTime",
							'endDateTime', pp."endDateTime"
							)
						) as permissions
				FROM "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" as pp
				INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" as upp
					ON pp.id = upp."ppermissionId"
				INNER JOIN "${this._options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as pt
					ON  pt.id = pp."ppermissionTypeId"
				WHERE pp.status = 1 AND pp."startDateTime" >= $${userFilter.bindingKeys.length + 1} AND pp."endDateTime" <= $${userFilter.bindingKeys.length + 2}
				GROUP BY upp."userId"
		) as perms
		ON perms."userId" = edc."userId"
		ORDER BY fullname ASC;
        `;
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            await trx.query(query, [...userFilter.bindingKeys, this._options.startDate.minus({ days: 10 }).toSQL(), this._options.endDate.toSQL()]);
            let { rows } = await trx.query("FETCH 100 FROM employee_day_cursor");
            while (rows.length > 0) {
                this.applyExtraWorkRule(rows);
                rows = (await trx.query("FETCH 100 FROM employee_day_cursor")).rows;
            }
            await trx.query(`CLOSE employee_day_cursor;`);
        });
        this.addGeneralRow();
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + (this._workSheetRowIndex + 5);
        this._wsGeneral.pageSetup.printTitlesRow = "5:7";
        this._wsFilterSheet.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + (this._workSheetRowIndex + 5);
        this._wsFilterSheet.pageSetup.printTitlesRow = "5:7";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    checkUserUnion(subGroup, salaryType) {
        if (!subGroup) {
            return false;
        }
        if (constants_1.unionInformation.withUnionGroupSecond.includes(subGroup)) {
            return UnionType.New;
        }
        else if (constants_1.unionInformation.withUnionGroupOne.includes(subGroup)) {
            return UnionType.Old;
        }
        else if (salaryType && NetSalaryValues.includes(salaryType)) {
            return UnionType.WithoutUnionNetSalary;
        }
        else if (salaryType && GrossSalaryValues.includes(salaryType)) {
            return UnionType.WithoutUnionGrossSalary;
        }
        else {
            return false;
        }
    }
    applyExtraWorkRule(userInfo) {
        userInfo.forEach((elem) => {
            if (!this._userInfoMapping.get(elem.uniqueId)) {
                if (elem.extensionFields?.org_employee_subgroup && constants_1.unionInformation.withUnionGroupSecond.includes(elem.extensionFields?.org_employee_subgroup)) {
                    this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.NewUnionInitializer.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                }
                else if (elem.extensionFields?.org_employee_subgroup && constants_1.unionInformation.withUnionGroupOne.includes(elem.extensionFields?.org_employee_subgroup)) {
                    this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.OldUnionInitializer.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                }
            }
            const userUnion = this.checkUserUnion(elem.extensionFields?.org_employee_subgroup, elem.extensionFields?.org_salary_type);
            if (userUnion) {
                const weeklySegmentedData = this.divideDataWeeklySegments(elem.monthlyData, elem);
                let countOfWorkingDays = 0;
                elem.monthlyData.forEach((data) => {
                    if (data.ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours).length > 0 &&
                        data.ex.s &&
                        this._monthInterval.contains(luxon_1.DateTime.fromISO(data.d))) {
                        countOfWorkingDays++;
                    }
                });
                switch (userUnion) {
                    case UnionType.New:
                        if (!this._userInfoMapping.get(elem.uniqueId)) {
                            this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.NewUnionInitializer.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                        }
                        this.applyNewUnionRules(weeklySegmentedData, elem, countOfWorkingDays);
                        break;
                    case UnionType.Old:
                        if (!this._userInfoMapping.get(elem.uniqueId)) {
                            this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.OldUnionInitializer.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                        }
                        this.applyOldUnionRules(weeklySegmentedData, elem, countOfWorkingDays);
                        break;
                    case UnionType.WithoutUnionNetSalary:
                        if (!this._userInfoMapping.get(elem.uniqueId)) {
                            this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.WithoutUnionNetSalary.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                        }
                        this.applyWithoutUnionNetSalary(weeklySegmentedData, elem);
                        break;
                    case UnionType.WithoutUnionGrossSalary:
                        if (!this._userInfoMapping.get(elem.uniqueId)) {
                            this._userInfoMapping.set(elem.uniqueId, lodash_1.default.cloneDeep(constants_1.WithoutUnionGrossSalary.map((item) => Object.assign(item, { fullname: elem.fullname, subGroup: constants_1.ADM_EMPLOYEE_SUB_GROUPS[elem.extensionFields["org_employee_subgroup"]] }))));
                        }
                        this.applyWithoutUnionGrossSalary(weeklySegmentedData, elem);
                        break;
                    default:
                        break;
                }
            }
        });
    }
    applyOldUnionRules(weeklySegmentedData, userInfo, countOfWorkingDays) {
        const userWorkingType = this.getUserWorkingType(userInfo.extensionFields?.org_workplan_rule);
        if (!userWorkingType) {
            return;
        }
        weeklySegmentedData.forEach((weekData) => {
            switch (userWorkingType) {
                case WorkType.NON_SHIFT:
                    let totalExtraWorkOnWorkingDaysFirstThree = 0;
                    let totalExtraWorkOnWorkingDaysAfterFirstThree = 0;
                    let normalWorkTime = 0;
                    weekData.forEach((data) => {
                        const extraWorkPermissions = data.p?.length > 0 ? data.p.filter((perm) => perm.pt === ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID) : undefined;
                        extraWorkPermissions?.forEach((extraWorkPerm) => {
                            const permDefinition = userInfo.permissions.find((item) => item.id === extraWorkPerm.p);
                            let permissionTime = this.hoursDiffBetweenToISOTime(permDefinition.startDateTime, permDefinition.endDateTime);
                            permissionTime = Math.ceil(permissionTime - this.applyBreakRestRule(permissionTime));
                            if (permDefinition.dynamicFormData?.ottype &&
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype))) {
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype)).value += permissionTime;
                            }
                            else {
                                const currentDay = luxon_1.DateTime.fromISO(data.d);
                                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(currentDay));
                                if (vacationExistency) {
                                    switch (vacationExistency.type) {
                                        case app_enums_1.enums.VacationType.BeforeNational:
                                        case app_enums_1.enums.VacationType.BeforeReligious:
                                            let workingTime = 0;
                                            data.c.forEach((segment) => {
                                                if (segment.es.w === true) {
                                                    const intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.s)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 13 }), luxon_1.DateTime.fromISO(currentDay.toISO()).plus({ day: 1 })));
                                                    workingTime += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                }
                                            });
                                            if (currentDay.weekday === 6) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2030).value += workingTime;
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2015).value +=
                                                    permissionTime - workingTime > 0 ? permissionTime - workingTime : 0;
                                            }
                                            else if (currentDay.weekday === 7) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2030).value += workingTime;
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value +=
                                                    permissionTime - workingTime > 0 ? permissionTime - workingTime : 0;
                                            }
                                            else {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2030).value += workingTime;
                                            }
                                            break;
                                        default:
                                            if (currentDay.weekday === 6) {
                                                if (permissionTime > 7.5) {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2015).value += 7.5;
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2055).value += permissionTime - 7.5;
                                                }
                                                else {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2015).value += permissionTime;
                                                }
                                            }
                                            else {
                                                if (permissionTime > 7.5) {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += 7.5;
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2055).value += permissionTime - 7.5;
                                                }
                                                else {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += permissionTime;
                                                }
                                            }
                                            break;
                                    }
                                }
                                else if (currentDay.weekday === 6) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2015).value += permissionTime;
                                }
                                else if (currentDay.weekday === 7) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    if (permissionTime > 7.5) {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += 7.5;
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2020).value += permissionTime - 7.5;
                                    }
                                    else {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += permissionTime;
                                    }
                                }
                                else {
                                    if (permissionTime > 3) {
                                        totalExtraWorkOnWorkingDaysFirstThree += 3;
                                        totalExtraWorkOnWorkingDaysAfterFirstThree += permissionTime - 3;
                                    }
                                    else {
                                        totalExtraWorkOnWorkingDaysFirstThree += permissionTime;
                                    }
                                    normalWorkTime += (data.s.n + data.s.afb) / 60;
                                }
                            }
                        });
                    });
                    if (totalExtraWorkOnWorkingDaysFirstThree + normalWorkTime + totalExtraWorkOnWorkingDaysFirstThree > 45) {
                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2000).value +=
                            totalExtraWorkOnWorkingDaysFirstThree + normalWorkTime > 45 ? totalExtraWorkOnWorkingDaysFirstThree + normalWorkTime - 45 : 0;
                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2005).value += totalExtraWorkOnWorkingDaysAfterFirstThree;
                    }
                    break;
                case WorkType.SHIFT:
                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2060).value = countOfWorkingDays / 2;
                    weekData.forEach((data) => {
                        const extraWorkPermissions = data.p?.length > 0 ? data.p.filter((perm) => perm.pt === ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID) : undefined;
                        extraWorkPermissions?.forEach((extraWorkPerm) => {
                            const permDefinition = userInfo.permissions.find((item) => item.id === extraWorkPerm.p);
                            let permissionTime = this.hoursDiffBetweenToISOTime(permDefinition.startDateTime, permDefinition.endDateTime);
                            if (permDefinition.dynamicFormData?.ottype &&
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype))) {
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype)).value += permissionTime;
                            }
                            else {
                                const currentDay = luxon_1.DateTime.fromISO(data.d);
                                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(currentDay));
                                if (vacationExistency) {
                                    switch (vacationExistency.type) {
                                        case app_enums_1.enums.VacationType.BeforeNational:
                                        case app_enums_1.enums.VacationType.BeforeReligious:
                                            let workingPart = 0;
                                            let intersection;
                                            if (luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(data.d).set({ hour: 7 }), luxon_1.DateTime.fromISO(data.d).set({ hour: 9 })).contains(luxon_1.DateTime.fromISO(data.ex.s))) {
                                                data.c.forEach((segment) => {
                                                    if (segment.es.w === true) {
                                                        intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.e)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 8 }), luxon_1.DateTime.fromISO(currentDay.toISO()).plus({ days: 1 })));
                                                        workingPart += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                    }
                                                });
                                                if (workingPart > 0) {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2025).value +=
                                                        workingPart >= 7.5 && workingPart <= 8 ? 7.5 : workingPart;
                                                }
                                            }
                                            else if (luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(data.d).set({ hour: 15 }), luxon_1.DateTime.fromISO(data.d).set({ hour: 17 })).contains(luxon_1.DateTime.fromISO(data.ex.s))) {
                                                data.c.forEach((segment) => {
                                                    if (segment.es.w === true) {
                                                        intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.e)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 16 }), luxon_1.DateTime.fromISO(currentDay.toISO()).plus({ days: 1 })));
                                                        workingPart += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                    }
                                                });
                                                if (workingPart > 0) {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2030).value +=
                                                        workingPart >= 7.5 && workingPart <= 8 ? 7.5 : workingPart;
                                                }
                                            }
                                            else if (luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(data.d)
                                                .minus({ day: 1 })
                                                .set({ hour: 23 }), luxon_1.DateTime.fromISO(data.d)).contains(luxon_1.DateTime.fromISO(data.ex.s)) ||
                                                luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(data.d), luxon_1.DateTime.fromISO(data.d).set({ hour: 1 })).contains(luxon_1.DateTime.fromISO(data.ex.s))) {
                                                data.c.forEach((segment) => {
                                                    if (segment.es.w === true) {
                                                        intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.e)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()), luxon_1.DateTime.fromISO(currentDay.toISO()).plus({ days: 1 })));
                                                        workingPart += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                    }
                                                });
                                                if (workingPart > 0) {
                                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2035).value +=
                                                        workingPart >= 7.5 && workingPart <= 8 ? 7.5 : workingPart;
                                                }
                                            }
                                            break;
                                        default:
                                            if (permissionTime > 7.5) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += 7.5;
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2055).value += permissionTime - 7.5;
                                            }
                                            else {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += permissionTime;
                                            }
                                            break;
                                    }
                                }
                                else if (currentDay.weekday === 6) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2015).value += permissionTime;
                                }
                                else if (currentDay.weekday === 7) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    if (permissionTime > 7.5) {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += 7.5;
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2020).value += permissionTime - 7.5;
                                    }
                                    else {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2050).value += permissionTime;
                                    }
                                }
                                else {
                                    if (permissionTime > 3) {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2000).value += 3;
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2005).value += permissionTime - 3;
                                    }
                                    else {
                                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2000).value += permissionTime;
                                    }
                                }
                            }
                        });
                    });
                    break;
                default:
                    break;
            }
        });
    }
    applyNewUnionRules(weeklySegmentedData, userInfo, countOfWorkingDays) {
        const userWorkingType = this.getUserWorkingType(userInfo.extensionFields?.org_workplan_rule);
        if (!userWorkingType) {
            return;
        }
        weeklySegmentedData.forEach((weekData) => {
            switch (userWorkingType) {
                case WorkType.NON_SHIFT:
                    let totalExtraWorkOnWorkingDays = 0;
                    weekData.forEach((data) => {
                        const extraWorkPermissions = data.p?.length > 0 ? data.p.filter((perm) => perm.pt === ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID) : undefined;
                        extraWorkPermissions?.forEach((extraWorkPerm) => {
                            const permDefinition = userInfo.permissions.find((item) => item.id === extraWorkPerm.p);
                            let permissionTime = this.hoursDiffBetweenToISOTime(permDefinition.startDateTime, permDefinition.endDateTime);
                            permissionTime = Number((permissionTime - this.applyBreakRestRule(permissionTime)).toFixed(2));
                            if (permDefinition.dynamicFormData?.ottype &&
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype))) {
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype)).value += permissionTime;
                            }
                            else {
                                const currentDay = luxon_1.DateTime.fromISO(data.d);
                                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(currentDay));
                                if (vacationExistency) {
                                    switch (vacationExistency.type) {
                                        case app_enums_1.enums.VacationType.BeforeNational:
                                        case app_enums_1.enums.VacationType.BeforeReligious:
                                            let workingTime = 0;
                                            data.c.forEach((segment) => {
                                                if (segment.es.w === true) {
                                                    const intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.e)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 13 }), luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 18 })));
                                                    workingTime += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                }
                                            });
                                            if (currentDay.weekday === 6 || currentDay.weekday === 7) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2010).value +=
                                                    permissionTime > workingTime ? permissionTime - workingTime : 0;
                                            }
                                            else {
                                                totalExtraWorkOnWorkingDays += permissionTime > workingTime ? permissionTime - workingTime : 0;
                                            }
                                            this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2040).value += workingTime;
                                            break;
                                        default:
                                            if (permissionTime > 7.5) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2045).value += permissionTime - 7.5;
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2040).value += 7.5;
                                            }
                                            else {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2040).value += permissionTime;
                                            }
                                            break;
                                    }
                                }
                                else if (currentDay.weekday === 6 || currentDay.weekday === 7) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2010).value += permissionTime;
                                }
                                else if (data.ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours).length > 0) {
                                    totalExtraWorkOnWorkingDays += permissionTime + (data.s.n + data.s.afb) / 60;
                                }
                            }
                        });
                        if (!extraWorkPermissions && data.ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours).length > 0) {
                            totalExtraWorkOnWorkingDays += (data.s.n + data.s.afb) / 60;
                        }
                    });
                    if (totalExtraWorkOnWorkingDays > 45) {
                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2010).value += Number((totalExtraWorkOnWorkingDays - 45).toFixed(1));
                    }
                    break;
                case WorkType.SHIFT:
                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2060).value = countOfWorkingDays / 2;
                    weekData.forEach((data, index, array) => {
                        const extraWorkPermissions = data.p?.length > 0 ? data.p.filter((perm) => perm.pt === ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID) : undefined;
                        extraWorkPermissions?.forEach((extraWorkPerm) => {
                            const permDefinition = userInfo.permissions.find((item) => item.id === extraWorkPerm.p);
                            let permissionTime = this.hoursDiffBetweenToISOTime(permDefinition.startDateTime, permDefinition.endDateTime);
                            if (permDefinition.dynamicFormData?.ottype &&
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype))) {
                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype)).value += permissionTime;
                            }
                            else {
                                const currentDay = luxon_1.DateTime.fromISO(data.d);
                                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(currentDay));
                                if (vacationExistency) {
                                    switch (vacationExistency.type) {
                                        case app_enums_1.enums.VacationType.BeforeNational:
                                        case app_enums_1.enums.VacationType.BeforeReligious:
                                            let workingTime = 0;
                                            data.c.forEach((segment) => {
                                                if (segment.es.w === true) {
                                                    const intersection = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(segment.s), luxon_1.DateTime.fromISO(segment.e)).intersection(luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(currentDay.toISO()).set({ hour: 13 }), luxon_1.DateTime.fromISO(currentDay.toISO()).plus({ day: 1 })));
                                                    workingTime += intersection ? intersection.toDuration("hours").toObject().hours : 0;
                                                }
                                            });
                                            this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2040).value +=
                                                workingTime >= 7.5 && workingTime <= 8 ? 7.5 : workingTime;
                                            break;
                                        default:
                                            if (permissionTime > 7.5) {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2045).value += permissionTime - 7.5;
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2040).value += 7.5;
                                            }
                                            else {
                                                this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2040).value += permissionTime;
                                            }
                                            break;
                                    }
                                }
                                else if (currentDay.weekday === 6 || currentDay.weekday === 7) {
                                    if (permissionTime < 3) {
                                        permissionTime = 3;
                                    }
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2010).value += permissionTime;
                                }
                                else {
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2010).value += permissionTime;
                                }
                            }
                        });
                        if (array[index + 1]) {
                            const firstOfConsecutiveDays = data.ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours);
                            const secondOfConsecutiveDays = array[index + 1].ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours);
                            if (firstOfConsecutiveDays.length > 0 && secondOfConsecutiveDays.length > 0) {
                                const hoursBetweenTwoShifts = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(firstOfConsecutiveDays[firstOfConsecutiveDays.length - 1].s), luxon_1.DateTime.fromISO(secondOfConsecutiveDays[secondOfConsecutiveDays.length - 1]))
                                    .toDuration("hours")
                                    .toObject().hours;
                                if (hoursBetweenTwoShifts < 16) {
                                    this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === 2065).value += 7.5;
                                }
                            }
                        }
                    });
                    break;
            }
        });
    }
    applyWithoutUnionNetSalary(weeklySegmentedData, userInfo) { }
    applyWithoutUnionGrossSalary(weeklySegmentedData, userInfo) {
        weeklySegmentedData.forEach((weekData) => {
            let weeklyWorkingTime = 0;
            weekData.forEach((data) => {
                const extraWorkPermissions = data.p?.length > 0 ? data.p.filter((perm) => perm.pt === ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID) : undefined;
                extraWorkPermissions?.forEach((extraWorkPerm) => {
                    const permDefinition = userInfo.permissions.find((item) => item.id === extraWorkPerm.p);
                    let permissionTime = this.hoursDiffBetweenToISOTime(permDefinition.startDateTime, permDefinition.endDateTime);
                    if (permDefinition.dynamicFormData?.ottype &&
                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype))) {
                        this._userInfoMapping.get(userInfo.uniqueId).find((elem) => elem.type === parseInt(permDefinition.dynamicFormData?.ottype)).value += permissionTime;
                    }
                    else {
                        const currentDay = luxon_1.DateTime.fromISO(data.d);
                        const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(currentDay));
                        if (vacationExistency) {
                            switch (vacationExistency.type) {
                                case app_enums_1.enums.VacationType.National:
                                case app_enums_1.enums.VacationType.Religious:
                                    this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2040).value +=
                                        permissionTime >= 7.5 && permissionTime <= 8 ? 7.5 : permissionTime;
                                    break;
                            }
                        }
                        else if (currentDay.weekday === 6 || currentDay.weekday === 7) {
                            this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2010).value += permissionTime;
                        }
                        else {
                            weeklyWorkingTime += permissionTime + (data.s.n + data.s.afb) / 60;
                        }
                    }
                });
            });
            if (weeklyWorkingTime > 45) {
                this._userInfoMapping.get(userInfo.uniqueId).find((item) => item.type === 2010).value += weeklyWorkingTime - 45;
            }
        });
    }
    divideDataWeeklySegments(monthlyData, userInfo) {
        let lastMonthDays = monthlyData.filter((data) => {
            return luxon_1.DateTime.fromISO(data.d) < this._options.startDate;
        });
        let lastIndex = 0;
        lastMonthDays.forEach((elem, index) => {
            if (elem.ws.filter((elem) => elem.wt !== app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours).length === 0) {
                lastIndex = index;
            }
        });
        lastMonthDays = lastMonthDays.slice(lastIndex + 1);
        let currentMonthDays = monthlyData.filter((data) => {
            return luxon_1.DateTime.fromISO(data.d) >= this._options.startDate;
        });
        const dataWithoutOffset = [...lastMonthDays, ...currentMonthDays];
        const userWorkingType = this.getUserWorkingType(userInfo.extensionFields?.org_workplan_rule);
        const weeklySegmentedData = [];
        switch (userWorkingType) {
            case WorkType.NON_SHIFT:
                let startingPoint = 0;
                for (let i = 0; i < dataWithoutOffset.length; i++) {
                    const currentDay = luxon_1.DateTime.fromISO(dataWithoutOffset[i].d);
                    if (currentDay.weekday === 7) {
                        weeklySegmentedData.push(dataWithoutOffset.slice(startingPoint, i + 1));
                        startingPoint = i + 1;
                    }
                }
                break;
            case WorkType.SHIFT:
                let slicingPoint = 0;
                for (let i = 0; i < dataWithoutOffset.length - 1; i++) {
                    if (dataWithoutOffset[i].ws.length === 1 && dataWithoutOffset[i + 1].ws.length != 1) {
                        weeklySegmentedData.push(dataWithoutOffset.slice(slicingPoint, i + 2));
                        slicingPoint = i + 2;
                    }
                }
                break;
            default:
                break;
        }
        return weeklySegmentedData;
    }
    getUserWorkingType(orgWorkplanRule) {
        if (!orgWorkplanRule) {
            return false;
        }
        if (orgWorkplanRule === "NOR1" || orgWorkplanRule === "NOR2") {
            return WorkType.NON_SHIFT;
        }
        else {
            return WorkType.SHIFT;
        }
    }
    applyBreakRestRule(permissionTime) {
        for (const rest of constants_1.BreakRestRules) {
            if (permissionTime >= rest.hoursRange.start && permissionTime < rest.hoursRange.end) {
                return rest.breakTime;
            }
        }
    }
    hoursDiffBetweenToISOTime(start, end) {
        const diff = luxon_1.DateTime.fromISO(end).diff(luxon_1.DateTime.fromISO(start), "hours").toObject().hours;
        return diff;
    }
    initGeneralSheet(wsRaw, startColIndex) {
        const columns = [];
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE_NUMBER", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, "ÇALIŞAN ALT GRUBU", generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.START_DATE", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.FEE_TYPE", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        this.addInitCell(wsRaw, this._workSheetRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.TOTAL_HOUR", locale: this._locale }), generator_1.CustomReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 45 });
        columns.push({
            width: 45,
        });
        this._workSheetRowIndex++;
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: 1, activeCell: "A1" }];
    }
    addInitCell(ws, row, col, value, border) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow() {
        this._userInfoMapping.forEach((value, key) => {
            value.forEach((info) => {
                if (info.value > 0) {
                    let columnIndex = 1;
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: info.fullname,
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: key,
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: info.subGroup ?? "",
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: this._options.endDate.setLocale(this._locale).toFormat("dd.MM.yyyy"),
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: info.type,
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this.setDataCell({
                        rowIndex: this._workSheetRowIndex,
                        colIndex: columnIndex++,
                        value: info.value,
                        ws: this._wsGeneral,
                        border: generator_1.CustomReportGenerator.Constants.Styling.Dotted,
                    });
                    this._workSheetRowIndex++;
                }
            });
        });
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        if (params.formula) {
            c.value = {
                formula: params.formula,
            };
        }
        else {
            c.value = params.value;
        }
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell(ws, startRowIndex, startColIndex, "EK ÖDEMELER RAPORU", Constants.Styling.Header.border, Constants.Styling.Header.font);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, "AİT OLDUĞU AY VE YIL", Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderTwo.font);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toLocaleString({ day: "2-digit", month: "long", year: "numeric" }) +
            " - " +
            this._options.endDate.setLocale(this._locale).toLocaleString({ day: "2-digit", month: "long", year: "numeric" }), Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderThird.font, generator_1.CustomReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, "KURUM", Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderTwo.font);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, "ADM", Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderThird.font);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell(ws, startRowIndex + 3, startColIndex, "UYGULANAN FILTRELER", Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderTwo.font);
            this.addFilterCell(ws, startRowIndex + 3, startColIndex + 1, filter, Constants.Styling.HeaderTwo.border, Constants.Styling.HeaderThird.font);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.ADMSAPExtraWorkCustomReport = ADMSAPExtraWorkCustomReport;
ADMSAPExtraWorkCustomReport.EXTRA_WORK_PERMISSION_TYPE_ID = "82fb2946-ea17-474d-857d-8f1eea31ba3d";
async function generateReport(request, locale) {
    const report = new ADMSAPExtraWorkCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
