"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ADMAdditionalPaymentsCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const constants_1 = require("./constants");
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const utils_1 = require("./utils");
class ADMAdditionalPaymentsCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._daysTotal = Math.round(this._options.endDate.diff(this._options.startDate, "days").toObject().days);
        this._offsetDayCount = this._options.startDate.minus({ days: 1 }).day - 25;
    }
    async generateReport() {
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "ADM SAP Ek Ödemeler",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._vacations = await dal_manager_1.dbManager.accessPacs.listVacations(this._options.organizationId, this._options.startDate.get("year"));
        this._wsGeneral = this._wb.addWorksheet("Ek Ödemeler Raporu", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet("Uygulanan Filtreler", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        this._workSheetRowIndex = 1;
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        if (this._options.userIds && this._options.userIds.length > 0) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let userNames = await dal_manager_1.dbManager.accessUser.getBasicUserInfoPg({ organizationId: this._request.organizationId, userIds: this._options.userIds, trx });
                filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.map((elem) => elem.fullname).join(","));
            });
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, 1);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                userIds: this._options.userIds,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            let { query, bindings } = (0, utils_1.generateEmployeeDataQuery)(this._options.organizationId, userFilter, this._options.startDate.toSQL(), this._options.startDate.minus({ month: 1 }).set({ day: 26 }).startOf("day").toSQL(), this._options.endDate.toSQL(), this._options.endDate.set({ day: 26 }).toSQL());
            await trx.query(query, bindings);
            let { rows } = await trx.query("FETCH 100 FROM employee_day_cursor");
            while (rows.length > 0) {
                await this.addGeneralRow(rows, trx);
                rows = (await trx.query("FETCH 100 FROM employee_day_cursor")).rows;
            }
            await trx.query(`CLOSE employee_day_cursor;`);
        });
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + (this._workSheetRowIndex + 5);
        this._wsGeneral.pageSetup.printTitlesRow = "5:7";
        this._wsFilterSheet.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + (this._workSheetRowIndex + 5);
        this._wsFilterSheet.pageSetup.printTitlesRow = "5:7";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startColIndex) {
        const columns = [];
        let data = [
            {
                phrase: i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE_NUMBER", locale: this._locale }),
                cellWidth: 45,
            },
            { value: "Çalışan Alt Grubu", cellWidth: 45 },
            {
                phrase: i18n_1.default.__({ phrase: "EXCEL-REPORT.START_DATE", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "EXCEL-REPORT.FEE_TYPE", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.DAY_COUNT", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.AMOUNT", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.CURRENCY", locale: this._locale }),
                cellWidth: 45,
            },
            {
                phrase: i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.COMMENT", locale: this._locale }),
                cellWidth: 45,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, this._workSheetRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + this._workSheetRowIndex,
            to: wsRaw.getColumn(columns.length).letter + this._workSheetRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: 1, activeCell: "A1", showGridLines: false }];
        this._workSheetRowIndex++;
    }
    async addGeneralRow(userInfo, trx) {
        for (const user of userInfo) {
            const employmentEndUtc = luxon_1.DateTime.fromJSDate(user.employmentEndUtc);
            let feeCount = user.employmentEndUtc && luxon_1.DateTime.now().hasSame(employmentEndUtc, "month") && luxon_1.DateTime.now().hasSame(employmentEndUtc, "year")
                ? Math.round(employmentEndUtc.diff(this._options.startDate, "days").toObject().days)
                : this._daysTotal;
            let copyingBackWorkPlanDefinitionCounter = 0;
            if (luxon_1.DateTime.now() < this._options.endDate) {
                for (let i = user.monthlyData.length; i < this._daysTotal + this._offsetDayCount; i++) {
                    if (copyingBackWorkPlanDefinitionCounter === 7) {
                        copyingBackWorkPlanDefinitionCounter = 0;
                    }
                    const data = user.monthlyData[user.monthlyData.length - 7 + copyingBackWorkPlanDefinitionCounter++];
                    user.countOfWorkingDays += data?.s.wd > 0 ? 1 : 0;
                }
            }
            for (const feeTypeRule of constants_1.ADMFeeTypeRules) {
                let feeCountToBeCalculated = feeCount;
                const rule = feeTypeRule.rules.find((rule) => {
                    if (rule.extensionFieldRules.every((extField) => {
                        return extField.validValues.includes(user.extensionFields[extField.key]);
                    })) {
                        return rule;
                    }
                });
                if (rule) {
                    let columnIndex = 1;
                    if (feeTypeRule.type === constants_1.feeType.YEMEK_UCRETI_NET || feeTypeRule.type === constants_1.feeType.YEMEK_UCRETI_NET_YEMEKHANE) {
                        feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated, this._options.organizationId, this._options.startDate.toSQL(), this._options.endDate.toSQL(), trx);
                    }
                    else {
                        feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated);
                    }
                    const rowData = [
                        user.fullname,
                        user.uniqueId,
                        user.extensionFields ? constants_1.ADM_EMPLOYEE_SUB_GROUPS[user.extensionFields["org_employee_subgroup"]] : "",
                        user.employmentEndUtc && luxon_1.DateTime.now().hasSame(employmentEndUtc, "month") && luxon_1.DateTime.now().hasSame(employmentEndUtc, "year")
                            ? employmentEndUtc.toFormat("MM/dd/yyyy")
                            : this._options.endDate.toFormat("MM/dd/yyyy"),
                        feeTypeRule.type,
                        feeCountToBeCalculated > 0 ? feeCountToBeCalculated : 0,
                        "",
                        "TRY",
                        feeTypeRule.name,
                    ];
                    this.addCellsToRow(rowData, columnIndex, this._workSheetRowIndex, this._wsGeneral);
                    this._workSheetRowIndex++;
                }
            }
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this.addFilterCell(ws, startRowIndex, startColIndex, "EK ÖDEMELER RAPORU", constants_1.HeaderStyle.border, constants_1.HeaderStyle.font);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, "AİT OLDUĞU AY VE YIL", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, "KURUM", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, "ADM", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this.addFilterCell(ws, startRowIndex + 3, startColIndex, "UYGULANAN FILTRELER", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
            this.addFilterCell(ws, startRowIndex + 3, startColIndex + 1, filter, constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.ADMAdditionalPaymentsCustomReport = ADMAdditionalPaymentsCustomReport;
async function generateReport(request, locale) {
    const report = new ADMAdditionalPaymentsCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
