"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateHook = void 0;
const app_logs_1 = require("../../../app.logs");
const business_hooks_1 = require("../../../business/business.hooks");
const keycloak_api_1 = require("../keycloak/keycloak-api");
const dal_manager_1 = require("../../../dal/dal.manager");
const restapi_1 = require("../../../lib/es/models/restapi");
const app_enums_1 = require("../../../app.enums");
const crm_utils_1 = require("../crm/crm-utils");
const crm_queue_1 = require("../crm/crm-queue");
const taskqueue_1 = require("../../../utils/taskqueue");
const notificationQ = new taskqueue_1.TaskQueue(2000, "EnerjiSA CRM");
function CreateHook() {
    return new EnerjiSADagitimHook();
}
exports.CreateHook = CreateHook;
const emailRegex = /^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$/;
const organizationId = "cf1a899a-bb3f-4379-9b25-4c6d7a89a80f";
const keyCloakSecretId = "1eeadb05-5411-4997-b1a2-8a093e57de78";
class EnerjiSADagitimHook extends business_hooks_1.OrganizationHookModels.IArmonHook {
    constructor() {
        super();
        this.beforeUserInsert = async function (params) {
            if (params.identity.organizationProfile.email) {
                const regexCheck = emailRegex.test(params.identity.organizationProfile.email.toLowerCase());
                if (!regexCheck) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                        errorMessage: `Email address is not valid! Please check email address provided!`,
                    };
                }
                const searchResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityExact(organizationId, {
                    userOrganizationProfile: {
                        email: params.identity.organizationProfile.email,
                    },
                });
                if (searchResult.items.length) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        errorMessage: `There is already a${searchResult.items[0].isDisabled ? " passive" : "n active"} user registered with this mail address, therefore user cannot be added with this email address`,
                    };
                }
            }
            return {
                isValid: true,
                errorMessage: null,
            };
        };
        this.beforeUserUpdate = async function (params) {
            if (params.identity.organizationProfile.email) {
                const regexCheck = emailRegex.test(params.identity.organizationProfile.email.toLowerCase());
                if (!regexCheck) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                        errorMessage: `Email address is not valid! Please check email address provided!`,
                    };
                }
                const searchResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityExact(organizationId, {
                    userOrganizationProfile: {
                        email: params.identity.organizationProfile.email,
                    },
                });
                let duplicateUser = searchResult.items.find((f) => f.id !== params.identity.id);
                if (duplicateUser) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        errorMessage: `There is already a${duplicateUser.isDisabled ? "passive" : "n active"} user registered with this mail address, therefore user cannot be updated with this email address`,
                    };
                }
            }
            return {
                isValid: true,
                errorMessage: null,
            };
        };
        this.afterIdentityUpsert = async function (oldIdentity, newIdentity) {
            if (!newIdentity.organizationProfile.email) {
                app_logs_1.logger.warn(`Email field is required for KeyCloak integration, but not supplied in armon, skipping user synchronization!`);
                return true;
            }
            let integratorInstance = await keycloak_api_1.KeyCloakAPI.getInstance(organizationId, keyCloakSecretId);
            if (!integratorInstance) {
                return false;
            }
            let checkUserResult;
            if (newIdentity.organizationProfile.isDisabled === true) {
                checkUserResult = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
                let disableResult = true;
                if (checkUserResult) {
                    app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} disabled, attempting to also disable at KeyCloak`);
                    disableResult = await disableKeyCloakUser(integratorInstance, newIdentity, checkUserResult);
                }
                else {
                    app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} does not exists at KeyCloak, skipping`);
                }
                return disableResult;
            }
            else if (oldIdentity &&
                oldIdentity.organizationProfile.email &&
                newIdentity.organizationProfile.email &&
                oldIdentity.organizationProfile.email !== newIdentity.organizationProfile.email) {
                app_logs_1.logger.warn(`Email address of user ${newIdentity.organizationProfile.uniqueId} is updated from ${oldIdentity.organizationProfile.email} to ${newIdentity.organizationProfile.email} !!!!`);
                checkUserResult = await integratorInstance.getUserByEmail(oldIdentity.organizationProfile.email);
                if (checkUserResult) {
                    app_logs_1.logger.warn(`Old email address ${oldIdentity.organizationProfile.email} of user ${newIdentity.organizationProfile.uniqueId} exists at KeyCloak, attempting to disable`);
                    await disableKeyCloakUser(integratorInstance, oldIdentity, checkUserResult);
                }
                else {
                    app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} does not exists at KeyCloak, skipping`);
                }
            }
            checkUserResult = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
            try {
                if (!checkUserResult) {
                    app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} does not exist at KeyCloak, trying to create new user`);
                    const addUserResult = await integratorInstance.addUser(newIdentity);
                    const newUser = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
                    const updatePasswordCallResult = await integratorInstance.updatePasswordCall(newUser.id);
                    const getClientIdResult = await integratorInstance.getClientId();
                    if (!getClientIdResult) {
                        app_logs_1.logger.error(`Error while getting client id!`);
                        return false;
                    }
                    const getKeyCloakRoleResult = await integratorInstance.getDefaultRole(getClientIdResult);
                    if (!getKeyCloakRoleResult) {
                        app_logs_1.logger.error(`Error while getting client role!`);
                        return false;
                    }
                    const mapWithRoleResult = await integratorInstance.mapUserWithRole(newUser.id, getClientIdResult, getKeyCloakRoleResult);
                    if (!addUserResult.success || !mapWithRoleResult.success || !updatePasswordCallResult.success || !mapWithRoleResult.success) {
                        app_logs_1.logger.error(addUserResult.errorMessage || mapWithRoleResult.errorMessage || updatePasswordCallResult.errorMessage || mapWithRoleResult.errorMessage);
                        return false;
                    }
                    app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully created at KeyCloak!`);
                }
                else {
                    if (checkUserResult.attributes?.source && checkUserResult.attributes?.source.length && checkUserResult.attributes?.source[0] === "armon") {
                        app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} previously sent to KeyCloak, updating!`);
                        const updateUserResult = await integratorInstance.updateUser({
                            id: checkUserResult.id,
                            attributes: {
                                mobile_number: [newIdentity.organizationProfile.phoneNumber],
                                source: ["armon"],
                            },
                            email: newIdentity.organizationProfile.email,
                            enabled: true,
                            firstName: newIdentity.organizationProfile.name,
                            lastName: newIdentity.organizationProfile.surname,
                            username: checkUserResult.username,
                        });
                        if (!checkUserResult.enabled) {
                            await integratorInstance.updatePasswordCall(checkUserResult.id);
                        }
                        if (!updateUserResult.success) {
                            app_logs_1.logger.error(`Newly added or updated armon user with email address ${newIdentity.organizationProfile.email} could not be updated at KeyCloak!`);
                            app_logs_1.logger.error(`This case should not occur under normal circumstances!`);
                            return false;
                        }
                        app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully updated at KeyCloak!`);
                    }
                    else {
                        app_logs_1.logger.warn(`User exists at KeyCloak with email address ${newIdentity.organizationProfile.email} but source is not armon. Can not add or modify user!`);
                    }
                    const userRoleMappingCheckResult = await integratorInstance.checkUserRoleMappings(checkUserResult.id);
                    if (!userRoleMappingCheckResult) {
                        const getClientIdResult = await integratorInstance.getClientId();
                        if (!getClientIdResult) {
                            app_logs_1.logger.error(`Error while getting client id!`);
                            return false;
                        }
                        const getKeyCloakRoleResult = await integratorInstance.getDefaultRole(getClientIdResult);
                        if (!getKeyCloakRoleResult) {
                            app_logs_1.logger.error(`Error while getting client role!`);
                            return false;
                        }
                        const mapWithRoleResult = await integratorInstance.mapUserWithRole(checkUserResult.id, getClientIdResult, getKeyCloakRoleResult);
                        if (!mapWithRoleResult.success) {
                            app_logs_1.logger.error(`Cannot map user with email address ${newIdentity.organizationProfile.email} with role ${getKeyCloakRoleResult.name}!`);
                            return false;
                        }
                        app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully mapped with role at KeyCloak!`);
                    }
                }
                app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully synced to KeyCloak in accordance with predefined conditions!`);
                return true;
            }
            catch (error) {
                app_logs_1.logger.error(`Error while running KeyCloak user synchronization!!!`);
                app_logs_1.logger.error(error);
            }
        };
        this.beforeUpdateOrganizationSettings = async function (settings) {
            if (settings.webRtc?.iceServers?.length || settings.webRtc?.twilio?.enabled) {
                return {
                    isValid: false,
                    errorMessage: "ERRORS.CUSTOM.ENERJISA_WEBRTC",
                };
            }
            else {
                return {
                    isValid: true,
                    errorMessage: null,
                };
            }
        };
        this.afterGetOrganizationSettings = async function (settings) {
            if (settings.webRtc?.iceServers?.length || settings.webRtc?.twilio.enabled) {
                settings.webRtc.iceServers = [];
                settings.webRtc.twilio = {
                    enabled: false,
                    accountSid: null,
                    authToken: null,
                };
            }
        };
        this.sendNotificationOverride = async function (params) {
            if (params.medium === restapi_1.NotificationMedium.Web || params.medium === restapi_1.NotificationMedium.PushNotification) {
                return {
                    isImplemented: false,
                };
            }
            let transformResult = null;
            try {
                transformResult = await (0, crm_utils_1.transformNotificationToCRMParameters)(params.message.o, { instanceId: params.notification.instanceId, medium: params.medium, message: params.message }, params.trx);
            }
            catch (error) {
                app_logs_1.logger.error(`Error while transforming notification parameters to CRM parameters. Notification Instance Id: ${params.notification.instanceId}`);
                app_logs_1.logger.error(error);
                app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.notification.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                        message: `Error while transforming notification parameters to CRM parameters. Error Details: ${error.toString()}`,
                    },
                    trx: params.trx,
                });
                return {
                    isImplemented: true,
                };
            }
            if (!transformResult.isImplemented) {
                return {
                    isImplemented: false,
                };
            }
            else {
                if (!transformResult.transformedParameters.reciever) {
                    app_logs_1.logger.error(`No valid recievers for notification via CRM. Notification Instance Id: ${params.notification.instanceId}`);
                    app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                        instanceId: params.notification.instanceId,
                        organizationId: params.organizationId,
                        state: 3,
                        actionNote: {
                            errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                            message: `No valid recievers for notification via CRM`,
                        },
                        trx: params.trx,
                    });
                    return {
                        isImplemented: true,
                    };
                }
            }
            if (!transformResult.transformedParameters.campaignCode) {
                app_logs_1.logger.error(`Campaign Code could not be determined for CRM Notification with instance id: ${params.notification.instanceId}`);
                app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.notification.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                        message: `Campaign Code could not be determined for CRM Notification. Please check notification instance data`,
                    },
                    trx: params.trx,
                });
                return {
                    isImplemented: true,
                };
            }
            notificationQ.push(new crm_queue_1.CRMNotificationTask({
                medium: params.medium,
                message: params.message,
                notification: params.notification,
                organizationId: params.organizationId,
            }, transformResult));
            return {
                isImplemented: true,
            };
        };
    }
}
const disableKeyCloakUser = async (integrator, armonIdentity, keyCloakCheckResult) => {
    if (keyCloakCheckResult.attributes?.source && keyCloakCheckResult.attributes?.source?.length && keyCloakCheckResult.attributes?.source[0] === "armon") {
        const updateUserResult = await integrator.updateUser({
            id: keyCloakCheckResult.id,
            attributes: {
                mobile_number: [armonIdentity.organizationProfile.phoneNumber],
                source: ["armon"],
            },
            email: armonIdentity.organizationProfile.email,
            enabled: false,
            firstName: armonIdentity.organizationProfile.name,
            lastName: armonIdentity.organizationProfile.surname,
            username: armonIdentity.username,
        });
        if (!updateUserResult) {
            app_logs_1.logger.error(`Error while attempting to disable user with email address ${armonIdentity.organizationProfile.email} even if the source is armon`);
            app_logs_1.logger.error(updateUserResult.errorMessage);
            return false;
        }
    }
    else {
        app_logs_1.logger.warn(`Old email address ${armonIdentity.organizationProfile.email} of user is not created by armon, cannot disable user, therefore ignoring`);
    }
    const getClientIdResult = await integrator.getClientId();
    if (!getClientIdResult) {
        app_logs_1.logger.error(`Error while getting client id!`);
        return false;
    }
    await integrator.removeUserRoleMapping(keyCloakCheckResult.id, getClientIdResult);
    return true;
};
