"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyCloakAPI = void 0;
const axios_1 = __importDefault(require("axios"));
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_access_psql_organization_1 = require("../../../dal/access/psql/dal.access.psql.organization");
class KeyCloakAPI {
    constructor() { }
    static async getInstance(organizationId, keyCloakSecretId) {
        if (!KeyCloakAPI.instance) {
            KeyCloakAPI.instance = new KeyCloakAPI();
            const keyCloakSecrets = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return await (0, dal_access_psql_organization_1.getOrganizationSecretById)(organizationId, keyCloakSecretId, trx);
            });
            if (keyCloakSecrets) {
                KeyCloakAPI.instance.keyCloakBaseUrl = keyCloakSecrets.keyCloakBaseUrl;
                KeyCloakAPI.instance.keyCloakAdminClientId = keyCloakSecrets.keyCloakAdminClientId;
                KeyCloakAPI.instance.keyCloakApplicationClientId = keyCloakSecrets.keyCloakApplicationClientId;
                KeyCloakAPI.instance.keyCloakDefaultRoleName = keyCloakSecrets.keyCloakDefaultRoleName;
                KeyCloakAPI.instance.keyCloakPassword = keyCloakSecrets.keyCloakPassword;
                KeyCloakAPI.instance.keyCloakRealm = keyCloakSecrets.keyCloakRealm;
                KeyCloakAPI.instance.keyCloakUsername = keyCloakSecrets.keyCloakUsername;
            }
            else {
                app_logs_1.logger.error(`KeyCloak Secrets could not be obtained from database, operation aborted`);
                return null;
            }
        }
        return KeyCloakAPI.instance;
    }
    async renewAccessToken() {
        const url = `${this.keyCloakBaseUrl}/realms/${this.keyCloakRealm}/protocol/openid-connect/token`;
        let response = null;
        let isRefreshed = false;
        if (KeyCloakAPI.refreshToken && KeyCloakAPI.refreshTokenExpiration > new Date()) {
            try {
                const requestParams = {
                    grant_type: "refresh_token",
                    refresh_token: KeyCloakAPI.refreshToken,
                    client_id: this.keyCloakAdminClientId,
                };
                response = await axios_1.default.post(url, new URLSearchParams(requestParams), {
                    headers: {
                        "Content-Type": "application/x-www-form-urlencoded",
                    },
                });
                isRefreshed = true;
            }
            catch (err) {
                app_logs_1.logger.error(`Even if refresh token seems active at armon, refresh operation failed at keycloak, relogin!!`);
                app_logs_1.logger.error(err);
            }
        }
        if (!isRefreshed) {
            const requestParams = {
                grant_type: "password",
                username: this.keyCloakUsername,
                password: this.keyCloakPassword,
                client_id: this.keyCloakAdminClientId,
            };
            response = await axios_1.default.post(url, new URLSearchParams(requestParams), {
                headers: {
                    "Content-Type": "application/x-www-form-urlencoded",
                },
            });
        }
        if (response) {
            KeyCloakAPI.accessToken = response.data.access_token;
            KeyCloakAPI.accessTokenExpiration = new Date(Date.now() + response.data.expires_in * 1000 - 10000);
            KeyCloakAPI.refreshToken = response.data.refresh_token;
            KeyCloakAPI.refreshTokenExpiration = new Date(Date.now() + response.data.refresh_expires_in * 1000 - 10000);
        }
        else {
            app_logs_1.logger.error(`Can not get access token for KeyCloak integration! Check server connection`);
            KeyCloakAPI.accessToken = null;
            KeyCloakAPI.accessTokenExpiration = null;
            KeyCloakAPI.refreshToken = null;
            KeyCloakAPI.refreshTokenExpiration = null;
        }
    }
    async sendDeleteRequest(reqUrl) {
        try {
            if (!KeyCloakAPI.accessToken || !KeyCloakAPI.accessTokenExpiration || KeyCloakAPI.accessTokenExpiration < new Date()) {
                await this.renewAccessToken();
            }
            const response = await axios_1.default.delete(reqUrl, {
                headers: {
                    "Content-Type": "application/json",
                    Authorization: "Bearer " + KeyCloakAPI.accessToken,
                },
            });
            return response;
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending DELETE request to KeyCloak!`);
            app_logs_1.logger.error(`Error url: ${reqUrl}`);
            if (error.response) {
                app_logs_1.logger.error(JSON.stringify(error.response.data, null, 4));
                app_logs_1.logger.error(error.response.status);
                app_logs_1.logger.error(error);
                return error.response;
            }
            app_logs_1.logger.error(error);
            return;
        }
    }
    async sendPostRequest(reqUrl, reqParams) {
        try {
            if (!KeyCloakAPI.accessToken || !KeyCloakAPI.accessTokenExpiration || KeyCloakAPI.accessTokenExpiration < new Date()) {
                await this.renewAccessToken();
            }
            const response = await axios_1.default.post(reqUrl, reqParams, {
                headers: {
                    "Content-Type": "application/json",
                    Authorization: "Bearer " + KeyCloakAPI.accessToken,
                },
            });
            return response;
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending POST request to KeyCloak!`);
            app_logs_1.logger.error(`Error url: ${reqUrl}, params: ${JSON.stringify(reqParams, null, 4)}`);
            if (error.response) {
                app_logs_1.logger.error(JSON.stringify(error.response.data, null, 4));
                app_logs_1.logger.error(error.response.status);
                app_logs_1.logger.error(error);
                return error.response;
            }
            app_logs_1.logger.error(error);
            return;
        }
    }
    async sendGetRequest(reqUrl, reqParams) {
        try {
            if (!KeyCloakAPI.accessToken || !KeyCloakAPI.accessTokenExpiration || KeyCloakAPI.accessTokenExpiration < new Date()) {
                await this.renewAccessToken();
            }
            const response = await axios_1.default.get(reqUrl, {
                params: reqParams,
                headers: {
                    Authorization: "Bearer " + KeyCloakAPI.accessToken,
                },
            });
            return response;
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending GET request to KeyCloak!`);
            app_logs_1.logger.error(`Error url: ${reqUrl}, params: ${JSON.stringify(reqParams, null, 4)}`);
            if (error.response) {
                app_logs_1.logger.error(JSON.stringify(error.response.data, null, 4));
                app_logs_1.logger.error(error.response.status);
                app_logs_1.logger.error(error);
                return error.response;
            }
            app_logs_1.logger.error(error);
            return;
        }
    }
    async sendPutRequest(reqUrl, reqParams) {
        try {
            if (!KeyCloakAPI.accessToken || !KeyCloakAPI.accessTokenExpiration || KeyCloakAPI.accessTokenExpiration < new Date()) {
                await this.renewAccessToken();
            }
            const response = await axios_1.default.put(reqUrl, reqParams, {
                headers: {
                    "Content-Type": "application/json",
                    Authorization: "Bearer " + KeyCloakAPI.accessToken,
                },
            });
            return response;
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending PUT request to KeyCloak!`);
            app_logs_1.logger.error(`Error url: ${reqUrl}, params: ${JSON.stringify(reqParams, null, 4)}`);
            if (error.response) {
                app_logs_1.logger.error(JSON.stringify(error.response.data, null, 4));
                app_logs_1.logger.error(error.response.status);
                app_logs_1.logger.error(error);
                return error.response;
            }
            app_logs_1.logger.error(error);
            return;
        }
    }
    async addUser(identity) {
        const url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users`;
        const requestParams = {
            username: identity.organizationProfile.email,
            email: identity.organizationProfile.email,
            firstName: identity.organizationProfile.name,
            lastName: identity.organizationProfile.surname,
            enabled: true,
            attributes: {
                mobile_number: identity.organizationProfile.phoneNumber ? [identity.organizationProfile.phoneNumber] : [],
                source: ["armon"],
            },
        };
        const response = await this.sendPostRequest(url, requestParams);
        return {
            success: response?.status >= 200 && response?.status < 300,
            errorMessage: response?.data,
        };
    }
    async getUserByEmail(email) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users`;
        const queryParams = {
            email: email,
            exact: true,
        };
        const response = await this.sendGetRequest(url, queryParams);
        if (response?.status >= 200 && response?.status < 300) {
            return response?.data?.length ? response.data[0] : null;
        }
        else {
            return null;
        }
    }
    async updateUser(user) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users/${user.id}`;
        const requestParams = user;
        const response = await this.sendPutRequest(url, requestParams);
        return {
            success: response?.status >= 200 && response?.status < 300,
            errorMessage: response?.data,
        };
    }
    async updatePasswordCall(userId) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users/${userId}/execute-actions-email?lifespan=86400`;
        const requestParams = ["UPDATE_PASSWORD"];
        const response = await this.sendPutRequest(url, requestParams);
        return {
            success: response?.status >= 200 && response?.status < 300,
            errorMessage: response?.data,
        };
    }
    async getClientId() {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/clients`;
        const requestParams = {
            clientId: this.keyCloakApplicationClientId,
        };
        const response = await this.sendGetRequest(url, requestParams);
        if (response?.status >= 200 && response?.status < 300) {
            return response?.data?.length ? response.data[0].id : null;
        }
        else {
            return null;
        }
    }
    async getDefaultRole(clientId) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/clients/${clientId}/roles/${this.keyCloakDefaultRoleName}`;
        const response = await this.sendGetRequest(url, null);
        if (response?.status >= 200 && response?.status < 300) {
            return response?.data ?? null;
        }
        else {
            return null;
        }
    }
    async mapUserWithRole(userId, clientId, role) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users/${userId}/role-mappings/clients/${clientId}`;
        const requestParams = [role];
        const response = await this.sendPostRequest(url, requestParams);
        return {
            success: response?.status >= 200 && response?.status < 300,
            errorMessage: response?.data,
        };
    }
    async removeUserRoleMapping(userId, clientId) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users/${userId}/role-mappings/clients/${clientId}`;
        const response = await this.sendDeleteRequest(url);
        if (response?.status === 200) {
            app_logs_1.logger.info(`removeUserRoleMapping run successfully`);
        }
        else {
            app_logs_1.logger.error(`Error while removeUserRoleMapping`);
        }
    }
    async checkUserRoleMappings(userId) {
        let url = `${this.keyCloakBaseUrl}/admin/realms/${this.keyCloakRealm}/users/${userId}/role-mappings/`;
        const response = await this.sendGetRequest(url, null);
        if (response?.status >= 200 &&
            response?.status < 300 &&
            response?.data?.clienMappings?.[this.keyCloakApplicationClientId]?.mappings?.find((f) => f.name === this.keyCloakDefaultRoleName)) {
            return true;
        }
        else {
            return false;
        }
    }
}
exports.KeyCloakAPI = KeyCloakAPI;
