"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ElimkoMonthlyPermissionCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.AllThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thick",
            },
            bottom: {
                style: "thick",
            },
            top: {
                style: "thick",
            },
        };
        Styling.BottomThin = {
            bottom: {
                style: "thin",
            },
        };
        Styling.BottomThinRightDotted = {
            right: {
                style: "dotted",
            },
            bottom: {
                style: "thin",
            },
        };
        Styling.AllDottedButTop = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllDottedButBottom = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
class ElimkoMonthlyPermissionCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._permissionsEmployeeDay = [];
        this._permissionTypesColIndex = [];
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._options = {
            startDate: (0, moment_1.default)(this._filter.filters.find((i) => i.type === 10).value).startOf("month"),
            endDate: (0, moment_1.default)(this._filter.filters.find((i) => i.type === 10).value).endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === 4).value,
            userGroupIds: this._filter.filters.find((i) => i.type === 3).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._pool = dal_manager_1.dbManager.poolMain;
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("[Custom Report] Fetch data %s rows", rows.length);
        this._permissionsEmployeeDay.push(...rows);
        return Promise.resolve();
    }
    async generateReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._options.organizationId);
        this._wsGeneral = this._wb.addWorksheet("Aylık Puantaj", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._generalRowIndex = 8;
        let userFilter = {
            organizationUnitIds: this._options.organizationUnitIds,
            userGroupIds: this._options.userGroupIds,
            userIds: null,
            userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
        };
        this._permissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(this._options.organizationId, {
            take: 1000,
            skip: 0,
        })).items;
        this.initGeneralSheet(this._wsGeneral, 7, 1);
        let activeUsers = (await dal_manager_1.dbManager.accessIdentity.searchIdentity({
            organizationId: this._options.organizationId,
            take: null,
            skip: null,
            status: app_enums_1.enums.IdentityStatusType.Enabled,
            profileFilter: null,
            organizationUnits: [],
            userGroupIds: [],
            credentialExtensionFilters: [],
            hasMail: null,
            permissions: [],
            workplans: [],
        })).items;
        await dal_manager_1.dbManager.accessPacs2.getAllPermissionsBetweenDates(this._options.organizationId, this._options.requesterUserId, this._options.startDate, this._options.endDate, userFilter, this.onDataCallbackFunction.bind(this));
        let filteredUserIds = this._permissionsEmployeeDay.map((e) => e.userId);
        activeUsers = activeUsers.filter((u) => filteredUserIds.includes(u.id));
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            this._organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.listOrganizationUnits(this._options.organizationId, {
                take: null,
                skip: null,
                organizationUnitIds: this._options.organizationUnitIds,
            });
        }
        else {
            let orgUnitIdsFromData = [];
            for (const usr of activeUsers) {
                if (usr.organizationUnits && usr.organizationUnits.length > 0) {
                    usr.organizationUnits.forEach((element) => {
                        orgUnitIdsFromData.push(element.id);
                    });
                }
            }
            this._organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.listOrganizationUnits(this._options.organizationId, {
                take: null,
                skip: null,
                organizationUnitIds: orgUnitIdsFromData,
            });
        }
        this._filePath = await this.generateDailyFirstInLastOutReport(activeUsers);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "Aylık Puantaj Raporu",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateDailyFirstInLastOutReport(userList) {
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("[Custom Report] Requester User : " + this._requestedUserCaption);
        app_logs_1.logger.debug("[Custom Report] Generating report...");
        let filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
        }
        if (this._options.userGroupIds && this._options.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._options.organizationId, this._options.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(", "));
        }
        app_logs_1.logger.debug("[Custom Report] Applied Filters:");
        app_logs_1.logger.debug(filters.join("\r\n"));
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters);
        if (this._permissionsEmployeeDay && this._permissionsEmployeeDay.length > 0) {
            for (const orgUnit of this._organizationUnits.items) {
                let usersInOrgUnit = userList.filter((u) => u.organizationUnits && u.organizationUnits.length > 0 && u.organizationUnits.map((e) => e.id).includes(orgUnit.id));
                if (usersInOrgUnit && usersInOrgUnit.length > 0) {
                    for (const user of usersInOrgUnit) {
                        let userData = this._permissionsEmployeeDay.find((d) => d.userId === user.id);
                        if (userData.permissions.length > 1) {
                            if ((0, moment_1.default)(userData.permissions[0][0].date).isBefore((0, moment_1.default)(userData.permissions[1][0].date))) {
                                userData.permissions[0] = userData.permissions[0].concat(userData.permissions[1]);
                                this.addGeneralRow(userData, user, orgUnit.captionLines[0]);
                            }
                            else {
                                userData.permissions[0] = userData.permissions[1].concat(userData.permissions[0]);
                                this.addGeneralRow(userData, user, orgUnit.captionLines[0]);
                            }
                        }
                        else {
                            this.addGeneralRow(userData, user, orgUnit.captionLines[0]);
                        }
                    }
                }
            }
        }
        else {
            app_logs_1.logger.debug("[Custom Report] No available data !");
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex + 5;
        this._wsGeneral.pageSetup.printTitlesRow = "5:7";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("[Custom Report] File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS_USER_GROUP", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 25 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS_ORG_UNIT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 25 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 35 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "SSK Gün", Constants.Styling.AllThin);
        columns.push({ width: 15 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Toplam Gün", Constants.Styling.AllThin);
        columns.push({ width: 15 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Normal Gün", Constants.Styling.AllThin);
        columns.push({ width: 15 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "PDF-REPORT.less_working_with_permission", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 15 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.OVERTIME", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 15 });
        for (const permissionType of this._permissionTypes) {
            this._permissionTypesColIndex.push({
                id: permissionType.id,
                index: startColIndex,
            });
            this.addInitCell(wsRaw, startRowIndex, startColIndex++, permissionType.name, Constants.Styling.AllThin);
            columns.push({ width: 15 });
        }
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(userData, userInfo, organizationUnit) {
        let days = userData.permissions[0];
        if (!days) {
            return;
        }
        let totals = [];
        this._permissionTypes.forEach((pt) => {
            totals.push({
                permissionId: pt.id,
                total: 0,
                isDailyScheduled: false,
            });
        });
        let colIndex = 1;
        let userGroups = "";
        if (userInfo.userGroups && userInfo.userGroups.length > 0) {
            userGroups += userInfo.userGroups.map((ug) => ug.name).join(", ");
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userGroups,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: organizationUnit,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userInfo.uniqueId,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userInfo.fullname.toUpperCase(),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userData.totaldays.reduce((a, b) => a + b, 0),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userData.totaldays.reduce((a, b) => a + b, 0),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userData.workeddays.reduce((a, b) => a + b, 0),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        let totalMissingWork = 0;
        let totalExtraWork = 0;
        if (days && days.length > 0) {
            for (const day of days) {
                let daySummary = day.summary;
                if (daySummary.m) {
                    totalMissingWork += daySummary.m;
                }
                if (daySummary.e) {
                    totalExtraWork += daySummary.e;
                }
                if (day.permission && day.permission.length > 0) {
                    for (const prms of day.permission) {
                        let permissionType = this._permissionTypes.find((p) => p.id === prms.pt);
                        if (permissionType && permissionType.isDailyScheduled !== null && permissionType.isDailyScheduled) {
                            let objIndex = totals.findIndex((obj) => obj.permissionId == prms.pt);
                            totals[objIndex].total += 1;
                            totals[objIndex].isDailyScheduled = true;
                        }
                        else {
                            let objIndex = totals.findIndex((obj) => obj.permissionId == prms.pt);
                            totals[objIndex].total += prms.u;
                        }
                    }
                }
            }
        }
        if (totalMissingWork > 0) {
            let hour = Math.floor(totalMissingWork / 60)
                .toString()
                .padStart(2, "0");
            let min = (totalMissingWork % 60).toString().padStart(2, "0");
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: null,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                nmbrfrmt: "hh:mm",
                formula: "TIME(" + hour + "," + min + ",0)",
            });
        }
        else {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: "",
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
        }
        if (totalExtraWork > 0) {
            let hour = Math.floor(totalExtraWork / 60)
                .toString()
                .padStart(2, "0");
            let min = (totalExtraWork % 60).toString().padStart(2, "0");
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: null,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                nmbrfrmt: "hh:mm",
                formula: "TIME(" + hour + "," + min + ",0)",
            });
        }
        else {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: "",
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
        }
        for (const t of totals) {
            let permissionType = this._permissionTypes.find((p) => p.id === t.permissionId);
            if (permissionType && permissionType.isDailyScheduled !== null && permissionType.isDailyScheduled) {
                let prmsIndex = this._permissionTypesColIndex.find((p) => p.id === t.permissionId).index;
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: prmsIndex,
                    value: String(t.total),
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                });
            }
            else {
                let prmsIndex = this._permissionTypesColIndex.find((p) => p.id === t.permissionId).index;
                let hour = Math.floor(t.total / 60)
                    .toString()
                    .padStart(2, "0");
                let min = (t.total % 60).toString().padStart(2, "0");
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: prmsIndex,
                    value: null,
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                    nmbrfrmt: "hh:mm",
                    formula: "TIME(" + hour + "," + min + ",0)",
                });
            }
        }
        this._generalRowIndex++;
        return {
            totals: totals,
            totalExtraWork: totalExtraWork,
            totalMissingWork: totalMissingWork,
            workeddays: userData.workeddays.reduce((a, b) => a + b, 0),
            totaldays: userData.totaldays.reduce((a, b) => a + b, 0),
        };
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        if (params.formula) {
            c.value = {
                formula: params.formula,
            };
        }
        else {
            c.value = params.value;
        }
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
        if (params.nmbrfrmt)
            c.numFmt = params.nmbrfrmt;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, "ELİMKO Aylık Puantaj Raporu", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE_REPORT_BELONGS_TO", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)(this._options.startDate).locale(this._locale).format("LL") + " - " + (0, moment_1.default)(this._options.endDate).locale(this._locale).format("LL"), border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, filter.join("\r\n"), border);
        let filterlength = filter.length;
        let row = ws.getRow(startRowIndex + 2);
        row.height = filterlength * 20;
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.ElimkoMonthlyPermissionCustomReport = ElimkoMonthlyPermissionCustomReport;
async function generateReport(request, locale) {
    let report = new ElimkoMonthlyPermissionCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
