"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ElimkoDailyPermissionCustomReport = void 0;
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const excel_report_refactory_1 = require("../../../business/report/excel.report.refactory");
const luxon_1 = require("luxon");
class ElimkoDailyPermissionCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._permissionTypesColIndex = [];
        this._request = request;
        this._filter = request.filter;
        this._options = {
            startDate: new Date(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
            endDate: new Date(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroupIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._activeUsers = [];
        this._reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
    }
    async generateReport() {
        this._permissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(this._request.organizationId, {
            take: 1000,
            skip: 0,
        })).items;
        const sheetInfo = [
            {
                sheetName: "Günlük Puantaj",
                data: [],
                headers: this.getHeaders(1),
            },
        ];
        const reportInfoSheetData = {
            reportNameValue: "Günlük Puantaj",
            filters: {
                orgUnitIds: this._options.organizationUnitIds,
                userGroupIds: this._options.userGroupIds,
                dateTimeRange: {
                    startDateTime: this._options.startDate,
                    endDateTime: this._options.endDate,
                },
            },
        };
        await this._reportFactory.initialize({
            requesterUserId: this._request.requesterUserId,
            userIds: [],
            reportInfoSheetData,
            sheetInfo: sheetInfo,
            profileColumnsIndex: 2,
        });
        const organizationUnitsFilterForGettingIdentity = this._options.organizationUnitIds?.map((elem) => {
            return {
                id: elem,
                includeDescendants: false,
            };
        });
        this._activeUsers = (await dal_manager_1.dbManager.accessIdentity.searchIdentity({
            organizationId: this._request.organizationId,
            take: null,
            skip: null,
            status: app_enums_1.enums.IdentityStatusType.Enabled,
            profileFilter: null,
            organizationUnits: organizationUnitsFilterForGettingIdentity,
            userGroupIds: this._options.userGroupIds,
            credentialExtensionFilters: [],
            hasMail: null,
            permissions: [],
            workplans: [],
        })).items;
        if (this._activeUsers.length > 0) {
            await dal_manager_1.dbManager.accessPacs2.getEmployeeDaysFiltered(this._request.organizationId, this._request.requesterUserId, {
                startDate: this._options.startDate,
                endDate: this._options.endDate,
                userIds: this._activeUsers.map((user) => user.id),
                sortType: 1,
                sortOrder: "ASC",
                take: null,
                skip: null,
            }, this.onDataCallbackFunctionReader.bind(this));
        }
        else {
            app_logs_1.logger.debug("[Custom Report] No available data !");
        }
        const fileObj = await this._reportFactory.finalize(reportInfoSheetData);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileObj.fileId,
            filePath: fileObj.filePath,
            notificationType: undefined,
            customReportName: "Günlük Puantaj Raporu",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async onDataCallbackFunctionReader(rows) {
        app_logs_1.logger.debug("Fetch reader data %s rows", rows.length);
        const userIds = rows.map((item) => item.userId);
        await this._reportFactory.appendToUserInfo(userIds);
        const sheetInfo = [
            {
                sheetName: "Günlük Puantaj",
                data: [],
            },
        ];
        const formatDurationFormula = (minutes) => {
            if (minutes <= 0)
                return { value: "" };
            const hour = Math.floor(minutes / 60)
                .toString()
                .padStart(2, "0");
            const min = (minutes % 60).toString().padStart(2, "0");
            return {
                value: null,
                numFmt: "hh:mm",
                formula: `TIME(${hour},${min},0)`,
            };
        };
        for (const row of rows) {
            const activeUser = this._activeUsers.find((item) => item.id === row.userId);
            let cells = [
                {
                    value: activeUser.userGroups.map((item) => item.name).join("\n") ?? "",
                    alignment: {
                        wrapText: true,
                    },
                },
                {
                    value: activeUser.organizationUnits.map((item) => item.name).join("\n") ?? "",
                    alignment: {
                        wrapText: true,
                    },
                },
                {
                    value: luxon_1.DateTime.fromJSDate(row.date).setLocale(this._options.locale).toLocaleString(luxon_1.DateTime.DATE_SHORT),
                },
                formatDurationFormula(row.data.s.m),
                formatDurationFormula(row.data.s.e),
            ];
            for (const permType of this._permissionTypesColIndex) {
                const permissions = row.data.p.filter((p) => p.pt === permType.id);
                if (permissions.length === 0) {
                    cells.push({ value: "" });
                    continue;
                }
                if (permType.isDailyScheduled) {
                    cells.push({ value: 1 });
                }
                else {
                    const totalMinutes = permissions.reduce((sum, p) => sum + p.u, 0);
                    cells.push(formatDurationFormula(totalMinutes));
                }
            }
            sheetInfo[0].data.push({
                userId: row.userId,
                cells,
            });
        }
        this._reportFactory.appendData(sheetInfo);
        return Promise.resolve();
    }
    getHeaders(startColIndex) {
        let headers = [
            {
                phrase: "EXCEL-REPORT.ACCESS_LOGS_USER_GROUP",
                width: 25,
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_LOGS_ORG_UNIT",
                width: 25,
            },
            {
                phrase: "EXCEL-REPORT.DATE",
                width: 12,
            },
            {
                phrase: "PDF-REPORT.less_working_with_permission",
                width: 15,
            },
            {
                phrase: "EXCEL-REPORT.OVERTIME",
                width: 15,
            },
        ];
        for (const permissionType of this._permissionTypes) {
            this._permissionTypesColIndex.push({
                id: permissionType.id,
                index: startColIndex + headers.length,
                isDailyScheduled: permissionType.isDailyScheduled,
            });
            headers.push({ phrase: permissionType.name, width: 15 });
        }
        return headers;
    }
}
exports.ElimkoDailyPermissionCustomReport = ElimkoDailyPermissionCustomReport;
async function generateReport(request, locale) {
    let report = new ElimkoDailyPermissionCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
