"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateHook = void 0;
const business_hooks_1 = require("../../../business/business.hooks");
const dal_manager_1 = require("../../../dal/dal.manager");
const moment_1 = __importDefault(require("moment"));
const app_logs_1 = require("../../../app.logs");
const luxon_1 = require("luxon");
const app_util_1 = require("../../../app.util");
const taskqueue_1 = require("../../../utils/taskqueue");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const dal_access_psql_organization_1 = require("../../../dal/access/psql/dal.access.psql.organization");
function CreateHook() {
    return new PelsanTekstilHook();
}
exports.CreateHook = CreateHook;
const pelsanOrganizationId = "1325e507-3b10-4047-b439-cf50d6fe1358";
const errorEmailSecretId = "caf366bb-00b2-47fa-975c-684525736212";
const pelsanMemotiveSecretId = "6f04d041-55d7-4d65-a708-b650c2993c1b";
const reverseSyncScheduledJobId = "9cb8ba15-245f-4b04-a9c0-abec6405fe10";
const oneTimeSyncScheduledJobId = "ac58b649-1035-43f7-9bc4-f3ca9d3a09ea";
const pelsanWomenGroupId = "33f54665-a499-4daf-91e1-74b16a68376e";
const pelsanMenGroupId = "65e1922e-1fb6-4d87-ab4a-969a69aacdc2";
const pelsanWhiteCollarGroupId = "82e22a56-f815-4b9f-ae99-5e0bac5f044f";
const pelsanBlueCollarGroupId = "ea48a16a-8877-4d14-a326-64dfe1a68dd5";
const pelsanSubcontractorGroupId = "1286ae2f-8b24-4758-b0b3-7b7a6081215e";
const telasisWomenGroupId = "d5942987-f085-4c09-9929-fef92253b366";
const telasisMenGroupId = "a11cb6cf-5817-4250-b038-6e469d38ea3d";
const telasisWhiteCollarGroupId = "32151769-fc8f-4017-8040-f0022afa8bd6";
const telasisBlueCollarGroupId = "019a14ae-58e6-4bdc-a769-75de76aee235";
const telasisSubcontractorGroupId = "8e4699be-7ba0-4dd9-865f-1f4005ccc892";
var MemotiveGender;
(function (MemotiveGender) {
    MemotiveGender[MemotiveGender["Kad\u0131n"] = 0] = "Kad\u0131n";
    MemotiveGender[MemotiveGender["Erkek"] = 1] = "Erkek";
    MemotiveGender[MemotiveGender["Di\u011Fer"] = 2] = "Di\u011Fer";
})(MemotiveGender || (MemotiveGender = {}));
var MemotiveWorkerType;
(function (MemotiveWorkerType) {
    MemotiveWorkerType[MemotiveWorkerType["Unknown"] = 0] = "Unknown";
    MemotiveWorkerType[MemotiveWorkerType["BeyazYaka"] = 1] = "BeyazYaka";
    MemotiveWorkerType[MemotiveWorkerType["MaviYaka"] = 2] = "MaviYaka";
    MemotiveWorkerType[MemotiveWorkerType["Taseron"] = 3] = "Taseron";
})(MemotiveWorkerType || (MemotiveWorkerType = {}));
function extractGenderFromGroups(groupIds) {
    let gender = MemotiveGender.Diğer;
    const isMan = groupIds && (groupIds.includes(pelsanMenGroupId) || groupIds.includes(telasisMenGroupId));
    const isWoman = groupIds && (groupIds.includes(pelsanWomenGroupId) || groupIds.includes(telasisWomenGroupId));
    if (isMan && !isWoman) {
        gender = MemotiveGender.Erkek;
    }
    else if (!isMan && isWoman) {
        gender = MemotiveGender.Kadın;
    }
    return gender;
}
function extractWorkerTypeFromGroups(groupIds) {
    let workerType = MemotiveWorkerType.Unknown;
    const isWhiteCollar = groupIds && (groupIds.includes(pelsanWhiteCollarGroupId) || groupIds.includes(telasisWhiteCollarGroupId));
    const isBlueCollar = groupIds && (groupIds.includes(pelsanBlueCollarGroupId) || groupIds.includes(telasisBlueCollarGroupId));
    const isSubcontractor = groupIds && (groupIds.includes(pelsanSubcontractorGroupId) || groupIds.includes(telasisSubcontractorGroupId));
    if (isWhiteCollar && !isBlueCollar && !isSubcontractor) {
        workerType = MemotiveWorkerType.BeyazYaka;
    }
    else if (!isWhiteCollar && isBlueCollar && !isSubcontractor) {
        workerType = MemotiveWorkerType.MaviYaka;
    }
    else if (!isWhiteCollar && !isBlueCollar && isSubcontractor) {
        workerType = MemotiveWorkerType.Taseron;
    }
    return workerType;
}
function armonIdentityToMemotiveIdentity(armonIdentity) {
    const userGroupIds = armonIdentity.userGroups.map((g) => g.id);
    const memotiveIdentity = {
        firstname: armonIdentity.organizationProfile.name,
        lastname: armonIdentity.organizationProfile.surname,
        birthdate: undefined,
        registrationNumber: armonIdentity.organizationProfile.uniqueId,
        email: armonIdentity.organizationProfile.email,
        phone: armonIdentity.organizationProfile.phoneNumber,
        locationName: armonIdentity.organizationProfile.extensionFields.location ?? "Pelsan",
        departmentName: undefined,
        startWorkDate: undefined,
        isArchived: armonIdentity.organizationProfile.isDisabled,
        endWorkDate: undefined,
        gender: extractGenderFromGroups(userGroupIds),
        workerType: extractWorkerTypeFromGroups(userGroupIds),
        title: undefined,
        workArea: undefined,
    };
    if (armonIdentity.organizationProfile.extensionFields?.department) {
        memotiveIdentity.departmentName = armonIdentity.organizationProfile.extensionFields.department;
    }
    else {
        memotiveIdentity.departmentName = "diger";
    }
    if (armonIdentity.organizationProfile.extensionFields?.title) {
        memotiveIdentity.title = armonIdentity.organizationProfile.extensionFields.title;
    }
    if (armonIdentity.organizationProfile.extensionFields?.workArea) {
        memotiveIdentity.workArea = armonIdentity.organizationProfile.extensionFields.workArea;
    }
    if (armonIdentity.organizationProfile.birthDateUtc) {
        memotiveIdentity.birthdate = (0, moment_1.default)(armonIdentity.organizationProfile.birthDateUtc)?.format("YYYY-MM-DD");
    }
    if (armonIdentity.organizationProfile.employmentStartUtc) {
        memotiveIdentity.startWorkDate = (0, moment_1.default)(armonIdentity.organizationProfile.employmentStartUtc)?.format("YYYY-MM-DD");
    }
    if (armonIdentity.organizationProfile.employmentEndUtc) {
        memotiveIdentity.endWorkDate = (0, moment_1.default)(armonIdentity.organizationProfile.employmentEndUtc)?.format("YYYY-MM-DD");
    }
    return memotiveIdentity;
}
async function userIdToMemotiveIdentity(userId) {
    const dbResult = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
        return (await trx.query(`SELECT 
					uop.name,
					uop.surname,
					uop."birthDateUtc",
					uop."uniqueId",
					uop.email,
					uop."phoneNumber",
					uop."extensionFields"->'department' as "departmentName",
					uop."extensionFields"->'title' as "title",
					uop."extensionFields"->'workArea' as "workArea",
					uop."extensionFields"->'location' as "location",
					uop."employmentStartUtc",
					uop."employmentEndUtc",
					uo."isDisabled",
					(
						SELECT array_agg(ug.id)
						FROM "1325e507-3b10-4047-b439-cf50d6fe1358"."userGroups" ug
						INNER JOIN "1325e507-3b10-4047-b439-cf50d6fe1358"."userGroupUserOrganizations" uguo
						ON uguo."deletedAt" IS NULL AND ug."deletedAt" IS NULL AND uguo."userOrganizationId" = uo.id AND uguo."userGroupId" = ug.id
					) as "userGroupIds"
				FROM "1325e507-3b10-4047-b439-cf50d6fe1358"."userOrganizationProfiles" uop
				INNER JOIN "1325e507-3b10-4047-b439-cf50d6fe1358"."userOrganizations" uo
					ON uo.id = uop."userOrganizationId"
				WHERE uop."userId" = $1`, [userId])).rows[0];
    });
    const memotiveIdentity = {
        firstname: dbResult.name,
        lastname: dbResult.surname,
        birthdate: undefined,
        registrationNumber: dbResult.uniqueId,
        email: dbResult.email,
        phone: dbResult.phoneNumber,
        locationName: dbResult.location ?? "Pelsan",
        departmentName: dbResult.departmentName ?? "diger",
        startWorkDate: undefined,
        isArchived: dbResult.isDisabled,
        endWorkDate: undefined,
        gender: extractGenderFromGroups(dbResult.userGroupIds),
        workerType: extractWorkerTypeFromGroups(dbResult.userGroupIds),
        title: dbResult.title ?? undefined,
        workArea: dbResult.workArea ?? undefined,
    };
    if (dbResult.birthDateUtc) {
        memotiveIdentity.birthdate = (0, moment_1.default)(dbResult.birthDateUtc)?.format("YYYY-MM-DD");
    }
    if (dbResult.employmentStartUtc) {
        memotiveIdentity.startWorkDate = (0, moment_1.default)(dbResult.employmentStartUtc)?.format("YYYY-MM-DD");
    }
    if (dbResult.employmentEndUtc) {
        memotiveIdentity.endWorkDate = (0, moment_1.default)(dbResult.employmentEndUtc)?.format("YYYY-MM-DD");
    }
    return memotiveIdentity;
}
async function getAccessToken() {
    const pelsanMemotiveSecret = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        return await (0, dal_access_psql_organization_1.getOrganizationSecretById)(pelsanOrganizationId, pelsanMemotiveSecretId, trx);
    });
    let accessToken = await dal_manager_1.dbManager.accessRedisCache.getValue(pelsanMemotiveSecret.tokenRedisKey);
    if (!accessToken) {
        try {
            const tokenResponse = await (0, app_util_1.sendHttpsPostRequest)({
                hostname: pelsanMemotiveSecret.hostname,
                path: pelsanMemotiveSecret.path,
                body: {
                    emailOrPhone: pelsanMemotiveSecret.emailOrPhone,
                    password: pelsanMemotiveSecret.password,
                },
            });
            const secondsToExpire = Math.round(tokenResponse.expiresIn - luxon_1.DateTime.local().toSeconds());
            accessToken = "Bearer " + tokenResponse.accessToken;
            await dal_manager_1.dbManager.accessRedisCache.setExpireValue(pelsanMemotiveSecret.tokenRedisKey, accessToken, secondsToExpire);
        }
        catch (error) {
            app_logs_1.logger.error("Error while getting memotive token: " + error);
        }
    }
    return accessToken;
}
async function sendMemotiveIdentity(identity) {
    const accessToken = await getAccessToken();
    const response = await (0, app_util_1.sendHttpsPostRequest)({
        hostname: "memotive.app",
        path: "/api/v1/users",
        headers: {
            Authorization: accessToken,
        },
        body: identity,
    });
    app_logs_1.logger.info("Memotive sync response: " + response.result);
}
const taskQueue = new taskqueue_1.TaskQueue(30000, "Pelsan Memotive");
let smtpSettings;
let emailRecipients;
function sendErrorMail(text) {
    if (smtpSettings?.enabled && emailRecipients?.length > 0) {
        emailRecipients.map((r) => {
            taskQueue.push(new taskqueue_1.SendEmailTask(smtpSettings, {
                from: smtpSettings.defaultFromAddress,
                to: r,
                subject: "Armon Memotive Integration Error",
                text: text,
            }));
        });
    }
}
class PelsanTekstilHook extends business_hooks_1.OrganizationHookModels.IArmonHook {
    constructor() {
        super();
        dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const orgExists = await trx.query(`SELECT * FROM public."organizationList" WHERE id = $1`, [pelsanOrganizationId]);
            if (orgExists.rowCount > 0) {
                const orgSettings = await trx.query(`SELECT settings FROM "${pelsanOrganizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
					WHERE id = $1`, [pelsanOrganizationId]);
                smtpSettings = orgSettings.rows[0]?.settings?.notification?.smtpSettings;
                emailRecipients = (await (0, dal_access_psql_organization_1.getOrganizationSecretById)(pelsanOrganizationId, errorEmailSecretId, trx))?.recipients || [];
            }
        });
        this.afterIdentityUpsert = async function (oldIdentity, newIdentity) {
            try {
                const newMemotiveIdentity = armonIdentityToMemotiveIdentity(newIdentity);
                if (!oldIdentity) {
                    await sendMemotiveIdentity(newMemotiveIdentity);
                    return true;
                }
                else {
                    const oldMemotiveIdentity = armonIdentityToMemotiveIdentity(oldIdentity);
                    if (oldMemotiveIdentity.birthdate !== newMemotiveIdentity.birthdate ||
                        oldMemotiveIdentity.departmentName !== newMemotiveIdentity.departmentName ||
                        oldMemotiveIdentity.email !== newMemotiveIdentity.email ||
                        oldMemotiveIdentity.endWorkDate !== newMemotiveIdentity.endWorkDate ||
                        oldMemotiveIdentity.firstname !== newMemotiveIdentity.firstname ||
                        oldMemotiveIdentity.gender !== newMemotiveIdentity.gender ||
                        oldMemotiveIdentity.isArchived !== newMemotiveIdentity.isArchived ||
                        oldMemotiveIdentity.lastname !== newMemotiveIdentity.lastname ||
                        oldMemotiveIdentity.phone !== newMemotiveIdentity.phone ||
                        oldMemotiveIdentity.registrationNumber !== newMemotiveIdentity.registrationNumber ||
                        oldMemotiveIdentity.startWorkDate !== newMemotiveIdentity.startWorkDate ||
                        oldMemotiveIdentity.workerType !== newMemotiveIdentity.workerType ||
                        oldMemotiveIdentity.title !== newMemotiveIdentity.title ||
                        oldMemotiveIdentity.workArea !== newMemotiveIdentity.workArea ||
                        oldMemotiveIdentity.locationName !== newMemotiveIdentity.locationName) {
                        await sendMemotiveIdentity(newMemotiveIdentity);
                    }
                    return true;
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error while syncing userId " + newIdentity.id + "to memotive: " + error);
                sendErrorMail("Error syncing user\nUser: " + newIdentity.organizationProfile.uniqueId + "\nDate: " + new Date().toLocaleString() + "\n" + error);
                return false;
            }
        }.bind(this);
        this.afterUserGroupSessionApply = async function (groupId, addedUsers, removedUsers) {
            if ([pelsanMenGroupId, pelsanWomenGroupId, pelsanWhiteCollarGroupId, pelsanBlueCollarGroupId, pelsanSubcontractorGroupId,
                telasisMenGroupId, telasisWomenGroupId, telasisWhiteCollarGroupId, telasisBlueCollarGroupId, telasisSubcontractorGroupId].includes(groupId)) {
                if (addedUsers) {
                    for (const userId of addedUsers) {
                        try {
                            await sendMemotiveIdentity(await userIdToMemotiveIdentity(userId));
                        }
                        catch (error) {
                            app_logs_1.logger.error("Error while syncing groupId: " + groupId + " and userId " + userId + "to memotive: " + error);
                            sendErrorMail("Error syncing group\nGroup:" + groupId + "\nUser: " + userId + "\nDate: " + new Date().toLocaleString() + "\n" + error);
                        }
                    }
                }
                if (removedUsers) {
                    for (const userId of removedUsers) {
                        try {
                            await sendMemotiveIdentity(await userIdToMemotiveIdentity(userId));
                        }
                        catch (error) {
                            app_logs_1.logger.error("Error while syncing groupId: " + groupId + " and userId " + userId + "to memotive: " + error);
                            sendErrorMail("Error syncing group\nGroup:" + groupId + "\nUser: " + userId + "\nDate: " + new Date().toLocaleString() + "\n" + error);
                        }
                    }
                }
            }
        }.bind(this);
        this.scheduledJobRoutine = async function scheduledJobHandler(organizationId, scheduledJobId) {
            try {
                if (scheduledJobId === reverseSyncScheduledJobId) {
                    app_logs_1.logger.info("Starting memotive reverse address sync");
                    const accessToken = await getAccessToken();
                    const addressChangesResponse = await (0, app_util_1.sendHttpsGetRequest)({
                        hostname: "memotive.app",
                        path: "/api/v1/users/changed-users",
                        headers: {
                            Authorization: accessToken,
                        },
                    });
                    if (addressChangesResponse && addressChangesResponse.length > 0) {
                        await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                            await trx.query(`UPDATE "1325e507-3b10-4047-b439-cf50d6fe1358"."userOrganizationProfiles" uop
								SET address = updated.address, "updatedAt" = now()
								FROM jsonb_to_recordset($1::jsonb) AS updated ("registrationNumber" text, "profileUpdateDateTime" text, "address" text)
								WHERE uop."uniqueId" = updated."registrationNumber" AND uop.address IS DISTINCT FROM updated.address`, [JSON.stringify(addressChangesResponse)]);
                        });
                    }
                    app_logs_1.logger.info("Completed memotive reverse address sync");
                }
                else if (scheduledJobId === oneTimeSyncScheduledJobId) {
                    app_logs_1.logger.info("Starting memotive user one time sync");
                    let userIds;
                    await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                        userIds = (await trx.query(`SELECT "id" FROM "1325e507-3b10-4047-b439-cf50d6fe1358"."users"`)).rows.map((r) => r.id);
                    });
                    app_logs_1.logger.info("Will try to sync " + userIds.length + " users");
                    for (const userId of userIds) {
                        try {
                            await sendMemotiveIdentity(await userIdToMemotiveIdentity(userId));
                        }
                        catch (error) {
                            app_logs_1.logger.error("Error while syncing userId " + userId + "to memotive: " + error);
                            sendErrorMail("Error syncing user\nUser: " + userId + "\nDate: " + new Date().toLocaleString() + "\n" + error);
                        }
                    }
                    app_logs_1.logger.info("Completed memotive user one time sync");
                }
                else {
                    app_logs_1.logger.info("Undefined scheduled job id for pelsan: " + scheduledJobId);
                }
            }
            catch (error) {
                app_logs_1.logger.error(`Error while Pelsan scheduled job ` + scheduledJobId + " - " + error);
                sendErrorMail("Error during sync scheduled job:" + scheduledJobId + "\nDate: " + new Date().toLocaleString() + "\n" + error);
            }
        }.bind(this);
    }
}
