"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.TKMarketEmployeeAttendanceControlCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const fs_1 = __importDefault(require("fs"));
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
var Constants;
(function (Constants) {
    Constants.HeadersStyling = {
        v1: {
            alignment: {
                horizontal: "center",
                vertical: "bottom",
                wrapText: true,
                textRotation: 90,
            },
            font: {
                bold: true,
                size: 18,
                name: "Calibri",
            },
            border: {
                right: {
                    style: "thick",
                    color: { argb: "FFFFFF" },
                },
                left: {
                    style: "thick",
                    color: { argb: "FFFFFF" },
                },
                bottom: {
                    style: "thick",
                    color: { argb: "FFFFFF" },
                },
                top: {
                    style: "thick",
                    color: { argb: "FFFFFF" },
                },
            },
            fill: {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "B4C6E7" },
            },
        },
        v2: {
            alignment: {
                horizontal: "center",
                vertical: "top",
                wrapText: true,
                textRotation: 90,
            },
            font: {
                bold: true,
                size: 10,
                name: "Calibri",
            },
            fill: {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "B4C6E7" },
            },
        },
        v3: {
            alignment: {
                horizontal: "center",
                vertical: "bottom",
                wrapText: true,
                textRotation: 90,
            },
            font: {
                bold: true,
                size: 11,
                name: "Calibri",
                color: { argb: "FFFFFF" },
            },
            fill: {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "70AD47" },
            },
        },
        v4: {
            alignment: {
                horizontal: "center",
                vertical: "top",
                wrapText: true,
                textRotation: 90,
            },
            font: {
                bold: true,
                size: 10,
                name: "Calibri",
                color: { argb: "FFFFFF" },
            },
            fill: {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "000000" },
            },
        },
        v5: {
            alignment: {
                horizontal: "center",
                vertical: "top",
                wrapText: true,
                textRotation: 90,
            },
            font: {
                bold: true,
                size: 10,
                name: "Calibri",
                color: { argb: "FFFFFF" },
            },
            fill: {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFC000" },
            },
        },
    };
    Constants.EmployeePermissionTypeFeatures = {
        "91e203e6-70e6-41fa-995a-1e0fb8c01564": {
            text: "MAR.019 - Sağlık Raporu(Ücretsiz)",
            code: "R",
            backgroundColor: "00B050",
            textColor: "FFFFFF",
        },
        "7c41f2d2-3ebc-425d-a0df-182f1ad8a561": {
            text: "MAR.FM.0002 - Fazla Mesai-Resmi Tatil (%100)",
            code: "T",
            backgroundColor: "F7CAAC",
            textColor: "C00000",
        },
        "2d730102-23bd-47d1-91b1-8f331b5b6b7b": {
            text: "Resmi Tatil",
            code: "T",
            backgroundColor: "F7CAAC",
            textColor: "C00000",
        },
        "90d46e7b-f46c-43ca-9274-83726d67b3ad": {
            text: "Rapor",
            code: "R",
            backgroundColor: "00B050",
            textColor: "FFFFFF",
        },
        "b959736b-c620-4dbe-ac6d-85979bb1a412": {
            text: "Devamsızlık",
            code: "Z",
            backgroundColor: "C00000",
            textColor: "000000",
        },
        "7c54477c-018a-4b13-a8c4-c89576c4feeb": {
            text: "Hafta Tatili",
            code: "H",
            backgroundColor: "FFC7CE",
            textColor: "9C0006",
        },
        "01a2a47a-b4e3-43d3-a406-d2ccd4e2212b": {
            text: "MAR.013 - Ücretsiz İzin",
            code: "Ü",
            backgroundColor: "FFEB9C",
            textColor: "9C5700",
        },
        "2994e472-b917-4b00-9b39-d384a707683c": {
            text: "MAR.001 - Yıllık İzin",
            code: "Y",
            backgroundColor: "FF0000",
            textColor: "FFFFFF",
        },
        "2d48787e-dd7e-4a9b-abd6-fcdcad82a6f6": {
            text: "MAR.004 - Vefat İzni",
            code: "V",
            backgroundColor: "BFBFBF",
            textColor: "9C0006",
        },
        "531d692e-2c5b-4808-9cef-64a7a2a6bfa1": {
            text: "MAR.005 - Babalık İzni",
            code: "B",
            backgroundColor: "BFBFBF",
            textColor: "C00000",
        },
        "c940b5a4-929e-4561-a98e-b40eab688ebb": {
            text: "MAR.003 - Evlilik İzni",
            code: "E",
            backgroundColor: "BFBFBF",
            textColor: "9C0006",
        },
        "2584fd56-3fff-451c-88c9-edd9f212f7e6": {
            text: "MAR.012 - Sağlık Raporu İzni (İş Kazası)",
            backgroundColor: "00B050",
            code: "R",
            textColor: "FFFFFF",
        },
        "a86e3654-b1ba-409e-890e-e64153fc5fb1": {
            text: "MAR.015 - Çalışma Karşılığı İzin",
            code: "K",
            backgroundColor: "FFC000",
            textColor: "000000",
        },
        "f985d5df-c274-42f5-b92c-3788e2557b45": {
            text: "Haftalık Tatili",
            code: "H",
            backgroundColor: "FFC7CE",
            textColor: "9C0006",
        },
        "40d27da1-db93-4ef9-ae1b-4e03f2353fba": {
            text: "Kısmi İstihdam (JHR)",
            code: "KI",
            backgroundColor: "D8D8D8",
            textColor: "000000",
        },
    };
    Constants.WebPuantajInfo = {
        1: {
            text: "ÇALIŞTI",
            code: "Ç",
            backgroundColor: "D8D8D8",
            textColor: "000000",
        },
        2: {
            text: "HAFTALIK İZİN",
            code: "H",
            backgroundColor: "FFC7CE",
            textColor: "9C0006",
        },
        3: {
            text: "YILLIK İZİN",
            code: "Y",
            backgroundColor: "FF0000",
            textColor: "FFFFFF",
        },
        4: {
            text: "DENKLEŞTİRME",
            code: "K",
            backgroundColor: "FFC000",
            textColor: "000000",
        },
        5: {
            text: "GÖREVLİ",
            code: "G",
            backgroundColor: "2E75B5",
            textColor: "9C0211",
        },
        6: {
            text: "RAPORLU",
            code: "R",
            backgroundColor: "00B050",
            textColor: "FFFFFF",
        },
        7: {
            text: "ÜCRETSİZ İZİN",
            code: "Ü",
            backgroundColor: "FFEB9C",
            textColor: "9C5700",
        },
        8: {
            text: "NAKİL",
            code: "N",
            backgroundColor: "F7CAAC",
            textColor: "9C0006",
        },
        9: {
            text: "AYRILDI",
            code: "A",
            backgroundColor: "000000",
            textColor: "FFFFFF",
        },
        10: {
            text: "DOĞUM İZNİ",
            code: "D",
            backgroundColor: "E2EFD9",
            textColor: "9C0006",
        },
        11: {
            text: "EVLİLİK İZNİ",
            code: "E",
            backgroundColor: "BFBFBF",
            textColor: "9C0006",
        },
        12: {
            text: "BABALIK İZNİ",
            code: "B",
            backgroundColor: "BFBFBF",
            textColor: "C00000",
        },
        13: {
            text: "VEFAT İZNİ",
            code: "V",
            backgroundColor: "BFBFBF",
            textColor: "9C0006",
        },
        14: {
            text: "DEVAMSIZLIK",
            code: "Z",
            backgroundColor: "C00000",
            textColor: "000000",
        },
        15: {
            text: "RESMİ TATİL",
            code: "T",
            backgroundColor: "F7CAAC",
            textColor: "C00000",
        },
        16: {
            text: "KISMİ İSTİHDAM",
            code: "KI",
            backgroundColor: "D8D8D8",
            textColor: "000000",
        },
    };
})(Constants || (Constants = {}));
class TKMarketEmployeeAttendanceControlCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._request = request;
        this._options = request.filter;
        this._filter = {
            dateRange: {
                startDateTime: new Date(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
                endDateTime: new Date(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            },
            userFilter: {
                organizationUnitIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                userGroupIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                status: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
        this._isGrey = false;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating TK Market employee working duration excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet("Sayfa1", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsStatusCode = this._wb.addWorksheet("Durum Kodu", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 1;
        this._wsStatusCode.properties.defaultRowHeight = 15;
        this._data = await this.getEmployeeWorkingDurationData(this._filter);
        this.initGeneralSheet(this._wsGeneral);
        this.fillStatusCodeSheet(this._wsStatusCode);
        if (this._data?.length > 0) {
            for (const item of this._data) {
                this.addGeneralRow(item);
            }
        }
        this._wsGeneral.columns.forEach((column) => {
            let maxLength = 0;
            let isFirstCell = true;
            column.eachCell((cell) => {
                if (isFirstCell) {
                    isFirstCell = false;
                    maxLength = maxLength;
                }
                else {
                    const cellValue = cell.value ? cell.value.toString() : "";
                    maxLength = Math.max(maxLength, cellValue.length);
                }
            });
            column.width = maxLength + 2;
        });
        this._wsStatusCode.columns.forEach((column) => {
            let maxLength = 0;
            let isFirstCell = true;
            column.eachCell((cell) => {
                if (isFirstCell) {
                    isFirstCell = false;
                    maxLength = maxLength;
                }
                else {
                    const cellValue = cell.value ? cell.value.toString() : "";
                    maxLength = Math.max(maxLength, cellValue.length);
                }
            });
            column.width = maxLength + 2;
        });
        this._wsGeneral.pageSetup.orientation = "portrait";
        this._wsStatusCode.pageSetup.orientation = "portrait";
        this._wsGeneral.commit();
        this._wsStatusCode.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Personel Çalışma Süresi Raporu",
        };
    }
    async getEmployeeWorkingDurationData(filter) {
        let rows = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: filter.organizationId,
                requesterUserId: filter.requesterUserId,
                idBasedUserFilter: {
                    userIds: filter.userFilter.userIds,
                    userGroupIds: filter.userFilter.userGroupIds,
                    organizationUnitIds: filter.userFilter.organizationUnitIds,
                    userOrganizationStatus: filter.userFilter.status,
                    applyOrganizationUnitFilterHierarchically: filter.userFilter.organizationUnitHierachially,
                },
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                bindingKeys: [],
            });
            let queryParams = [...userFilter.bindingKeys, (0, moment_1.default)(filter.dateRange.startDateTime).format("YYYY-MM-DD"), (0, moment_1.default)(filter.dateRange.endDateTime).format("YYYY-MM-DD")];
            let query = `
				WITH latest_access_logs AS (
					SELECT 
						log,
						(log->>'o')::uuid as "userId",
						"actionUtc",
						ROW_NUMBER() OVER (PARTITION BY (log->>'o')::uuid, date_trunc('day', "actionUtc") ORDER BY "actionUtc" ASC) as rn
					FROM "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
				)
				SELECT 
					ed."userId",
					uop."uniqueId",
					uop."name",
					uop."surname",
					u.username,
					json_agg(json_build_object(
						'expectedWork', (ed.data->'s'->>'ew')::integer,
						'normalWork', (ed.data->'s'->>'n')::integer,
						'missingWork', (ed.data->'s'->>'m')::integer,
						'extraWork', (ed.data->'s'->>'e')::integer,
						'date', ed."date",
						'permissions', (ed.data->'p'), 
						'holidays', (ed.data->'h'), 
						'accessControlPointName', acp."name",
						'routineType', 
							CASE
								WHEN jsonb_array_length(ed.data->'w') = 1 THEN (vw.details->>'routineType')::integer
								ELSE NULL
							END
					) ORDER BY ed."date" DESC) AS items,
					json_agg(DISTINCT acp."name") FILTER (WHERE acp."name" IS NOT NULL) AS accessControlPointNames
				FROM "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				INNER JOIN "${filter.organizationId}"."userOrganizationProfiles" AS uop
					ON uo.id = uop."userOrganizationId"
				LEFT JOIN "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" AS u
					ON u.id = uop."userId"
				INNER JOIN 
				(
					SELECT "userId", "data", "date"  FROM "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					UNION
					SELECT "userId", "data", "startAt"::date as date  FROM "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
				) AS ed
				ON ed."userId" = uop."userId"
				INNER JOIN
					(${userFilter.query}) AS T3
					ON T3."userId" = ed."userId" AND ed."data" IS NOT NULL
				LEFT JOIN latest_access_logs AS al
					ON ed."userId" = al."userId"
					AND al.rn = 1
					AND al."actionUtc" BETWEEN ed."date"::timestamptz 
										AND (ed."date" + interval '1 day' - interval '1 second')::timestamptz
				LEFT JOIN "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp 
					ON acp.id = (al.log->>'a')::uuid
				LEFT JOIN LATERAL (
					SELECT *
					FROM "${filter.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange}" vw
					WHERE vw.range && tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 2})
					AND vw."workPlanId" = (ed.data->'w'->>0)::uuid
					ORDER BY LOWER(vw.range) ASC
					LIMIT 1
				) AS vw
				ON TRUE
				WHERE ed.date >= $${userFilter.bindingKeys.length + 1} AND ed.date <= $${userFilter.bindingKeys.length + 2}
				GROUP BY ed."userId", uop."uniqueId", uop."name", uop."surname", u.username
				ORDER BY uop."name", uop."surname";
			`;
            return (await trx.query(query, queryParams)).rows;
        });
        let currentDate = new Date(filter.dateRange.startDateTime);
        let endDate = new Date(filter.dateRange.endDateTime);
        rows.forEach((item) => {
            while (currentDate <= endDate) {
                const dateString = (0, moment_1.default)(currentDate).format("YYYY-MM-DD").toString();
                if (!item.items) {
                    item.items = [];
                    item.items.push({
                        date: dateString,
                        AccessControlPointName: null,
                        normalWork: 0,
                        extraWork: 0,
                        missingWork: 0,
                        expectedWork: 0,
                        permissions: [],
                        holidays: [],
                    });
                    currentDate.setDate(currentDate.getDate() + 1);
                    continue;
                }
                const index = item.items.map((e) => e.date.toString()).indexOf(dateString);
                if (index === -1) {
                    item.items.push({
                        date: dateString,
                        AccessControlPointName: null,
                        normalWork: 0,
                        extraWork: 0,
                        missingWork: 0,
                        expectedWork: 0,
                        permissions: [],
                        holidays: [],
                    });
                }
                currentDate.setDate(currentDate.getDate() + 1);
            }
            item.items.sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());
            currentDate = new Date(filter.dateRange.startDateTime);
        });
        return rows;
    }
    addInitGeneralCell(ws, row, col, value, key) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = Constants.HeadersStyling[key]?.alignment;
        c.font = Constants.HeadersStyling[key]?.font;
        c.border = Constants.HeadersStyling[key]?.border;
        c.fill = Constants.HeadersStyling[key]?.fill;
    }
    initGeneralSheet(wsRaw) {
        let colIndex = 1;
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Kullanıcı Adı", "v1");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Tekil ID", "v1");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Ad", "v1");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Soyad", "v1");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Erişim Kontrol Noktası", "v1");
        if (this._data?.length) {
            this._data[0]?.items.forEach((item) => {
                this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, item.date, "v3");
            });
        }
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Normal Gün     "Ç"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Hafta Tatili    "H"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Yıllık İzin     "Y"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Resmi Tatil      "T"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Ücretli İzin     "E,B,V"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Ücretsiz İzin      "Ü"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Rapor       "R"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Devamsızlık     "Z"', "v2");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, 'Kısmi İstihdam "KI"', "v4");
        this.addInitGeneralCell(wsRaw, this._generalRowIndex, colIndex++, "Toplam Puantaj Günü", "v5");
        wsRaw.getRow(this._generalRowIndex).height = 22 * 11;
        this._generalRowIndex++;
    }
    generateRowDataForStatusCodeSheet() {
        let rowData = [
            [
                {
                    value: "nr",
                    backgroundColorCode: "D8D8D8",
                    border: {
                        left: {
                            style: "thick",
                        },
                    },
                },
                {
                    value: "DURUM AÇIKLAMASI",
                    backgroundColorCode: "D8D8D8",
                    border: {},
                },
                {
                    value: "KODU",
                    backgroundColorCode: "D8D8D8",
                    border: {},
                },
                {
                    value: "PUANTAJ KODU",
                    backgroundColorCode: "D8D8D8",
                    border: {
                        right: {
                            style: "thick",
                        },
                    },
                },
                {
                    value: "Puantaj Kartı Karşılığı",
                    backgroundColorCode: "C5E0B3",
                    border: {},
                },
                {
                    value: "Eksik Gün Nedeni",
                    backgroundColorCode: "C5E0B3",
                    border: {
                        right: {
                            style: "thick",
                        },
                    },
                },
            ],
        ];
        const numberOfRows = 16;
        for (let i = 1; i < numberOfRows + 1; i++) {
            if (i === numberOfRows) {
                let data = [
                    {
                        value: i.toString(),
                        border: {
                            left: {
                                style: "thick",
                            },
                            bottom: {
                                style: "thick",
                            },
                        },
                    },
                    {
                        value: Constants.WebPuantajInfo[i].text,
                        border: {
                            bottom: {
                                style: "thick",
                            },
                        },
                    },
                    {
                        value: Constants.WebPuantajInfo[i].code,
                        border: {
                            bottom: {
                                style: "thick",
                            },
                        },
                    },
                    {
                        value: Constants.WebPuantajInfo[i].code,
                        backgroundColorCode: Constants.WebPuantajInfo[i].backgroundColor,
                        textColorCode: Constants.WebPuantajInfo[i].textColor,
                        alignment: "center",
                        border: {
                            right: {
                                style: "thick",
                            },
                            bottom: {
                                style: "thick",
                            },
                        },
                    },
                ];
                rowData[i] = data;
                break;
            }
            let data = [
                {
                    value: i.toString(),
                    border: {
                        left: {
                            style: "thick",
                        },
                    },
                },
                {
                    value: Constants.WebPuantajInfo[i].text,
                    border: {},
                },
                {
                    value: Constants.WebPuantajInfo[i].code,
                    border: {},
                },
                {
                    value: Constants.WebPuantajInfo[i].code,
                    backgroundColorCode: Constants.WebPuantajInfo[i].backgroundColor,
                    textColorCode: Constants.WebPuantajInfo[i].textColor,
                    alignment: "center",
                    border: {
                        right: {
                            style: "thick",
                        },
                    },
                },
            ];
            rowData[i] = data;
        }
        rowData[1] = [
            ...rowData[1],
            {
                value: "Normal Gün",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[2] = [
            ...rowData[2],
            {
                value: "Haftalık İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[3] = [
            ...rowData[3],
            {
                value: "Yıllık İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[4] = [
            ...rowData[4],
            {
                value: "Ücretli İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[5] = [
            ...rowData[5],
            {
                value: "Normal Gün",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[6] = [
            ...rowData[6],
            {
                value: "Ücretsiz Rapor",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "1 İstirahat",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[7] = [
            ...rowData[7],
            {
                value: "Ücretsiz İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "21 Diğer Ücretsiz izin",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[8] = [
            ...rowData[8],
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[9] = [
            ...rowData[9],
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[10] = [
            ...rowData[10],
            {
                value: "Ücretsiz İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "21 Diğer Ücretsiz izin",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[11] = [
            ...rowData[11],
            {
                value: "Ücretli İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[12] = [
            ...rowData[12],
            {
                value: "Ücretli İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[13] = [
            ...rowData[13],
            {
                value: "Ücretli İzin",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[14] = [
            ...rowData[14],
            {
                value: "Devamsızlık",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "15 Devamsızlık",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[15] = [
            ...rowData[15],
            {
                value: "Resmi Tatil",
                backgroundColorCode: "C5E0B3",
                border: {},
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                },
            },
        ];
        rowData[16] = [
            ...rowData[16],
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    bottom: {
                        style: "thick",
                    },
                },
            },
            {
                value: "",
                backgroundColorCode: "C5E0B3",
                border: {
                    right: {
                        style: "thick",
                    },
                    bottom: {
                        style: "thick",
                    },
                },
            },
        ];
        return rowData;
    }
    fillStatusCodeSheet(wsRaw) {
        let row = 2;
        let col = 2;
        wsRaw.mergeCells("B2:E2");
        wsRaw.mergeCells("F2:G2");
        this.setDataCell11({
            rowIndex: row,
            colIndex: col,
            value: "Web Puantaj Bilgileri",
            ws: wsRaw,
            border: {
                right: {
                    style: "thick",
                },
                left: {
                    style: "thick",
                },
                top: {
                    style: "thick",
                },
            },
            backgroundColorCode: "F7CAAC",
            alignment: "center",
        });
        col += 4;
        this.setDataCell11({
            rowIndex: row,
            colIndex: col,
            value: "Logo Puantaj Kartı Bilgileri",
            ws: wsRaw,
            border: {
                right: {
                    style: "thick",
                },
                left: {
                    style: "thick",
                },
                top: {
                    style: "thick",
                },
            },
            backgroundColorCode: "C5E0B3",
            alignment: "center",
        });
        row++;
        let rowData = this.generateRowDataForStatusCodeSheet();
        rowData.forEach((data) => {
            this.addCellsToRowWithOptions(data, 2, row++, wsRaw);
        });
    }
    addAccessControlPointNamesToEmptyOnes(record) {
        const accessControlPointNames = record.items.map((item) => item.accessControlPointName ?? null);
        record.items.forEach((item, index) => {
            if (!item.accessControlPointName) {
                if (index === 0) {
                    item.accessControlPointName = this.findNearestAccessControlPointName(accessControlPointNames, index);
                }
                else {
                    item.accessControlPointName = record.items[index - 1].accessControlPointName;
                }
            }
        });
    }
    findNearestAccessControlPointName(names, targetIndex) {
        let nearestIndex = 0;
        let smallestDifference = Number.MAX_SAFE_INTEGER;
        for (let i = 0; i < names.length; i++) {
            if (names[i] && Math.abs(i - targetIndex) < smallestDifference) {
                nearestIndex = i;
                smallestDifference = Math.abs(i - targetIndex);
            }
        }
        return names[nearestIndex];
    }
    getRowData(acp, record) {
        let rowData = [
            {
                value: record.username,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
            },
            {
                value: record.uniqueId,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
            },
            {
                value: record.name,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
            },
            {
                value: record.surname,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
            },
        ];
        let sumData = {
            Ç: 0,
            H: 0,
            Y: 0,
            T: 0,
            E: 0,
            B: 0,
            V: 0,
            Ü: 0,
            R: 0,
            Z: 0,
            K: 0,
            KI: 0,
        };
        rowData.push({
            value: acp ?? "",
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
            alignment: "center",
        });
        let currentDate = new Date(this._filter.dateRange.startDateTime);
        let endDate = new Date(this._filter.dateRange.endDateTime);
        while (currentDate <= endDate) {
            const dateString = (0, moment_1.default)(currentDate).format("YYYY-MM-DD").toString();
            const index = record.items.map((e) => e.date.toString()).indexOf(dateString);
            if (index === -1) {
                rowData.push({
                    value: "",
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                    alignment: "center",
                });
            }
            else if (record.items[index].accessControlPointName === acp) {
                const recordItem = record.items[index];
                let value = null;
                const isThereHoliday = record.items[index]?.holidays?.length > 0;
                if (isThereHoliday) {
                    value = "T";
                    if ("T" in sumData) {
                        sumData["T"]++;
                    }
                    rowData.push({
                        value: value,
                        border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                        backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                        alignment: "center",
                    });
                }
                else {
                    const hasNormalWork = recordItem.normalWork > 0;
                    const hasExpectedWork = recordItem.expectedWork > 0;
                    const hasPermissions = recordItem.permissions && recordItem.permissions.length > 0;
                    if (hasNormalWork) {
                        value = "Ç";
                        if ("Ç" in sumData) {
                            sumData["Ç"]++;
                        }
                    }
                    else if (!hasExpectedWork && !hasPermissions && recordItem.routineType === dal_constants_1.DalConstants.PacsWorkplanPeriodRoutineTypes.Week) {
                        value = "H";
                        if ("H" in sumData) {
                            sumData["H"]++;
                        }
                    }
                    if (recordItem.permissions?.[0]?.pt) {
                        const permission = Constants.EmployeePermissionTypeFeatures[recordItem.permissions[0].pt];
                        if (permission?.code) {
                            if (value) {
                                value = value + "-" + permission.code;
                            }
                            else {
                                value = permission.code;
                            }
                            if (permission.code in sumData) {
                                sumData[permission.code]++;
                            }
                        }
                        rowData.push({
                            value: value,
                            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                            textColorCode: permission?.textColor,
                            backgroundColorCode: permission?.backgroundColor,
                            alignment: "center",
                        });
                    }
                    else {
                        rowData.push({
                            value: value || "",
                            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                            backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                            alignment: "center",
                        });
                    }
                }
            }
            else {
                rowData.push({
                    value: "",
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                    alignment: "center",
                });
            }
            currentDate.setDate(currentDate.getDate() + 1);
        }
        rowData = [
            ...rowData,
            {
                value: sumData["Ç"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["H"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["Y"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["T"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["E"] + sumData["B"] + sumData["V"] + sumData["K"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["Ü"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["R"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["Z"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["KI"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
            {
                value: sumData["Ç"] + sumData["H"] + sumData["Y"] + sumData["T"] + sumData["E"] + sumData["B"] + sumData["V"],
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: this._isGrey ? "D8D8D8" : "FFFFFF",
                alignment: "center",
            },
        ];
        return rowData;
    }
    addGeneralRow(record) {
        let colIndex = 1;
        let rowData = [];
        if (!record.accesscontrolpointnames) {
            rowData = this.getRowData(null, record);
            colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
            this._isGrey = !this._isGrey;
            this._generalRowIndex++;
        }
        else {
            this.addAccessControlPointNamesToEmptyOnes(record);
            record.accesscontrolpointnames.forEach((acp) => {
                let colIndex = 1;
                rowData = this.getRowData(acp, record);
                colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
                this._generalRowIndex++;
            });
            this._isGrey = !this._isGrey;
        }
    }
}
exports.TKMarketEmployeeAttendanceControlCustomReport = TKMarketEmployeeAttendanceControlCustomReport;
async function generateReport(request, locale) {
    let report = new TKMarketEmployeeAttendanceControlCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
