"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EkmasMonthlyWorkingDetailCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const controller_report_1 = require("../../../api/pacs/v2/controllers/controller.report");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.AllThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thick",
            },
            bottom: {
                style: "thick",
            },
            top: {
                style: "thick",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    let report = new EkmasMonthlyWorkingDetailCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class EkmasMonthlyWorkingDetailCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._request = request;
        this._filter = request.filter;
        this._options = {
            dateRange: {
                startDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay)
                    .startOf("day")
                    .toDate(),
                endDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay)
                    .endOf("day")
                    .toDate(),
            },
            userFilter: {
                organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userGroupIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                workPlanIds: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
                userOrganizationStatus: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
    }
    async generateReport() {
        app_logs_1.logger.info("Generating custom report");
        if ((0, moment_1.default)(this._options.dateRange.endDateTime).isSame((0, moment_1.default)(), "day")) {
            await (0, controller_report_1.refreshUsersSummaryForToday)({
                requesterUserId: this._options.requesterUserId,
                applyOrganizationUnitFilterHierarchically: this._options.userFilter.organizationUnitHierachially,
                userGroupIds: this._options.userFilter.userGroupIds,
                organizationId: this._options.organizationId,
                organizationUnitIds: this._options.userFilter.organizationUnitIds,
                userIds: this._options.userFilter.userIds,
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
        }
        const dbResult = (await dal_manager_1.dbManager.accessPacs2.getFirstInLastOutSummaryForUsers(this._options.organizationId, this._options.requesterUserId, {
            startDate: this._options.dateRange.startDateTime,
            endDate: this._options.dateRange.endDateTime,
            organizationUnitIds: this._options.userFilter.organizationUnitIds,
            organizationUnitHierarchically: this._options.userFilter.organizationUnitHierachially,
            userGroupIds: this._options.userFilter.userGroupIds,
            userIds: this._options.userFilter.userIds,
            workPlanIds: this._options.userFilter.workPlanIds,
            status: this._options.userFilter.userOrganizationStatus,
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
            workStatusFilter: [],
            take: null,
            skip: null,
        })).items;
        this._permissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(this._options.organizationId, {
            take: 1000,
            skip: 0,
        })).items;
        this._userIds = dbResult.map((u) => u.userId);
        this._userDaysDetails = [];
        this._maxExtraWorkDays = 0;
        for (const userId of this._userIds) {
            let permissionUsage = this._permissionTypes.map((elem) => {
                return {
                    id: elem.id,
                    name: elem.name,
                    count: 0,
                };
            });
            const item = {
                userId: userId,
                days: await dal_manager_1.dbManager.accessPacs2.workSummaryOfEmployeeInDateRange(this._options.requesterUserId, this._options.organizationId, {
                    dateRange: this._options.dateRange,
                    paginationRequest: { take: 365, skip: 0 },
                    userId: userId,
                    sortType: dal_constants_1.DalConstants.WorkSummarySortType.Date,
                    sortOrder: "ASC",
                }),
            };
            let extraWorkDurationHourFormat = 0;
            let weeklyHolidayCount = 0;
            let generalHolidayCount = 0;
            let normalWorkingDayCount = 0;
            let publicHolidayTotalWorkingHour = 0;
            let fullName = "";
            let registerNo = "";
            const workPlanNames = [];
            let missingWorkCounter = 0;
            item.days.items.forEach((elem) => {
                if (elem.extraWorkDuration > 0) {
                    extraWorkDurationHourFormat += elem.extraWorkDuration;
                }
                if (elem.expectedHolidayDuration > 0) {
                    generalHolidayCount += elem.expectedHolidayDuration / 60 / 24;
                    publicHolidayTotalWorkingHour += elem.expectedHolidayDuration - elem.usedHolidayDuration;
                }
                if (!elem.workplanDuration && !elem.expectedHolidayDuration && !elem.usedPermissions.length) {
                    weeklyHolidayCount++;
                }
                if (elem.expectedWorkDuration > 0) {
                    normalWorkingDayCount++;
                }
                if (elem.usedPermissions.length) {
                    elem.usedPermissions.forEach((permission) => (permissionUsage.find((item) => item.id === permission.typeId).count += 1));
                }
                if (elem.missingWorkDuration) {
                    missingWorkCounter += elem.missingWorkDuration / 60 > 3.75 ? 1 : 0.5;
                }
                fullName = elem.userCaption[0].text[0];
                registerNo = elem.userCaption[1].text[0];
            });
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.listUserOrganizationUnitNames(this._options.organizationId, item.days.items[0].userId);
            const userOrganizationProfiles = await dal_manager_1.dbManager.accessUser.getUserWorkPlanAndOrganizationProfilesInfo({
                organizationId: this._options.organizationId,
                userId: item.days.items[0].userId,
            });
            userOrganizationProfiles.forEach((elem) => {
                workPlanNames.push(elem.name);
            });
            const monthlyDetailOfUser = {
                registerNo: registerNo,
                fullName: fullName,
                workPlanName: workPlanNames.join("\n"),
                companyEntryDate: userOrganizationProfiles[0].employeeWorkingStartTime ? userOrganizationProfiles[0].employeeWorkingStartTime.toFormat("yyyy-MM-dd") : "",
                timeToLeaveTheCompany: userOrganizationProfiles[0].employeeWorkingEndTime ? userOrganizationProfiles[0].employeeWorkingEndTime.toFormat("yyyy-MM-dd") : "",
                normalWorkingDayCount: normalWorkingDayCount - missingWorkCounter,
                weeklyHolidayCount: weeklyHolidayCount,
                generalHolidayCount: generalHolidayCount,
                extraWorkDurationHourFormat: extraWorkDurationHourFormat,
                publicHolidayTotalWorkingHour: publicHolidayTotalWorkingHour,
                organizationUnitName: organizationUnitNames.join("\n"),
                permissionUsage: permissionUsage,
                totalMissingDays: missingWorkCounter,
                totalDays: item.days.items.length,
            };
            this._userDaysDetails.push(monthlyDetailOfUser);
            const count = item.days.items.filter((d) => d.extraWorkDuration).length;
            this._maxExtraWorkDays = count > this._maxExtraWorkDays ? count : this._maxExtraWorkDays;
        }
        this._filePath = await this.generate();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "EKMAS EMPLOYEE MONTHLY WORKING DETAIL PRIVATE REPORT",
        };
    }
    async generate() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                " AYLIK ÇALIŞMA DETAYI ÖZEL RAPORU " +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.dateRange.startDateTime).locale(this._locale).format("LL") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Requester user not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        for (const item of this._userDaysDetails) {
            this.addGeneralRow(item);
        }
        let filters = [];
        if (this._options.userFilter.userOrganizationStatus) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[this._options.userFilter.userOrganizationStatus].toUpperCase(), locale: this._locale }));
        }
        if (this._options.userFilter.organizationUnitIds && this._options.userFilter.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.userFilter.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._options.userFilter.organizationUnitHierachially) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userFilter.userGroupIds && this._options.userFilter.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userFilter.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(","));
        }
        if (this._options.userFilter.workPlanIds && this._options.userFilter.workPlanIds.length > 0) {
            let workplanNames = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._request.organizationId, this._options.userFilter.workPlanIds);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.items.map((elem) => elem.name).join(","));
        }
        if (this._options.userFilter.userIds && this._options.userFilter.userIds.length > 0) {
            let userNames = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(this._request.organizationId, this._options.userFilter.userIds);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.map((elem) => elem.fullname).join(","));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"), filters.length);
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.info("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Sicil No", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Adı Soyadı", Constants.Styling.AllThin);
        columns.push({ width: 35 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Organizasyon Birimi", Constants.Styling.AllThin);
        columns.push({ width: 25 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Çalışma Planı", Constants.Styling.AllThin);
        columns.push({ width: 25 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Giriş Tarihi", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Çıkış Tarihi", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex + 1, startColIndex - 1);
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Normal Çalışma", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Hafta Tatili", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Genel Tatil", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Saat", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Fazla Mesai", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Saat", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Resmi Tatil Mesaisi", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Devamsızlık", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this._permissionTypes.forEach((elem) => {
            this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
            columns.push({ width: 20 });
            this.addInitCell(wsRaw, startRowIndex, startColIndex++, elem.name, Constants.Styling.AllThin);
            columns.push({ width: 20 });
        });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex, "Gün", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Toplam", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        this._generalRowIndex++;
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(item) {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.registerNo,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.fullName,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.organizationUnitName,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.workPlanName,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.companyEntryDate,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.timeToLeaveTheCompany,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.normalWorkingDayCount,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.weeklyHolidayCount,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.generalHolidayCount,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (item.extraWorkDurationHourFormat % 60),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (item.publicHolidayTotalWorkingHour % 60),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.totalMissingDays,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        item.permissionUsage.forEach((elem) => {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: elem.count,
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: item.totalDays,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this._generalRowIndex++;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextRed)
            c.font = {
                color: { argb: "FF0000" },
                bold: true,
            };
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + "Aylık Çalışma Detayı Raporu", border, {
            bold: true,
        });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)(this._options.dateRange.startDateTime).locale(this._locale).format("LL") + " - " + (0, moment_1.default)(this._options.dateRange.endDateTime).locale(this._locale).format("LL"), border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, filter, border);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 15;
        }
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.EkmasMonthlyWorkingDetailCustomReport = EkmasMonthlyWorkingDetailCustomReport;
