"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.MarsMonthSummaryCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const luxon_1 = require("luxon");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_config_1 = require("../../../app.config");
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const generator_1 = require("../../../business/report/generator");
const report_util_1 = require("../../../business/report/report.util");
var Constants;
(function (Constants) {
    Constants.MarsAcps = {
        BrandiumAcps: [
            "d1ea9b8e-d97d-4da3-ab24-543d7587c6a9",
            "bb08125b-8310-4e2b-8710-5c3bfb4714a2",
        ],
        HQAcps: [
            "d93ea41e-69fb-4e75-a13b-cc8bd8293648",
            "38abdf7c-1ffd-450a-be58-00237fa497cf",
            "88f022b5-5151-4ab2-9d90-77f455fdaff3",
            "f9cfe25b-1703-46ab-8051-2bad3c6f924b",
            "2551f405-7fd7-4622-a278-e7d0888d5e38",
            "523bf569-d2a4-4f38-b24c-e77f9d753bc0",
        ],
    };
})(Constants || (Constants = {}));
class MarsMonthSummaryCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._options = request.filter;
        this._filter = {
            dateRange: {
                startDateTime: luxon_1.DateTime.fromISO(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value).startOf("month"),
                endDateTime: luxon_1.DateTime.fromISO(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value).endOf("month"),
            },
            userFilter: {
                organizationUnitIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                userGroupIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                status: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
                workPlanIds: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
        this._userUniqueIdsAndFullamesData = [];
        this._acpData = {};
        this._monthName = this._filter.dateRange.startDateTime.setLocale(locale).toLocaleString({ month: "long" });
    }
    async generateReport() {
        const dataSummary = await dal_manager_1.dbManager.accessPacs2.getFirstInLastOutSummaryForUsers(this._request.organizationId, this._request.requesterUserId, {
            startDate: this._filter.dateRange.startDateTime.toJSDate(),
            endDate: this._filter.dateRange.endDateTime.toJSDate(),
            organizationUnitIds: this._filter.userFilter?.organizationUnitIds,
            organizationUnitHierarchically: this._filter.userFilter?.organizationUnitHierachially,
            userGroupIds: this._filter.userFilter?.userGroupIds,
            userIds: this._filter.userFilter?.userIds,
            workPlanIds: this._filter.userFilter?.workPlanIds,
            status: this._filter.userFilter?.status,
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
            workStatusFilter: this._filter.userFilter?.workStatusFilter,
            take: null,
            skip: null,
        });
        this._itemsSummary = dataSummary.items;
        if (this._filter.userFilter?.workPlanIds) {
            this._workplans = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._request.organizationId, this._filter.userFilter.workPlanIds);
        }
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Ay Özet Raporu",
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsSummary = this._wb.addWorksheet("Ay Özet", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsSummary.properties.defaultRowHeight = 15;
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._summaryRowIndex = 5;
        this._generalRowIndex = 6;
        this._wsSummary.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " Ay Özet Raporu" +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                this._filter.dateRange.startDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL) +
                " - " +
                this._filter.dateRange.endDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL) +
                "\n";
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_FIRST_IN_LAST_OUT_SUMMARY", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                this._filter.dateRange.startDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL) +
                " - " +
                this._filter.dateRange.endDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL) +
                "\n";
        this._wsSummary.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initSummarySheet(this._wsSummary, 4, 1);
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        const filterBeginingRow = 5;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        await this.fillSheets(filterBeginingRow);
        this._wsSummary.autoFilter = {
            from: "A4",
            to: this._wsSummary.getColumn(this._wsSummary.columns.length).letter + this._summaryRowIndex,
        };
        let filters = [];
        if (this._filter.userFilter?.status) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[this._filter.userFilter?.status].toUpperCase(), locale: this._locale }));
        }
        if (this._filter.userFilter?.workStatusFilter) {
            this._filter.userFilter?.workStatusFilter.forEach((workStatusFilter) => {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORK_STATUS", locale: this._locale }) +
                    " : " +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.WorkStatusFilter[workStatusFilter].toUpperCase(), locale: this._locale }));
            });
        }
        if (this._filter.userFilter?.organizationUnitIds && this._filter.userFilter?.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._filter.userFilter?.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._filter.userFilter?.organizationUnitHierachially) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._filter.userFilter?.userGroupIds && this._filter.userFilter?.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._filter.userFilter?.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(", "));
        }
        if (this._filter.userFilter?.workPlanIds && this._filter.userFilter?.workPlanIds.length > 0) {
            let workplanNames = this._workplans.items.filter((wp) => this._filter.userFilter?.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.join(", "));
        }
        if (this._filter.userFilter?.userIds && this._filter.userFilter?.userIds.length > 0) {
            let userNames = this._userUniqueIdsAndFullamesData.filter((i) => this._filter.userFilter?.userIds.includes(i.userId)).map((i) => i.fullName);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.join(", "));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"), filters.length);
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    addInitCellForSummary(ws, row, col, value) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        c.fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "dce6f2" },
        };
    }
    initSummarySheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.name_surname",
                cellWidth: 30,
            },
            {
                phrase: "CUSTOM-REPORT.POSITION",
                cellWidth: 40,
            },
            {
                phrase: "CUSTOM-REPORT.DEPARTMENT",
                cellWidth: 30,
            },
            {
                value: this._monthName + " Ayı Çalışma Günü",
                cellWidth: 30,
            },
            {
                value: "HQ Durum",
                cellWidth: 20,
            },
            {
                value: "Brandium Durum",
                cellWidth: 20,
            },
            {
                value: "Toplam Durum",
                cellWidth: 20,
            },
            {
                value: "Toplam Çalışma Süresi",
                cellWidth: 30,
            },
        ];
        for (let i = 0; i < data.length; i++) {
            this.addInitCellForSummary(wsRaw, startRowIndex, startColIndex++, data[i].value || i18n_1.default.__({ phrase: data[i].phrase, locale: this._locale }));
            columns.push({ width: data[i].cellWidth || 30 });
        }
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.DATE",
                cellWidth: 35,
            },
            {
                phrase: "EXCEL-REPORT.DAY",
                cellWidth: 35,
            },
            {
                phrase: "EXCEL-REPORT.name_surname",
                cellWidth: 35,
            },
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.FIRST_START",
                cellWidth: 15,
            },
            {
                phrase: "PDF-REPORT.access_point",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LAST_LEAVE",
                cellWidth: 15,
            },
            {
                phrase: "PDF-REPORT.access_point",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.STATUS",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.USED_PERMISSION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.WORKPLAN",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.EXPECTED_ENTRANCE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.EXPECTED_EXIT",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.EXPECTED_WORK_DURATION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.NORMAL_WORK_DURATION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.MISSING_WORK",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.OVERTIME",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.PHYSICALLY_IN_REGION_DURATION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.USED_TOTAL_BREAK_DURATION",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    async fillSheets(filterBeginingRow) {
        const userIds = this._itemsSummary.map((i) => i.userId);
        const userOrganizationDetails = await this.getUsersOrganizationUnitDetails({ organizationId: this._request.organizationId, userIds });
        const filter = {
            dateRange: {
                startDateTime: this._filter.dateRange.startDateTime.toJSDate(),
                endDateTime: this._filter.dateRange.endDateTime.toJSDate(),
            },
            workStatusFilter: this._filter.userFilter?.workStatusFilter,
            userIds,
        };
        const userDatas = await dal_manager_1.dbManager.accessPacs2.getDailySummaryReportForMultipleUser(this._request.organizationId, filter);
        for (let item of this._itemsSummary) {
            const userData = userDatas.find((userData) => userData.userId === item.userId);
            await this.mapAccessLogsToAcpNames(userData.items);
            const userOrganizationDetail = userOrganizationDetails.find((i) => i.userId === item.userId);
            await this.addSummaryRow(userData, userOrganizationDetail);
            for (let dayInfo of userData.items) {
                this.addGeneralRow(dayInfo, userData.userFullnameAndUniqueIds[0]);
            }
            this._wsGeneral.autoFilter = {
                from: "A" + filterBeginingRow,
                to: this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex,
            };
            this._generalRowIndex += 3;
        }
        if (this._isDummyAccessLogAppeared) {
            this.addInitCell11(this._wsGeneral, this._generalRowIndex + 2, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.WARNING_FOR_ACCESS_LOGS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        }
    }
    async mapAccessLogsToAcpNames(items) {
        let accessLogIds = [];
        items.forEach((item) => {
            if (item.firstIn && item.firstIn.accessLogId) {
                accessLogIds.push(item.firstIn.accessLogId);
            }
            if (item.lastOut && item.lastOut.accessLogId) {
                accessLogIds.push(item.lastOut.accessLogId);
            }
        });
        this._acpData = await this.getAcpNamesForAccessLogs(accessLogIds);
    }
    async getAcpNamesForAccessLogs(accessLogIds) {
        const data = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const { rows } = await trx.query(`
			SELECT 
				al.id AS "accessLogId",
				acp.name AS "acpName"
			FROM 
				"${this._request.organizationId}"."access_logs" AS al
			JOIN 
				"${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
			ON 
				(acp.id = (al.log->>'a')::uuid)
			WHERE 
				al.id = ANY($1::uuid[])
				AND acp."organizationId" = $2
				AND acp."deletedAt" IS NULL
			`, [accessLogIds, this._request.organizationId]);
            return rows;
        });
        if (!data || data.length === 0) {
            return {};
        }
        const result = {};
        data.forEach((row) => {
            result[row.accessLogId] = row.acpName;
        });
        return result;
    }
    async addSummaryRow(userData, userOrganizationDetail) {
        const { userFullnameAndUniqueIds, summary, items } = userData;
        const userInfo = userFullnameAndUniqueIds && userFullnameAndUniqueIds.length > 0 ? userFullnameAndUniqueIds[0] : { uniqueId: "", fullName: "", userId: "" };
        const summaryRowData = await this.getSummaryRowData(items, userInfo.userId);
        const rowData = [
            {
                value: userInfo.uniqueId.toUpperCase(),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            { value: userInfo.fullName.toLocaleUpperCase("tr"), border: generator_1.ReportGenerator.Constants.Styling.AllDotted },
            { value: userOrganizationDetail?.userTitle || "", border: generator_1.ReportGenerator.Constants.Styling.AllDotted },
            { value: userOrganizationDetail?.userOrganizationUnitName || "", border: generator_1.ReportGenerator.Constants.Styling.AllDotted },
            { value: summary.workDaysCount, border: generator_1.ReportGenerator.Constants.Styling.AllDotted },
            ...summaryRowData,
        ];
        const colIndex = 1;
        this.addCellsToRowWithOptions(rowData, colIndex, this._summaryRowIndex, this._wsSummary);
        this._summaryRowIndex++;
    }
    addGeneralRow(row, UserFullNameAndUniqueId) {
        let colIndex = 1;
        let firstIn = "";
        let firstInAcp = "";
        let lastOut = "";
        let lastOutAcp = "";
        let isDummyAccessLogForFirstIn = false;
        if (row.firstIn && row.firstIn.actionUtc) {
            if (row.firstIn.accessLogId) {
                firstIn = (0, report_util_1.formatDateForExcel)(row.firstIn.actionUtc);
                firstInAcp = this._acpData[row.firstIn.accessLogId] || "";
            }
            else {
                isDummyAccessLogForFirstIn = true;
                this._isDummyAccessLogAppeared = true;
            }
        }
        let isDummyAccessLogForLastOut = false;
        if (row.lastOut && row.lastOut.actionUtc) {
            if (row.lastOut.accessLogId) {
                lastOut = (0, report_util_1.formatDateForExcel)(row.lastOut.actionUtc);
                lastOutAcp = this._acpData[row.lastOut.accessLogId] || "";
            }
            else {
                isDummyAccessLogForLastOut = true;
                this._isDummyAccessLogAppeared = true;
            }
        }
        let statusText;
        let permissionArr = [];
        let permissionText;
        if (row.firstIn) {
            statusText = i18n_1.default.__({ phrase: "EXCEL-REPORT.ARRIVED", locale: this._locale });
        }
        else if (row.missingWorkInMinutes > 0) {
            statusText = i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: this._locale });
        }
        else {
            statusText = "";
        }
        let colorText = undefined;
        let hasPermissionInDay = row.ppermissions && row.ppermissions.length > 0;
        let hasHolidayInDay = row.holidays && row.holidays.length > 0;
        if (hasPermissionInDay || hasHolidayInDay) {
            colorText = generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION;
            row.ppermissions.forEach((permission) => {
                switch (permission?.status) {
                    case app_enums_1.enums.PPermissionStatus.Approved: {
                        let interval = luxon_1.DateTime.fromJSDate(permission.startUtc).setLocale(this._locale).toFormat("dd/MM/yyyy HH:mm") +
                            " - " +
                            luxon_1.DateTime.fromJSDate(permission.endUtc).setLocale(this._locale).toFormat("dd/MM/yyyy HH:mm");
                        permissionArr.push(permission.ppermissionTypeName + "\n(" + interval + ")");
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Rejected: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED_INFO", locale: this._locale }));
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Waiting: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING_INFO", locale: this._locale }));
                        break;
                    }
                }
            });
            row.holidays.forEach((holiday) => {
                let interval = luxon_1.DateTime.fromJSDate(holiday.startUtc).setLocale(this._locale).toFormat("dd/MM/yyyy HH:mm") +
                    " - " +
                    luxon_1.DateTime.fromJSDate(holiday.endUtc).setLocale(this._locale).toFormat("dd/MM/yyyy HH:mm");
                permissionArr.push(holiday.holidayName + "\n(" + interval + ")");
            });
        }
        else {
            colorText = this.findColorCodeByPerformance(row);
        }
        permissionText = permissionArr.join("\n");
        const expectedEntrance = row.expectedWorkingRanges.map((val) => (0, report_util_1.formatDateToString)(val.start)).join("\n");
        const expectedLeave = row.expectedWorkingRanges.map((val) => (0, report_util_1.formatDateToString)(val.end)).join("\n");
        const automaticallyUsedFixedBreakDuration = row.automaticallyUsedFixedBreakDuration ?? 0;
        const usedFixedBreakDuration = row.usedFixedBreakDuration ?? 0;
        const fixedBreakDuration = automaticallyUsedFixedBreakDuration + usedFixedBreakDuration;
        const rowData = [
            {
                value: luxon_1.DateTime.fromJSDate(row.date).setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: luxon_1.DateTime.fromJSDate(row.date).setLocale(this._locale).weekdayLong,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: UserFullNameAndUniqueId.fullName.toUpperCase(),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: UserFullNameAndUniqueId.uniqueId,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: isDummyAccessLogForFirstIn ? "__:__" : firstIn,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: isDummyAccessLogForFirstIn ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForFirstIn ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForFirstIn ? "center" : "left",
            },
            {
                value: firstInAcp,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: isDummyAccessLogForLastOut ? "__:__" : lastOut,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                backgroundColorCode: isDummyAccessLogForLastOut ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForLastOut ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForLastOut ? "center" : "left",
            },
            {
                value: lastOutAcp,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: statusText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
                backgroundColorCode: colorText,
                isTextWhite: true,
            },
            {
                value: permissionText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
                backgroundColorCode: colorText,
                isTextWhite: true,
            },
            {
                value: row.workPlan.name ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            },
            {
                value: expectedEntrance ? expectedEntrance : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.workPlan.startOfWorkUtc ? generator_1.ReportGenerator.Constants.TimeFormat : null,
            },
            {
                value: expectedLeave ? expectedLeave : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.workPlan.endOfWorkUtc ? generator_1.ReportGenerator.Constants.TimeFormat : null,
            },
            {
                value: row.expectedWorkDuration ? (row.expectedWorkDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: row.expectedWorkDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.normalWorkDuration ? (row.normalWorkDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: row.normalWorkDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.missingWorkInMinutes
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.extraWorkInMinutes ? (row.extraWorkInMinutes / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.extraWorkInMinutes
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.physicallyInRegionDuration ? (row.physicallyInRegionDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.physicallyInRegionDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: fixedBreakDuration !== 0 ? (fixedBreakDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: fixedBreakDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
        ];
        colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._generalRowIndex++;
    }
    async getSummaryRowData(items, userId) {
        let normalWorkDuration = 0;
        let hqStatus = 0;
        let brandiumStatus = 0;
        let totalStatus = 0;
        let accessLogDataForSummary = [];
        await dal_manager_1.dbManager.accessLog.getAccessLogsNew(this._request.organizationId, {
            userIds: [userId],
            sortDateDesc: false,
            accessResult: dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Success,
            direction: dal_constants_1.DalConstants.AccessDirection.Entrance,
            startUtc: this._filter.dateRange.startDateTime.toJSDate(),
            endUtc: this._filter.dateRange.endDateTime.toJSDate(),
        }, async (rows) => {
            accessLogDataForSummary.push(...rows);
        });
        for (const item of items) {
            const startTime = luxon_1.DateTime.fromJSDate(item.date).startOf("day").toJSDate();
            const endTime = luxon_1.DateTime.fromJSDate(item.date).endOf("day").toJSDate();
            const filteredLogs = accessLogDataForSummary.filter((row) => {
                const actionDate = row.actionUtc;
                return actionDate >= startTime && actionDate <= endTime && row.log.o === userId;
            });
            const hasHQ = filteredLogs.some((row) => Constants.MarsAcps.HQAcps.includes(row.log.a));
            const hasBrandium = filteredLogs.some((row) => Constants.MarsAcps.BrandiumAcps.includes(row.log.a));
            if (hasHQ || hasBrandium) {
                totalStatus++;
                if (hasHQ)
                    hqStatus++;
                if (hasBrandium)
                    brandiumStatus++;
            }
            normalWorkDuration += item.normalWorkDuration;
        }
        const rowData = [
            {
                value: hqStatus,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: brandiumStatus,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: totalStatus,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: normalWorkDuration ? (normalWorkDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: normalWorkDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
        ];
        return rowData;
    }
    async getUsersOrganizationUnitDetails(params) {
        if (params.userIds.length === 0) {
            return [];
        }
        const data = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const { rows } = await trx.query(`SELECT uo."userId", 
                    uop."extensionFields"->>'title' as "title", 
                    ou."name" as "organizationUnitName"
            FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
                ON uo."id" = uoou."userOrganizationId" 
                AND uo."deletedAt" IS NULL 
                AND uoou."deletedAt" IS NULL
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
                ON ou."id" = uoou."organizationUnitId" 
                AND ou."deletedAt" IS NULL
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
                ON uo."userId" = uop."userId" 
                AND uop."deletedAt" IS NULL
            WHERE uo."userId" = ANY ($1::UUID[]);`, [params.userIds]);
            return rows;
        });
        return data.map((row) => ({
            userId: row.userId,
            userTitle: row.title ?? "",
            userOrganizationUnitName: row.organizationUnitName ?? "",
        }));
    }
    findColorCodeByPerformance(employeeData) {
        let colorCode = null;
        if (employeeData.firstIn && employeeData.firstIn.actionUtc) {
            if (employeeData.entrancePenaltyTimeInSeconds > 0) {
                let percentage = employeeData.entrancePenaltyTimeInSeconds / 60 / employeeData.entranceTolerance;
                if (percentage > 1) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED_2;
                }
                else if (percentage > 0.75) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_75_100;
                }
                else if (percentage > 0.5) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_50_75;
                }
                else if (percentage > 0.25) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_25_50;
                }
                else if (percentage > 0) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_0_25;
                }
            }
            else {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.ON_TIME_2;
            }
        }
        else {
            if (employeeData.expectedWorkingRanges.length > 0) {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED_2;
            }
        }
        return colorCode;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        this.addInitCell11(this._wsSummary, 1, 1, this._monthName + " Ayı", generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationBasic.name + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_FIRST_IN_LAST_OUT_SUMMARY", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, { bold: true });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._filter.dateRange.startDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL) +
            " - " +
            this._filter.dateRange.endDateTime.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 20;
        }
    }
}
exports.MarsMonthSummaryCustomReport = MarsMonthSummaryCustomReport;
async function generateReport(request, locale) {
    let report = new MarsMonthSummaryCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
