"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.KorfezTransportationFirstEntranceLastExitSummaryReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const report_util_1 = require("../../../business/report/report.util");
const enums_1 = require("../../../lib/access-model/v2/enums");
const luxon_1 = require("luxon");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.Exceed24HourTimeFormat = "[hh]:mm";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
    Constants.ColorCodes = {
        UNCOMING: "62A325",
        WORKING_TIME: "FFFF00",
        RED: "C80202",
    };
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    let report = new KorfezTransportationFirstEntranceLastExitSummaryReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class KorfezTransportationFirstEntranceLastExitSummaryReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._filter = request.filter;
        this._options = {
            dateRange: {
                startDateTime: luxon_1.DateTime.fromSQL(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
                endDateTime: luxon_1.DateTime.fromSQL(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
    }
    async generateReport() {
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "KORFEZ TRANSPORTATION FIRST IN LAST OUT SUMMARY REPORT",
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                "Çalışma Süresi Özel Raporu" +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.dateRange.startDateTime).locale(this._locale).format("LL") +
                " - " +
                (0, moment_1.default)(this._options.dateRange.endDateTime).locale(this._locale).format("LL") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        this.initGeneralSheet(this._wsGeneral, 5, 1, this._options.dateRange.startDateTime, this._options.dateRange.endDateTime);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        let userData = await dal_manager_1.dbManager.accessLog.getUsersAccessLogsActionTimesWithDirection({
            organizationId: this._request.organizationId,
        }, {
            startDate: this._options.dateRange.startDateTime,
            endDate: this._options.dateRange.endDateTime,
        });
        const userProfiles = await dal_manager_1.dbManager.accessUser.getUsersFullnamesAndUniqueIds(this._request.organizationId, userData.map((i) => i.userId));
        for (const data of userData) {
            const userProfile = userProfiles.find((i) => i.userId === data.userId);
            this.addGeneralRow(data, userProfile);
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: this._totalWorkingTimesStartingColumnIndex - 1,
            value: "TOPLAM",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        for (let i = 0; i < 2; i++) {
            this.cellForTotalTimesRowRanges(this._wsGeneral, 7, this._generalRowIndex, this._totalWorkingTimesStartingColumnIndex, Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
            this._totalWorkingTimesStartingColumnIndex++;
        }
        for (let i = 0; i < 2; i++) {
            this.cellForTotalTimesRowRanges(this._wsGeneral, 7, this._generalRowIndex, this._totalWorkingTimesStartingColumnIndex, Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
            this._totalWorkingTimesStartingColumnIndex++;
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex, startDate, endDate) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex, "", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 1);
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Sicil No", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Personel Ad Soyad", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        for (let i = 0; i <= Math.ceil(endDate.diff(startDate, ["days"]).days); i++) {
            wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
            this.addInitCell(wsRaw, startRowIndex, startColIndex, startDate.plus({ days: i }).toFormat("dd/MM/yyyy"), Constants.Styling.AllThin);
            columns.push({ width: 20 });
            this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Görev Alış", Constants.Styling.AllThin);
            columns.push({ width: 20 });
            this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Görev Bırakış", Constants.Styling.AllThin);
            columns.push({ width: 20 });
            this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Çalışma Süresi", Constants.Styling.AllThin);
            columns.push({ width: 20 });
            this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Fazla Çalışma Süresi", Constants.Styling.AllThin);
            columns.push({ width: 20 });
        }
        wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 7);
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, luxon_1.Interval.fromDateTimes(startDate, endDate).toFormat("dd/MM/yyyy"), Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex - 1, "", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this._totalWorkingTimesStartingColumnIndex = startColIndex;
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Toplam Çalışma Süresi", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Toplam Fazla Çalışma Süresi", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "Aylık Normal Çalışma Saati", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "225 Saat Üzeri Çalışma Süresi", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "11 SAATİ AŞAN VE ÖDENEN ÇALIŞMA SAATLERİ (26.5-31.5)", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "11 SAATİ AŞAN VE ÖDENEN ÇALIŞMA SAATLERİ (1.6 - 22.6)", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex + 1, startColIndex++, "11 SAAT ÜZERİ ÖDENENLER DÜŞÜLMÜŞ HALİ", Constants.Styling.AllThin, Constants.ColorCodes.WORKING_TIME);
        columns.push({ width: 20 });
        this._generalRowIndex++;
        wsRaw.columns = columns;
    }
    addInitCell(ws, row, col, value, border, backgroundColorCode) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
        if (backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: backgroundColorCode },
            };
        }
    }
    addGeneralRow(row, userProfile) {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userProfile.uniqueId,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userProfile.fullName,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        const dateLoop = luxon_1.Interval.fromDateTimes(this._options.dateRange.startDateTime, this._options.dateRange.endDateTime.plus({ days: 1 })).splitBy({ days: 1 });
        let index = 0;
        let totalWorkingTime = 0;
        let extraWorkingTime = 0;
        for (const date of dateLoop) {
            const pairs = [];
            for (index; index < row.actionTimeWithDirection.length; index++) {
                if (date.contains(luxon_1.DateTime.fromISO(row.actionTimeWithDirection[index].actionTime))) {
                    pairs.push(row.actionTimeWithDirection[index].actionTime);
                }
                else if (row.actionTimeWithDirection[index].direction === enums_1.EnumsV2.AccessDirection.Exit) {
                    pairs.push(row.actionTimeWithDirection[index].actionTime);
                }
                else {
                    break;
                }
            }
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: (pairs[0] ? (0, report_util_1.formatDateForExcel)(pairs[0]) : null),
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                numFmt: pairs[0] ? Constants.TimeFormat : null,
                backgroundColorCode: pairs[0] ? null : Constants.ColorCodes.UNCOMING,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: (pairs[pairs.length - 1] ? (0, report_util_1.formatDateForExcel)(pairs[pairs.length - 1]) : null),
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                numFmt: pairs[pairs.length - 1] ? Constants.TimeFormat : null,
                backgroundColorCode: pairs[pairs.length - 1] ? null : Constants.ColorCodes.UNCOMING,
            });
            const minutes = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(pairs[0]), luxon_1.DateTime.fromISO(pairs[pairs.length - 1])).toDuration("minutes").minutes;
            if (this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).value) {
                totalWorkingTime += minutes;
                extraWorkingTime += minutes - 11 * 60 > 0 ? minutes - 11 * 60 : 0;
            }
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                        '<>"",' +
                        "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                        "-" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 3).address +
                        '),""' +
                        ")",
                },
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                numFmt: pairs[pairs.length - 1] ? Constants.Exceed24HourTimeFormat : null,
                backgroundColorCode: pairs[pairs.length - 1] ? Constants.ColorCodes.WORKING_TIME : Constants.ColorCodes.UNCOMING,
                isTextRed: minutes >= 660 ? true : false,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(AND(" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                        '<>"",' +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                        ">TIME(11, 0, 0)),SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                        '-TIME(11, 0, 0)),"")',
                },
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                numFmt: minutes > 11 * 60 ? Constants.Exceed24HourTimeFormat : null,
            });
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userProfile.fullName,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (Math.floor((totalWorkingTime / 60) * 100) / 100),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (Math.floor((extraWorkingTime / 60) * 100) / 100),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (dateLoop.length * 7.5),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        const workingTimeHigherThen225Hour = Math.floor((extraWorkingTime / 60) * 100) / 100 + Math.floor((totalWorkingTime / 60) * 100) / 100 > 225
            ? Math.floor((extraWorkingTime / 60) * 100) / 100 + Math.floor((totalWorkingTime / 60) * 100) / 100 - 225
            : 0;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: workingTimeHigherThen225Hour,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            backgroundColorCode: Constants.ColorCodes.WORKING_TIME,
        });
        this._generalRowIndex++;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.font;
        c.value = params.value;
        c.alignment = {
            horizontal: params.alignment ? params.alignment : "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextRed)
            c.font = {
                color: { argb: Constants.ColorCodes.RED },
                bold: true,
            };
    }
    cellForTotalTimesRowRanges(ws, startingRowIndex, endingRowIndex, colIndex, border, backgroundColorCode) {
        this.setDataCell({
            rowIndex: endingRowIndex,
            colIndex: colIndex,
            value: endingRowIndex > startingRowIndex
                ? {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(startingRowIndex).getCell(colIndex).address +
                        ":" +
                        this._wsGeneral.getRow(endingRowIndex - 1).getCell(colIndex).address +
                        ")",
                }
                : 0,
            ws: ws,
            border: border,
            backgroundColorCode: backgroundColorCode,
        });
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + " Çalışma Süreleri Raporu", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.dateRange.startDateTime.setLocale(this._locale).toFormat("dd LLLL yyyy") +
            " - " +
            this._options.dateRange.endDateTime.setLocale(this._locale).toFormat("dd LLLL yyyy"), border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.KorfezTransportationFirstEntranceLastExitSummaryReport = KorfezTransportationFirstEntranceLastExitSummaryReport;
