"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EkmasDailyWorkingDetailsAtDateCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const business_pacs_report_1 = require("../../../business/pacs/business.pacs.report");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
    Constants.ColorCodes = {
        ON_TIME: "62A325",
        TOLERATED: "D56626",
        NOT_TOLERATED: "FF1301",
        PERMISSION: "4C2AA8",
        HASREPORT: "F0FF0F",
    };
    Constants.PermissionIds = {
        raporlu: "c2d6fea7-999c-4618-b23a-8b9906b2fb27",
    };
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    const report = new EkmasDailyWorkingDetailsAtDateCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class EkmasDailyWorkingDetailsAtDateCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._options = request.filter;
        this._requestFilter = request.filter;
        this._options = {
            date: (0, moment_1.default)(this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value).toDate(),
            userFilter: {
                organizationUnitIds: this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                applyOrganizationUnitFilterHierarchically: this._requestFilter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userGroupIds: this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                userIds: this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                workPlanIds: this._requestFilter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
                userOrganizationStatus: this._requestFilter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus).value,
            },
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
        };
    }
    async generateReport() {
        app_logs_1.logger.debug("Generating daily working details custom report");
        const option = Object.assign(this._options, {
            paginationRequest: {
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        });
        this._data = await dal_manager_1.dbManager.accessPacs2.workSummaryOfFilteredEmployeesAtDate(this._request.requesterUserId, this._request.organizationId, option);
        const userIds = this._data.items.map((i) => i.userId);
        const firstLastData = await (0, business_pacs_report_1.getDailySummaryReport)(this._request.organizationId, {
            date: new Date(this._options.date),
            userIds: userIds,
            status: this._options.userFilter.userOrganizationStatus,
            sortingMethod: this._options.sortType,
            pagination: {
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
            },
            hasOrganizationWide: undefined,
            permittedUnitIds: undefined,
        }, this._request.requesterUserId);
        this._items = firstLastData.items;
        this._workplans = firstLastData.workPlans;
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsDailyWorkingHours,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.items.length === 0 : undefined,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_WORKING_DETAILS_REPORT", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.date).format("DD/MM/YYYY") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        const userIds = this._data.items.map((i) => i.userId);
        const usersOrganizationDetails = await this.getUsersOrganizationUnitDetails({ organizationId: this._request.organizationId, userIds });
        for (const item of this._data.items) {
            const userOrganizationDetail = usersOrganizationDetails.filter((i) => i.userId === item.userId);
            this.addGeneralRow(item, userOrganizationDetail);
        }
        let filters = [];
        if (this._options.userFilter.userOrganizationStatus) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[this._options.userFilter.userOrganizationStatus].toUpperCase(), locale: this._locale }));
        }
        if (this._options.userFilter.workStatusFilter) {
            this._options.userFilter.workStatusFilter.forEach((workStatusFilter) => {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORK_STATUS", locale: this._locale }) +
                    " : " +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.WorkStatusFilter[workStatusFilter].toUpperCase(), locale: this._locale }));
            });
        }
        if (this._options.userFilter.organizationUnitIds && this._options.userFilter.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.userFilter.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._options.userFilter.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userFilter.userGroupIds && this._options.userFilter.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userFilter.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(","));
        }
        if (this._options.userFilter.workPlanIds && this._options.userFilter.workPlanIds.length > 0) {
            let workplanNames = this._workplans.filter((wp) => this._options.userFilter.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.join(","));
        }
        if (this._options.userFilter.userIds && this._options.userFilter.userIds.length > 0) {
            let userNames = this._items.filter((i) => this._options.userFilter.userIds.includes(i.user.id)).map((i) => i.user.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.join(","));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"), filters.length);
        this._wsGeneral.pageSetup.printArea = "A6:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready");
        return this._filePath;
    }
    async getUsersOrganizationUnitDetails(params) {
        const data = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const { rows } = await trx.query(`
					SELECT uop."extensionFields"->>'duty' as "dutyName", up."name" || ' ' || up."surname" as fullname, ou."name" as "organizationUnitName", uo."userId" 
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
					ON uo."id" = uoou."userOrganizationId" 
					AND uo."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
					INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
					ON ou."id" = uoou."organizationUnitId" AND ou."deletedAt" IS NULL
					INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					ON uo."userId" = uop."userId" AND uop."deletedAt" IS NULL
					LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up
					ON ou."managerUserId" = up."userId"
					WHERE uo."userId" = ANY ($1::UUID[]);
					
				`, [params.userIds]);
            return rows;
        });
        return data.map((details) => ({
            userId: details.userId,
            userDutyName: details.dutyName,
            userManagerFullName: details.fullname ?? "",
            userOrganizationUnitName: details.organizationUnitName,
        }));
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 35 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.DEPARTMENT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 40 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.POSITION_TITLE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 40 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "CUSTOM-REPORT.MANAGER", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.FIRST_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.LAST_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.STATUS", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ENTRANCE_PENALTY_TIME", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.MISSING_WORK", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.OVERTIME", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(data, userOganizationUnitDetail) {
        const row = this._items.find((i) => i.user.id === data.userId);
        if (!row) {
            return;
        }
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.user.fullname.toUpperCase(),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userOganizationUnitDetail.map((elem) => elem.userOrganizationUnitName).join("\n"),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userOganizationUnitDetail.map((elem) => elem.userDutyName).join("\n"),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: userOganizationUnitDetail.map((elem) => elem.userManagerFullName).join("\n"),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entranceDateTime ? (0, moment_1.default)(row.entranceDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.entranceDateTime ? Constants.TimeFormat : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.exitDateTime ? (0, moment_1.default)(row.exitDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.exitDateTime ? Constants.TimeFormat : null,
        });
        let permissionColorAndTextInfo = { statusText: "", colorText: undefined };
        permissionColorAndTextInfo.statusText = row.entranceDateTime
            ? i18n_1.default.__({ phrase: "EXCEL-REPORT.ARRIVED", locale: this._locale })
            : i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: this._locale });
        let hasPermissionNow;
        let permissionNowType;
        let hasPermissionInDay = row.ppermissions && row.ppermissions.length > 0;
        if (hasPermissionInDay) {
            permissionColorAndTextInfo.colorText = Constants.ColorCodes.PERMISSION;
            row.ppermissions.forEach((permission) => {
                if ((0, moment_1.default)().isBetween(permission.startUtc, permission.endUtc)) {
                    hasPermissionNow = true;
                    permissionNowType = permission.ppermissionTypeName;
                }
                if (permission.ppermissionTypeId) {
                    permissionColorAndTextInfo.colorText = this.findColorCodeByPPermissionId(permission.ppermissionTypeId);
                }
            });
        }
        if (hasPermissionInDay) {
            if (hasPermissionNow) {
                permissionColorAndTextInfo.statusText += "\n" + i18n_1.default.__({ phrase: "PDF-REPORT.day_off", locale: this._locale }) + "(" + permissionNowType + ")";
            }
            else if (hasPermissionInDay) {
                row.ppermissions.forEach((permission) => {
                    let interval = (0, moment_1.default)(permission.startUtc).locale(this._locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(permission.endUtc).locale(this._locale).format("DD/MM/YYYY HH:mm");
                    permissionColorAndTextInfo.statusText += "\n" + permission.ppermissionTypeName + "\n(" + interval + ")";
                });
            }
        }
        else {
            permissionColorAndTextInfo = this.findColorCodeByPerformance({ employeeData: data, statusText: permissionColorAndTextInfo.statusText });
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: permissionColorAndTextInfo.statusText,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: null,
            backgroundColorCode: permissionColorAndTextInfo.colorText,
            isTextWhite: permissionColorAndTextInfo.colorText == Constants.ColorCodes.HASREPORT || permissionColorAndTextInfo.colorText == Constants.ColorCodes.TOLERATED ? false : true,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entrancePenaltyTimeInSeconds ? (Math.round(row.entrancePenaltyTimeInSeconds / 60) / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.entrancePenaltyTimeInSeconds
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.missingWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.extraWorkInMinutes ? (row.extraWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.extraWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this._generalRowIndex++;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
        else if (params.isTextWhite != null) {
            c.font = {
                color: { argb: "000000" },
                bold: true,
            };
        }
    }
    findColorCodeByPPermissionId(ppermissionId) {
        let colorValue;
        switch (ppermissionId) {
            case Constants.PermissionIds.raporlu:
                colorValue = Constants.ColorCodes.HASREPORT;
                break;
            default:
                colorValue = Constants.ColorCodes.PERMISSION;
                break;
        }
        return colorValue;
    }
    findColorCodeByPerformance(params) {
        let colorCode = null;
        let statusText = params.statusText;
        if (!params.employeeData.firstStart || params.employeeData.lateStart > 0) {
            colorCode = Constants.ColorCodes.NOT_TOLERATED;
        }
        else {
            if (params.employeeData.toleratedLateStart && params.employeeData.toleratedLateStart > 0) {
                colorCode = Constants.ColorCodes.TOLERATED;
                statusText = i18n_1.default.__({ phrase: "EXCEL-REPORT.ENTRANCE_PENALTY_TIME", locale: this._locale });
            }
            else {
                colorCode = Constants.ColorCodes.ON_TIME;
            }
        }
        return { colorText: colorCode, statusText: statusText };
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_WORKING_DETAILS_REPORT", locale: this._locale }), border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + (0, moment_1.default)(this._options.date).year() + "," + ((0, moment_1.default)(this._options.date).month() + 1) + "," + (0, moment_1.default)(this._options.date).date() + ")",
        }, border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, filter, border);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 15;
        }
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.EkmasDailyWorkingDetailsAtDateCustomReport = EkmasDailyWorkingDetailsAtDateCustomReport;
