"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FevziGandurSAPIntegrationCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const uuid_1 = __importDefault(require("uuid"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const business_report_export_1 = require("../../../business/report/business.report.export");
const fs_1 = __importDefault(require("fs"));
const csv_1 = __importDefault(require("csv"));
const luxon_1 = require("luxon");
var Constants;
(function (Constants) {
    Constants.columns = [
        "userId",
        "status",
        "hours",
        "startDate",
        "startTime",
        "cust_ClockIN_Terminal",
        "endDate",
        "endTime",
        "cust_ClockOUT_Terminal",
        "timeType",
        "deleteExternalTime",
        "costCenter",
        "externalCode",
    ];
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    let report = new FevziGandurSAPIntegrationCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class FevziGandurSAPIntegrationCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._options = {
            dateRange: {
                startDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.Date).value)
                    .startOf("day")
                    .toDate(),
                endDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.Date).value)
                    .endOf("day")
                    .toDate(),
            },
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            filterOrganizationUnitMembersHierarchically: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            userGroupIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            status: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
            hasOrganizationWide: true,
        };
        this._fileId = uuid_1.default.v4();
    }
    async generateReport() {
        app_logs_1.logger.info("Generating custom report");
        this._data = await (0, business_report_export_1.generateFirstAndLastAccessLogsReportData)(this._request.organizationId, this._request.requesterUserId, this._options.hasOrganizationWide, this._options);
        await this.generate();
        await this.addGeneralRow(this._data.report.items);
        return {
            format: app_enums_1.enums.ReportFormat.Csv,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Fevzi Gandur Sap Integration Custom Report",
        };
    }
    async generate() {
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".csv");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._writableStream = fs_1.default.createWriteStream(this._filePath, { encoding: "utf8" });
        this._csvStringifier = csv_1.default.stringify({ encoding: "utf8", bom: true });
        this._csvStringifier.write(Object.values(Constants.columns));
        this._csvStringifier.write(Object.values(Constants.columns));
    }
    async addGeneralRow(item) {
        const csvFormatItems = item.map((elem) => {
            const csvRow = {
                uniqueId: elem.user.uniqueId,
                status: "NEW",
                hours: "",
            };
            if (elem.firstUtc) {
                csvRow.firstUtcDate = luxon_1.DateTime.fromJSDate(elem.firstUtc).toFormat("MM/dd/yyyy");
                csvRow.firstUtcTime = luxon_1.DateTime.fromJSDate(elem.firstUtc).toFormat("HH:mm:ss");
            }
            else {
                csvRow.firstUtcDate = " ";
                csvRow.firstUtcTime = " ";
            }
            if (elem.firstAccessControlPoint) {
                csvRow.firstAccessControlPoint = elem.firstAccessControlPoint.name;
            }
            else {
                csvRow.firstAccessControlPoint = " ";
            }
            if (elem.lastUtc) {
                csvRow.lastUtcDate = luxon_1.DateTime.fromJSDate(elem.lastUtc).toFormat("MM/dd/yyyy");
                csvRow.lastUtcTime = luxon_1.DateTime.fromJSDate(elem.lastUtc).toFormat("HH:mm:ss");
            }
            else {
                csvRow.lastUtcDate = " ";
                csvRow.lastUtcTime = " ";
            }
            if (elem.lastAccessControlPoint) {
                csvRow.lastAccessControlPoint = elem.lastAccessControlPoint.name;
            }
            else {
                csvRow.lastAccessControlPoint = " ";
            }
            csvRow.timeType = "9000";
            csvRow.deleteExternalTime = "";
            csvRow.costCenter = "";
            csvRow.externalCode = elem.firstAccessLogId ?? "";
            if (csvRow.externalCode) {
                csvRow.externalCode += elem.lastAccessLogId ? "-" + elem.lastAccessLogId : "";
            }
            else {
                csvRow.externalCode = elem.lastAccessLogId ?? "";
            }
            return Object.values(csvRow);
        });
        csvFormatItems.forEach((elem) => {
            this._csvStringifier.write(elem);
        });
        this._csvStringifier.end();
        this._csvStringifier.pipe(this._writableStream);
        const endOfWriteStream = new Promise((resolve) => {
            this._writableStream.on("finish", () => {
                console.log("finish");
                this._writableStream.close();
                resolve();
            });
        });
        await endOfWriteStream;
    }
}
exports.FevziGandurSAPIntegrationCustomReport = FevziGandurSAPIntegrationCustomReport;
