"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ArkasSummaryCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_report_export_1 = require("../../../business/report/business.report.export");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const business_pacs_ppermission_1 = require("../../../business/pacs/business.pacs.ppermission");
const report_util_1 = require("../../../business/report/report.util");
const fs_1 = __importDefault(require("fs"));
class ArkasSummaryCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._request = request;
        this._options = request.filter;
        this.filter = {
            dateRange: {
                startDateTime: new Date(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
                endDateTime: new Date(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            },
            userId: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
        };
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating monthly summary excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsSummary = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.SUMMARY", locale: "tr" }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsDetailed = this._wb.addWorksheet("GünBazındaDetay", {
            pageSetup: { paperSize: 9, orientation: "landscape" },
        });
        this._wsSummary.properties.defaultRowHeight = 15;
        this._summaryRowIndex = 1;
        this._wsDetailed.properties.defaultRowHeight = 15;
        this._detailedRowIndex = 1;
        this._ppermissionTypes = await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(this._request.organizationId, { take: 100, skip: 0 });
        this._data = await (0, business_report_export_1.generatePACSDailyForSingleUserData)(this._request.organizationId, this.filter.userId, this.filter.userId, {
            dateRange: this.filter.dateRange,
            userId: this.filter.userId,
            pagination: {
                skip: undefined,
                take: app_config_1.appConfig.reportExportRowLimit,
            },
            sortOrder: "DESC",
            sortType: app_enums_1.enums.WorkSummarySortType.Date,
        });
        this.permissionDataForSummary = await this.getPermissionDataForSummary();
        this._userData = (await dal_manager_1.dbManager.accessUser.getUsersFullnamesAndUniqueIds(this._request.organizationId, [this.filter.userId]))[0];
        if (!this._userData) {
            app_logs_1.logger.debug(`fullname and uniqueId of the user with id ${this.filter.userId} could not be obtained.`);
        }
        this.initSummarySheet(this._wsSummary, 1, 2);
        this.initDetailedSheet(this._wsDetailed, 1, 1);
        this.fillSummarySheet(this._wsSummary, this._summaryRowIndex, 3);
        this.addNoteToSummarySheet(this._wsSummary, this._summaryRowIndex);
        if (this._data.report.items?.length > 0) {
            for (const item of this._data.report.items) {
                this.addDetailedRow(item);
            }
        }
        this._wsSummary.pageSetup.orientation = "portrait";
        this._wsDetailed.pageSetup.orientation = "portrait";
        this._wsSummary.commit();
        this._wsDetailed.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "ARKAS ÇALIŞAN ÖZET RAPOR",
        };
    }
    addColumnCellForName(ws, row, col, value) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "bottom",
            wrapText: true,
        };
        c.border = {
            bottom: {
                style: "thick",
            },
        };
    }
    addCell(ws, row, col, value, height, border, isBold) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: isBold ?? false,
        };
        c.border = border ?? generator_1.ReportGenerator.Constants.Styling.AllThin;
        const rowCell = ws.getRow(row);
        rowCell.height = height;
    }
    initDetailedSheet(wsRaw, startRowIndex, startColIndex) {
        const rowData = [
            {
                value: "Sicil No",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
            {
                value: "Ad Soyad",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
            {
                value: "Tarih",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
            {
                value: "Giriş Saati",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
            {
                value: "Çıkış Saati",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
            {
                value: "Mazeret Durumu",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "left",
                backgroundColorCode: "b9cde5",
            },
        ];
        startColIndex = this.addCellsToRowWithOptions(rowData, startColIndex, startRowIndex, wsRaw);
        const columns = [{ width: 20 }, { width: 20 }, { width: 15 }, { width: 15 }, { width: 15 }, { width: 20 }];
        wsRaw.columns = columns;
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        this._detailedRowIndex = 2;
    }
    initSummarySheet(wsRaw, startRowIndex, startColIndex) {
        let c = wsRaw.getCell(startRowIndex, startColIndex + 2);
        c.alignment = {
            horizontal: "left",
            vertical: "bottom",
            wrapText: true,
        };
        c.value = (0, moment_1.default)(new Date()).format("M/D/YYYY");
        startRowIndex++;
        this.addColumnCellForName(wsRaw, startRowIndex, startColIndex++, "AD SOYAD:");
        this.addColumnCellForName(wsRaw, startRowIndex++, startColIndex--, this._userData?.fullName ?? "Kullanıcı ismi alınamadı.");
        startRowIndex++;
        c = wsRaw.getCell(startRowIndex, startColIndex);
        c.alignment = {
            horizontal: "left",
            vertical: "bottom",
            wrapText: true,
        };
        c.value = "DÖNEM:";
        c = wsRaw.getCell(startRowIndex++, startColIndex + 1);
        c.alignment = {
            horizontal: "left",
            vertical: "bottom",
            wrapText: true,
        };
        c.value = (0, moment_1.default)(this.filter.dateRange.startDateTime).format("M/D/YYYY") + " - " + (0, moment_1.default)(this.filter.dateRange.endDateTime).format("M/D/YYYY");
        startRowIndex++;
        this.addCell(wsRaw, startRowIndex, startColIndex++, "Açıklama", 25, generator_1.ReportGenerator.Constants.Styling.AllThinButBottomThick, true);
        this.addCell(wsRaw, startRowIndex++, startColIndex--, "Gün/Saat Sayısı", 25, generator_1.ReportGenerator.Constants.Styling.AllThinButBottomThick, true);
        this.addCell(wsRaw, startRowIndex++, startColIndex, "Toplam İş Günü", 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, "Normal Saatinde Gelinen Gün", 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, "Geç Geldiği Gün", 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, "Erken Çıktığı Gün", 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, "Geç Çıktığı Gün", 25);
        const columns = [{ width: 5 }, { width: 30 }, { width: 30 }, { width: 15 }];
        wsRaw.columns = columns;
        this._summaryRowIndex = 7;
    }
    addNoteToSummarySheet(wsRaw, startRowIndex) {
        let c = wsRaw.getCell(startRowIndex + 5, 2);
        c.alignment = {
            horizontal: "left",
            vertical: "bottom",
            wrapText: true,
        };
        c.font = {
            size: 10,
        };
        c.value = "Not: Giriş ve Çıkışlarda 15 dakika gecikmeler dikkate alınmamıştır.";
        wsRaw.mergeCells(startRowIndex + 5, 2, startRowIndex + 5, 2 + 1);
    }
    async getPermissionDataForSummary() {
        const usageList = this._data.report.summary.ppermissionUsageList;
        let permissions = [];
        for (let i = 0; i < usageList.length; i++) {
            const permissionType = this._ppermissionTypes.items.find((pt) => pt.id === usageList[i].typeId);
            let permissionUsageCount = 0;
            if (permissionType.isDailyScheduled) {
                permissionUsageCount = await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(this._request.organizationId, permissionType.id, usageList[i].range.start.toDate(), usageList[i].range.end.toDate(), null, null, permissionType);
                permissions.push({
                    id: usageList[i].typeId,
                    name: usageList[i].name,
                    data: permissionUsageCount,
                    isDaily: true,
                });
            }
            else {
                permissionUsageCount = await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(this._request.organizationId, permissionType.id, usageList[i].range.start.toDate(), usageList[i].range.end.toDate(), null, permissionType);
                permissions.push({
                    id: usageList[i].typeId,
                    name: usageList[i].name,
                    data: permissionUsageCount,
                    isDaily: false,
                });
            }
        }
        const resultHashmap = permissions.reduce((obj, item) => {
            obj[item.id] ? (obj[item.id].data += item.data) : (obj[item.id] = { ...item });
            return obj;
        }, {});
        const tempResult = Object.values(resultHashmap);
        let result = [];
        tempResult.forEach((element) => {
            let permissionUsageCount = "";
            if (element.isDaily) {
                permissionUsageCount = element.data + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_DAY", locale: "tr" });
            }
            else {
                const permissionMinutes = element.data;
                const hours = Math.floor(permissionMinutes / 60);
                const minutes = permissionMinutes % 60;
                permissionUsageCount =
                    (hours ? hours + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_HOUR", locale: "tr" }) : "") +
                        (minutes ? ", " + minutes + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_MINUTE", locale: "tr" }) : "");
            }
            result.push({
                name: element.name,
                data: permissionUsageCount,
            });
        });
        return result;
    }
    fillSummarySheet(wsRaw, startRowIndex, startColIndex) {
        const summary = this._data.report.summary;
        this.addCell(wsRaw, startRowIndex++, startColIndex, summary.workDaysCount, 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, summary.arrivedOnTimeDaysCount, 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, summary.lateArrivedDaysCount, 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, summary.earlyLeftDaysCount, 25);
        this.addCell(wsRaw, startRowIndex++, startColIndex, summary.lateLeftDaysCount, 25);
        for (let i = 0; i < this.permissionDataForSummary.length; i++) {
            this.addCell(wsRaw, startRowIndex, startColIndex - 1, this.permissionDataForSummary[i].name, 25);
            this.addCell(wsRaw, startRowIndex++, startColIndex, this.permissionDataForSummary[i].data, 25);
        }
        this._summaryRowIndex = startRowIndex;
    }
    addDetailedRow(record) {
        let colIndex = 1;
        let date = (0, report_util_1.formatDateForExcel)(record.date);
        let statusDetail = [];
        if (!record.firstIn && record.missingWorkInMinutes > 0) {
            statusDetail.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: "tr" }));
        }
        let hasPermissionInDay = record.ppermissions && record.ppermissions.length > 0;
        let hasHolidayInDay = record.holidays && record.holidays.length > 0;
        if (hasPermissionInDay || hasHolidayInDay) {
            record.ppermissions.forEach((permission) => {
                statusDetail.push(permission.ppermissionTypeName);
            });
            record.holidays.forEach((holiday) => {
                statusDetail.push(holiday.holidayName);
            });
        }
        else {
            if (record.expectedWorkingRanges.length === 0) {
                statusDetail = ["Hafta Tatili"];
            }
        }
        const statusDetailStr = statusDetail.join("\n");
        const rowData = [
            {
                value: this._userData?.uniqueId ?? "Kullanıcı Sicil No bulunamadı. ",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
            },
            {
                value: this._userData?.fullName ?? "Kullanıcı ismi bulunamadı.",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
            },
            {
                value: date,
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "right",
            },
            {
                value: record?.firstIn?.actionUtc ? (0, moment_1.default)(record.firstIn.actionUtc).locale(this._locale).format("HH:mm") : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "right",
            },
            {
                value: record?.lastOut?.actionUtc ? (0, moment_1.default)(record.lastOut.actionUtc).locale(this._locale).format("HH:mm") : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "right",
            },
            {
                value: statusDetailStr,
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
            },
        ];
        colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._detailedRowIndex, this._wsDetailed);
        this._detailedRowIndex++;
    }
}
exports.ArkasSummaryCustomReport = ArkasSummaryCustomReport;
async function generateReport(request, locale) {
    let report = new ArkasSummaryCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
