"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomEndPoints = void 0;
const api_securityhelper_1 = require("../../../api/api.securityhelper");
const app_auth_1 = require("../../../app.auth");
const app_config_1 = require("../../../app.config");
const app_logs_1 = require("../../../app.logs");
const anadolu_integration_1 = require("../anadolu-integration");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_device_1 = require("../../../business/business.device");
const app_enums_1 = require("../../../app.enums");
const uuid_1 = __importDefault(require("uuid"));
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const moment_1 = __importDefault(require("moment"));
const got_1 = require("got");
const enums_1 = require("../../../lib/access-model/v2/enums");
const dal_access_psql_log_1 = require("../../../dal/access/psql/dal.access.psql.log");
var jwt = require("jsonwebtoken");
const organizationId = "0c80f542-d777-4f42-8820-52345194ca40";
const permittedDeviceUserIds = [
    "a149057f-2d46-4fed-8613-c14fe766fac7",
    "a4964ac2-406c-49ca-8342-8afc2e32af7e",
    "0094632b-844e-4e51-a6c3-4be23f0b6ade",
    "454c7ba2-b90a-4975-9b3e-7e1898ada39f",
    "93e75ca0-a246-4649-b21f-3300fa75dd4f",
    "9a73d9b9-7df9-480d-97dd-454a7aa1ae40",
    "bb078aa7-626a-4e49-b856-0de96fc4bf48",
    "59e52809-3135-4e21-8ed5-eb4fcbd897af",
    "45ac956a-3bf3-4886-b82e-70034f1a89f4",
    "85e2143e-4bf5-4e7f-b667-fe7ecc178d19",
    "8809bb29-a068-4a51-aa75-2cdce6238d7d",
    "2df21fb5-f4d6-4617-a702-989081925fe2",
    "192e42e5-16f5-4d96-9841-a7615035511a",
    "157e253f-8980-4e17-8248-38c520c15d53",
    "83a9469e-0dba-43bb-a117-e0cad04ed01b",
    "2b8b8aa3-4cfd-4aae-82da-2fe8a45f8373",
    "857064a0-9040-4880-b81f-303a19ff239b",
    "cbb721c7-ec1e-497f-97d4-535cc03f1098",
    "10c13057-14e7-4a20-bae3-13231114bebf",
    "82efc9e8-b434-46e6-b978-b21819014f6f",
    "efadad11-5af9-45f4-831e-2d35bd6fc0a0",
    "cab43a48-1b55-4366-b65f-a1be331a5aff",
];
function addCustomEndPoints(app) {
    app.post("/custom/authorization", async (req, res, next) => {
        const authHeader = req.header("Authorization");
        if (!authHeader) {
            res.status(400).send({
                message: `Authorization header is missing!`,
            });
            return;
        }
        try {
            let decodedToken = jwt.verify(authHeader.substring(7), app_config_1.appConfig.jwtSecret);
            req["auth_token"] = new app_auth_1.UserJwtPayload(decodedToken);
        }
        catch (error) {
            app_logs_1.logger.error(`[anadolu server-hook.ts] JWT validation error! ${error}`);
            if (error.name === "TokenExpiredError") {
                res.status(401).send({ message: "JWT timed out, please refresh access token!" });
            }
            else if (error.name === "JsonWebTokenError") {
                res.status(400).send({
                    message: `JWT could not be validated, probably malformed, please refresh access token or login again!`,
                });
            }
            else {
                res.status(500).send(error);
            }
            return;
        }
        let userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
        if (!permittedDeviceUserIds.includes(userId)) {
            res.status(401).send(`
				Intruder Alert! You are trying to do something nasty and we are aware of it! 
			`);
        }
        let reqBody = req.body;
        let credentialType;
        switch (reqBody.erisimTipi) {
            case anadolu_integration_1.AnadoluIntegrationAccessType.MIFARE_CARD:
                credentialType = app_enums_1.enums.CredentialType.MiFare;
                break;
            case anadolu_integration_1.AnadoluIntegrationAccessType.BARCODE:
            case anadolu_integration_1.AnadoluIntegrationAccessType.QRCODE:
                credentialType = app_enums_1.enums.CredentialType.QrCode;
                break;
        }
        app_logs_1.logger.debug("[anadolu server-hook.ts] request params: " + JSON.stringify(req.body, null, 4));
        try {
            const accessTime = (0, moment_1.default)();
            const response = await (0, anadolu_integration_1.anadoluIntegrationAuthorizeUser)({ ...req.body, erisimZamani: accessTime.utc(true).toISOString().replace("Z", ""), organizationId });
            if (response.statusCode === 200) {
                await addAccessLog(userId, reqBody.terminalId, {
                    a: "",
                    id: uuid_1.default.v4(),
                    r: response.data?.sonuc === anadolu_integration_1.AnadoluIntegrationAccessResult.ERISIM_KABUL_EDILDI ? app_enums_1.enums.AccessLogReason.Success : app_enums_1.enums.AccessLogReason.UnknownCredentialOwner,
                    s: response.data?.sonuc === anadolu_integration_1.AnadoluIntegrationAccessResult.ERISIM_KABUL_EDILDI ? true : false,
                    u: accessTime.toISOString(),
                    o: dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId,
                    v: false,
                    cid: userId,
                    cx: [
                        {
                            t: credentialType,
                            d: reqBody.data,
                        },
                    ],
                    d: app_enums_1.enums.AccessDirection.All,
                    sid: [],
                    c: null,
                    cs: null,
                    di: null,
                    i: null,
                    ir: null,
                    iq: null,
                    ln: null,
                    lr: null,
                    lt: null,
                    mp: null,
                    re: null,
                    rg: null,
                    rx: null,
                    ss: null,
                });
            }
            res.status(response.data.sonuc === anadolu_integration_1.AnadoluIntegrationAccessResult.ERISIM_REDDEDILDI ? 404 : 200).send(response.data);
        }
        catch (error) {
            app_logs_1.logger.error(JSON.stringify(error, null, 4));
            if (error instanceof got_1.HTTPError) {
                res.status(error.statusCode).send(error.body);
            }
            else {
                res.status(500).send(error.message || error);
            }
        }
    });
}
exports.addCustomEndPoints = addCustomEndPoints;
const addAccessLog = async (deviceId, deviceSerialNumber, deviceLog) => {
    let insertedLog = null;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const { rows: acpIdRows, rowCount: acpIdRowCount } = await trx.query(`
				SELECT id 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
				WHERE "deviceId" = $1 AND "deletedAt" IS NULL;
			`, [deviceId]);
        if (acpIdRowCount > 1) {
            throw new Error(`There are multiple access point mappings for device with serialnumber ${deviceSerialNumber}, configuration error. Please fix configuration first!`);
        }
        else if (acpIdRowCount < 1) {
            throw new Error(`No access point mapping for device with serialnumber ${deviceSerialNumber}, configuration error. Please create access point first!`);
        }
        deviceLog.a = acpIdRows[0].id;
        const logItem = await (0, business_device_1.updateLogFromDevice)({ organizationId, deviceLog, logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant });
        await dal_manager_1.dbManager.accessLog.addAccessLogFromDevice(organizationId, logItem, trx);
    }, deviceId, organizationId);
    if (insertedLog.ir === undefined && insertedLog.d && insertedLog.d === enums_1.EnumsV2.AccessDirection.Entrance && insertedLog.s === true) {
        await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, { credentialOwnerUserId: insertedLog.o, generationTime: insertedLog.u, redisCache: dal_manager_1.dbManager.accessRedisCache, logId: insertedLog.id });
    }
};
